package activities.api;

/**
 * Created by oracle on 2017-07-10.
 */

import activities.dto.goods.GoodsPictureDto;
import activities.dto.modeldto.ActivityDefinitionDto;
import activities.vo.ActivityConfigurationVo;
import activities.vo.ActivityInfoVO;
import activities.vo.ActivityVo;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 活动配置api
 */
public interface ActivityConfigurationApi {
    String ACTIVITY_REDISKEY_STARTACTIVITY = "jicaiScore:activity:startActivity:";
    String ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME = "jicaiScore:activity:lastActivityTime:";
    String ACTIVITY_REDISKEY_INREDIS = "jicaiScore:activity:inRedis";

    String ACTIVITY_REDISKEYS = "itaipingPC:activitiesKeys:set";
    String ACTIVITY_HAS_BEGIN = "BEGIN"; //查询场次开始时间标志位
    String ACTIVITY_HAS_END = "END"; //查询场次结束时间标志位


    String LOCK = "lock";

    String SPLITER = ":";

    String ACTIVITY = "activity";

    /**
     * 活动KEY
     */
    String KEYS = "keys";

    /**
     * 活动库存
     */
    String STOCK = "stock";

    /**
     * 当前进行的活动场次ID
     */
    String CURRENT_ACTIVITY_ID = "current-activity-id";

    /**
     * 活动信息
     */
    String ACTIVITY_INFO = "activity-info";

    /**
     * 活动初始时间
     */
    String INIT_TIME = "init-time";

    /**
     * 活动开始时间
     */
    String BEGIN_TIME = "begin-time";

    /**
     * 活动结束时间
     */
    String END_TIME = "end-time";

    /**
     * 活动商品
     */
    String GOODS = "goods";

    /**
     * 判断该渠道活动已配置并开启进行中
     *
     * @param organizationCode
     * @return
     */
    Boolean isInActivity(String organizationCode);

    /**
     * 判断指定活动是否已配置并开启进行中
     *
     * @param organizationCode
     * @param activityCode
     * @return
     */
    Boolean isInActivity(String organizationCode, String activityCode);

    /**
     * 开启并初始化指定渠道的已配置活动
     *
     * @param organizationCode
     */
    void startActivity(String organizationCode);

    /**
     * 获取已配置的活动
     *
     * @param organizationCode
     * @param activityCode
     * @param time
     * @return
     */
    List<ActivityDefinitionDto> getActivitiesOnGoging(String organizationCode, String activityCode, Date time);


    /**
     * 根据活动code查询所有的活动信息
     * @param organizationCode
     * @param activityCode
     * @return
     */
    ActivityConfigurationVo getActivityByCode(String organizationCode, String activityCode);

    /**
     * 清除活动缓存并重新加载
     * @param organizationCode 机构代码
     * @param activityCodes 活动代码集合
     * @return
     */
    String reload(String organizationCode, List<String> activityCodes);

    /**
     * 根据活动场次ID获得场次对应商品
     * @param organizationCode 机构代码
     * @param activityId 活动场次ID
     * @param activityCode 活动代码
     * @param storeCode 店铺代码
     * @return
     */
    List<ActivityVo> getGoodsListByActivityId(String organizationCode, String activityId, String activityCode, String storeCode);

    /**
     * 批量获取 GoodsPictureDTO, GoodsPictureDTO 下包含 Goods , 排除 Goods 下所有其表他关联对象
     *
     * @param goodsIds List<String> goodsId 集合
     * @return List<GoodsPictureDto>
     */
    List<GoodsPictureDto> batchQueryGoodsPictureForActivityGoodsList(List<String> goodsIds);

    /**
     * 倒计时功能
     * @param activityIndex 场次对应的list下标
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return
     */
    Map<String, Object> getSystemTime(Integer activityIndex, String organizationCode, String activityCode);

    /**
     * 获取正在进行的活动场次id
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return 活动场次id
     */
    String getCurrentActivityId(String organizationCode, String activityCode);

    /**
     * 获得某个商品正参与的活动场次ID
     * @param organizationCode 机构代码
     * @param goodsId 商品ID
     * @param activityCode 活动代码
     * @return String
     */
    String getCurrentActivityId(String organizationCode, String goodsId, String activityCode);

    /**
     * 查询redis是否存在
     * @param key key
     * @return boolean
     */
    boolean redisHas(String key);

    /**
     * 存储redis
     * @param key key
     */
    void redisSave(String key,Object obj);

    /**
     * 得到redis
     * @param key key
     * @return Object
     */
    Object redisReturn(String key);

    /**
     * 删除redis
     * @param key key
     */
    void deleteRedisKey(String key);

    /**
     * 从缓存中获取活动商品库存数据
     * @param organizationCode 机构代码
     * @return map集合 对应缓存 redis 的 key value
     */
    Map<String, Integer> getStockFromRedis(String organizationCode, String activityCode);

    /**
     * 根据活动id 和 活动商品id获取百分比
     * @param activityDefinitionId，activityGoodsId
     * @return String
     */
    String getStockPercent(String activityDefinitionId, String activityGoodsId);

    /**
     * 构建机构活动库存全匹配 key
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return String
     */
    String buildKeyForActivityStock(String organizationCode, String activityCode);

    /**
     * 构建活动商品库存 key
     * @param organizationCode organizationCode 机构代码
     * @param goodsId 商品id
     * @param activityId 活动场次ID
     * @param activityCode 活动代码
     * @return String
     */
    String buildKeyForActivityStock(String organizationCode, String goodsId, String activityId, String activityCode);

    /**
     * 记录机构活动 key
     * @param organizationCode 机构代码
     * @param key key
     */
    void setActivityKey(String organizationCode, String key);

    /**
     * 构建当前场次活动ID key
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return String
     */
    String buildKeyForCurrentActivityId(String organizationCode, String activityCode);

    /**
     * 批量构建库存缓存 KEY
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @param activityId 活动场次ID
     * @param goodsIds 商品ID列表
     * @return Map<String, String> KEY = goodsId, VALUE = 库存缓存 KEY
     */
    Map<String, String> buildKeyForActivityStock(String organizationCode, String activityCode, String activityId, List<String> goodsIds);

    /**
     * 构建活动商品列表 key
     * @param organizationCode 机构代码
     * @param activityId 活动场次ID
     * @param activityCode 活动代码
     * @return String
     */
    String buildKeyForActivityGoods(String organizationCode, String activityId, String activityCode);

    /**
     * 为当前加载活动线程加锁防止并发
     *
     * @param organizationCode 机构代码
     * @param activityId 活动ID
     * @param activityCode 活动代码
     * @param storeCode 店铺代码
     * @return 锁住返回 true 否则返回 false
     */
    boolean lockForLoadActivityGoods(String organizationCode, String activityId, String activityCode, String storeCode);

    /**
     * 为当前加载活动线程解锁
     *
     * @param organizationCode 机构代码
     * @param activityId 活动ID
     * @param activityCode 活动代码
     * @param storeCode 店铺代码
     */
    void unlockForLoadActivityGoods(String organizationCode, String activityId, String activityCode, String storeCode);

    /**
     * 构建当前场次活动信息 key
     *
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return String
     */
    String buildKeyForCurrentActivityInfo(String organizationCode, String activityCode);

    /**
     * 批量设置缓存数据
     *
     * @param map Map<Object, Object>
     */
    void redisSave(Map<Object, Object> map);

    /**
     * 清除指定机构代码下指定活动代码缓存数据
     *
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     */
    void clearActivityKeys(String organizationCode, String activityCode);

    /**
     * 从缓存中获取活动信息
     *
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return ActivityInfoVO
     */
    ActivityInfoVO getActivityInfo(String organizationCode, String activityCode);
}
