package activities.service.apiImpl;

import activities.api.ActivityApi;
import activities.api.ActivityConfigurationApi;
import activities.api.ActivityGoodsApi;
import activities.dto.goods.GoodsPictureDto;
import activities.dto.modeldto.ActivityDefinitionDto;
import activities.dto.modeldto.ActivityGoodsDto;
import activities.model.ActivityDefinition;
import activities.model.ActivityGoods;
import activities.model.repository.ActivityDefinitionRepos;
import activities.service.activities.dao.ActivitiesDao;
import activities.service.service.ActivityGoodsService;
import activities.vo.ActivityConfigurationVo;
import activities.vo.ActivityInfoVO;
import activities.vo.ActivityVo;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import goods.model.repository.GoodsPictureRepos;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import utils.Lang;
import utils.data.BeanMapper;
import utils.lang.ObjectConvert;
import utils.lock.RedisLockUtil;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by oracle on 2017-07-10.
 */

@Slf4j
@MotanService
public class ActivityConfigurationApiImpl implements ActivityConfigurationApi {

    @Autowired
    ActivityDefinitionRepos activityDefinitionRepos;

    @Autowired
    RedisTemplate redisTemplate;

    @Autowired
    ActivityApi activityApi;

    @Autowired
    ActivityGoodsApi activityGoodsApi;

    @Autowired
    ActivitiesDao activitiesDao;

    @Autowired
    MapperFacade mapperFacade;

    @Autowired
    ActivityGoodsService activityGoodsService;

    @Autowired
    GoodsPictureRepos goodsPictureRepos;

    @Value("${itaipingMobile.organization.code}")
    private String itaipingMobileOrganizationCode;

    @Value("${itaipingPC.organization.code}")
    private String itaipingPCOrganizationCode;

    @Value("${secKillActivity.activityCode}")
    private String secKillActivityCode;

    @Value("${tuanGouActivity.activityCode}")
    private String tuanGouActivityCode;

    @Autowired
    RedisLockUtil redisLockUtil;

    @Autowired
    JdbcTemplate jdbcTemplate;

    @Override
    public Boolean isInActivity(String organizationCode) {
        Object startActivity = redisTemplate.opsForValue().get(ACTIVITY_REDISKEY_STARTACTIVITY + organizationCode);
        if (startActivity != null) {
            Object o = redisTemplate.opsForValue().get(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode);
            if (o != null && ((Date) o).after(new Date())) {//缓存中最后一次活动时间
                return true;
            }
            List<ActivityDefinition> activityDefinitions = activityDefinitionRepos.findByOrganizationCodeAndEndTimeGreaterThanOrderByEndTimeDesc(organizationCode, new Date());
            if (activityDefinitions == null || activityDefinitions.isEmpty()) {//没配置活动
                deleteActivityKeys();
                return false;
            }
            ActivityDefinition activityDefinition = activityDefinitions.get(0);
            redisTemplate.boundValueOps(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode).set(activityDefinition.getEndTime());
            saveActivityKeys(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode);
            return true;
        }
        return false;
    }

    public void saveActivityKeys(String key) {
        Object activityKeySetObj = redisTemplate.boundValueOps(ACTIVITY_REDISKEY_INREDIS).get();
        if (activityKeySetObj == null) {
            activityKeySetObj = new HashSet<>();
        }
        Set<String> activityKeySet = (Set<String>) activityKeySetObj;
        activityKeySet.add(key);
    }

    public void deleteActivityKeys() {
        Object activityKeySetObj = redisTemplate.boundValueOps(ACTIVITY_REDISKEY_INREDIS).get();
        if (activityKeySetObj != null) {
            Set<String> activityKeySet = (Set<String>) activityKeySetObj;
            for (String s : activityKeySet) {
                if (redisTemplate.boundValueOps(s).get() != null) {
                    redisTemplate.delete(s);
                }
            }
        }
    }

    @Override
    public Boolean isInActivity(String organizationCode, String activityCode) {
        Boolean inActivity = isInActivity(organizationCode);
        if (inActivity) {
            Object o = redisTemplate.opsForValue().get(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode + ":" + activityCode);
            if (o != null && ((Date) o).after(new Date())) {//缓存中最后一次活动时间
                return true;
            }
            List<ActivityDefinition> activityDefinitions = activityDefinitionRepos.findByOrganizationCodeAndActivityCodeAndEndTimeGreaterThanOrderByEndTimeDesc(organizationCode, activityCode, new Date());
            if (activityDefinitions == null || activityDefinitions.isEmpty()) {
                if (redisTemplate.boundValueOps(ACTIVITY_REDISKEY_STARTACTIVITY + organizationCode + ":" + activityCode).get() != null) {
                    redisTemplate.delete(ACTIVITY_REDISKEY_STARTACTIVITY + organizationCode + ":" + activityCode);
                }
                if (redisTemplate.boundValueOps(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode + ":" + activityCode).get() != null) {
                    redisTemplate.delete(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode + ":" + activityCode);
                }
                return false;
            }
            ActivityDefinition activityDefinition = activityDefinitions.get(0);
            redisTemplate.boundValueOps(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode + ":" + activityCode).set(activityDefinition.getEndTime());
            saveActivityKeys(ACTIVITY_REDISKEY_LAST_ACTIVITY_TIME + organizationCode + ":" + activityCode);
            return true;
        }
        return false;
    }

    @Override
    public void startActivity(String organizationCode) {
        deleteActivityKeys();
        redisTemplate.boundValueOps(ACTIVITY_REDISKEY_STARTACTIVITY + organizationCode).set("startActivity,organizationCode:" + organizationCode + " time: " + new SimpleDateFormat("yyyy-MM-dd HH-mm-ss").format(new Date()));
    }

    @Override
    public List<ActivityDefinitionDto> getActivitiesOnGoging(String organizationCode, String activityCode, Date time) {
        List<ActivityDefinition> activityDefinitions = activityDefinitionRepos.findByOrganizationCodeAndActivityCodeAndEndTimeGreaterThanOrderByBeginTime(organizationCode, activityCode, time);
        List<ActivityDefinition> activityDefinitionList = new ArrayList<>();
        for (ActivityDefinition activityDefinition : activityDefinitions) {
            activityDefinitionList.add(ObjectConvert.convert(activityDefinition, ActivityDefinition.class));
        }
        List<ActivityDefinitionDto> activityDefinitionDtoList = BeanMapper.mapList(activityDefinitionList, ActivityDefinition.class, ActivityDefinitionDto.class);
        return activityDefinitionDtoList;
    }

    @Override
    public ActivityConfigurationVo getActivityByCode(String organizationCode, String activityCode) {
        ActivityConfigurationVo activityConfigurationVo = new ActivityConfigurationVo();
        List<Map<String, Object>> resultSuccess = initActivity(organizationCode, activityCode);
        if (resultSuccess.size() > 0) {
            String activityBeginTimeKey = buildKeyForActivityBeginTime(organizationCode, activityCode);
            String activityEndTimeKey = buildKeyForActivityEndTime(organizationCode, activityCode);
            redisTemplate.opsForValue().set(activityBeginTimeKey, resultSuccess.get(0).get("beginTime"));
            redisTemplate.opsForValue().set(activityEndTimeKey, resultSuccess.get(resultSuccess.size() - 1).get("endTime"));
            setActivityKey(organizationCode, activityBeginTimeKey);
            setActivityKey(organizationCode, activityEndTimeKey);
            //所有场次开始时间、结束时间、活动id
            activityConfigurationVo.setActivityTimes(resultSuccess);
            //判断活动状态——是否已经开始或结束
            activityConfigurationVo.setActivityStatusStart(activityBeginOrEnd(organizationCode, activityCode, ACTIVITY_HAS_BEGIN));
            activityConfigurationVo.setActivityStatusEnd(activityBeginOrEnd(organizationCode, activityCode, ACTIVITY_HAS_END));
            //设置js全局变量活动标志位
            activityConfigurationVo.setActivityFlag(activityCode);
        } else {
            activityConfigurationVo.setActivityStatusEnd(true);
        }
        return activityConfigurationVo;
    }


    public List<Map<String, Object>> initActivity(String organizationCode, String activityCode) {
        //查询所有场次
        List<Map<String, Object>> result = null;
        String resultKey = buildKeyForActivityInitTime(organizationCode, activityCode);
        setActivityKey(organizationCode, resultKey);
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(resultKey);
        if (!Lang.isEmpty(boundValueOperations) && !Lang.isEmpty(boundValueOperations.get())) {
            result = (List<Map<String, Object>>) boundValueOperations.get();
            // 过滤时间过期但是未被置为下线的活动场次问题数据
            for (Map<String, Object> map : result) {
                Date endTime = (Date) map.get("endTime");
                if (endTime.before(new Date())) {
                    String activityId = (String) map.get("id");
                    ActivityDefinitionDto activityDefinitionDto = activityApi.findById(activityId);
                    if (activityDefinitionDto.getIsOnline() && activityDefinitionDto.getEndTime().equals(endTime)) {
                        activityDefinitionDto.setIsOnline(false);
                        activityDefinitionRepos.saveAndFlush(BeanMapper.map(activityDefinitionDto, ActivityDefinition.class));
                    }
                }

            }
            boundValueOperations.set(result);
        } else {
            result = getAllActivityList(organizationCode, activityCode);
            boundValueOperations.set(result);
        }
        return result;
    }

    public List<Map<String, Object>> getAllActivityList(String organizationCode, String activityCode) {
        //获取所有在线活动场次开始时间及id
        List<Map<String, Object>> resultList = new ArrayList<Map<String, Object>>();
        //查询出所有秒杀活动
        List<ActivityDefinitionDto> list = activityApi.findSecKillActivity(organizationCode, activityCode);
        if (list.size() > 0) {
            for (ActivityDefinitionDto activityDefinition : list) {
                Map<String, Object> map = new HashMap<String, Object>(3);
                if (!Lang.isEmpty(activityDefinition.getBeginTime())) {
                    map.put("beginTime", activityDefinition.getBeginTime());
                    map.put("endTime", activityDefinition.getEndTime());
                    map.put("id", activityDefinition.getId());
                    //所有场次开始时间
                    resultList.add(map);
                }
            }
        }
        return resultList;
    }

    public boolean activityBeginOrEnd(String organizationCode, String activityCode, String flag) {
        //判断活动已经开始或结束  flag 开始：ACTIVITY_HAS_BEGIN；  结束：ACTIVITY_HAS_END
        Date currentTime = new Date();
        Date beginOrEndTime = null;
        boolean beginOrEnd = false;
        switch (flag) {
            case ACTIVITY_HAS_BEGIN:
                beginOrEndTime = (Date) redisTemplate.boundValueOps(buildKeyForActivityBeginTime(organizationCode, activityCode)).get();
                break;
            case ACTIVITY_HAS_END:
                beginOrEndTime = (Date) redisTemplate.boundValueOps(buildKeyForActivityEndTime(organizationCode, activityCode)).get();
                break;
        }
        if (!Lang.isEmpty(beginOrEndTime)) {
            beginOrEnd = currentTime.after(beginOrEndTime);
        }
        return beginOrEnd;
    }

    @Override
    public String reload(String organizationCode, List<String> activityCodes) {
        if (Lang.isEmpty(organizationCode)) {
            return "RELOAD FAIL FOR EMPTY ORGANIZATION CODE!";
        }

        // 20180624 不清除所有活动而根据机构以及活动代码清楚制定活动缓存数据
       /* String key = buildKeyForActivity(organizationCode);
        Set<String> keys = redisTemplate.keys(key);
        if (!Lang.isEmpty(keys)) {
            redisTemplate.delete(keys);
        }*/

        if (Lang.isEmpty(activityCodes)) {
            return "RELOAD FAIL FOR EMPTY ACTIVITY CODE!";
        }

        for (String activityCode : activityCodes) {
            clearActivityKeys(organizationCode, activityCode);
            activityGoodsService.loadStockToRedis(organizationCode, activityCode);
        }

        return organizationCode + " ACTIVITY DATA RELOAD SUCCESS ";
    }

    /**
     * 设置为活动KEY
     *
     * @param key
     */
    @Override
    public void setActivityKey(String organizationCode, String key) {
        String activityKey = buildKeyForActivityKeys(organizationCode);
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(activityKey);
        Object value = boundValueOperations.get();
        Set<String> activityKeySet;
        if (Lang.isEmpty(value)) {
            value = new HashSet<>();
        }
        activityKeySet = (Set<String>) value;
        activityKeySet.add(key);
        boundValueOperations.set(activityKeySet);
    }

    @Override
    public List<ActivityVo> getGoodsListByActivityId(String organizationCode, String activityId, String activityCode, String storeCode) {
        //根据id得到某场次的商品信息
        List<ActivityVo> listImg = new ArrayList<>();
        if (!Lang.isEmpty(activityId)) {
            String activityGoodsKey = buildKeyForActivityGoods(organizationCode, activityId, activityCode);
            BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(activityGoodsKey);
            setActivityKey(organizationCode, activityGoodsKey);
            if (!Lang.isEmpty(boundValueOperations) && !Lang.isEmpty(boundValueOperations.get())) {
                listImg = (List<ActivityVo>) boundValueOperations.get();
            } else {
                // 201804 Drury 版本重构 加载活动数据防止并发大量查询
                String lockKey = buildKeyForLoadActivityGoodsLock(organizationCode, activityId, activityCode, storeCode);
                Boolean isLock = redisTemplate.boundValueOps(lockKey).setIfAbsent(true);
                if (isLock) {
                    try {
                        Long getGoodsTime = System.currentTimeMillis();
                        listImg = getActivity(activityId, storeCode);
                        log.info("加载活动代码为 {} 的商品列表耗时 {} ms", activityCode, System.currentTimeMillis() - getGoodsTime);
                        boundValueOperations.set(listImg);
                    } catch (Exception e) {
                        log.error("加载活动商品列表异常", e);
                    } finally {
                        redisTemplate.delete(lockKey);
                    }
                } else {
                    log.info("并发加载活动商品列表数据等待...");
                    try {
                        Thread.sleep(100);
                        return getGoodsListByActivityId(organizationCode, activityId, activityCode, storeCode);
                    } catch (InterruptedException e) {
                        log.error("线程等待异常", e);
                    }
                }
            }
        }
        return listImg;
    }

    /**
     * 批量获取 GoodsPictureDTO, GoodsPictureDTO 下包含 Goods , 排除 Goods 下所有其表他关联对象
     *
     * @param goodsIds List<String> goodsId 集合
     * @return List<GoodsPictureDto>
     */
    @Override
    public List<GoodsPictureDto> batchQueryGoodsPictureForActivityGoodsList(List<String> goodsIds) {
        long methodStart = System.currentTimeMillis();
        List<GoodsPictureDto> goodsPictureDTOsReturn = new Vector<>();
        goodsIds.parallelStream().forEach(goodsId -> {
            String goodsPictureSql = "SELECT * FROM goods_picture gp WHERE gp.goods_id = ? AND gp.show_index = 0";
            List<activities.dto.goods.GoodsPictureDto> goodsPictureDTOs = jdbcTemplate.query(goodsPictureSql, new BeanPropertyRowMapper<>(activities.dto.goods.GoodsPictureDto.class), goodsId);
            if(!Lang.isEmpty(goodsPictureDTOs)) {
                String goodsSql = "SELECT * FROM goods g WHERE g.id = ?";
                List<activities.dto.goods.GoodsDto> goodsDTOs = jdbcTemplate.query(goodsSql, new BeanPropertyRowMapper<>(activities.dto.goods.GoodsDto.class), goodsId);
                if(!Lang.isEmpty(goodsDTOs)) {
                    activities.dto.goods.GoodsPictureDto goodsPictureDTO = goodsPictureDTOs.get(0);
                    activities.dto.goods.GoodsDto goodsDTO = goodsDTOs.get(0);
                    goodsPictureDTO.setGoods(goodsDTO);
                    goodsPictureDTOsReturn.add(goodsPictureDTO);
                }
            }
        });
        log.info("批量查询活动商品图片列表耗时 {} ms", System.currentTimeMillis() - methodStart);
        return goodsPictureDTOsReturn;
    }

    private List getActivity(String activityDefinitionId, String storeCode) {
        //根据id查找活动定义对象
        long getActivityDefinitionTime = System.currentTimeMillis();
        ActivityDefinition activityDefinition = activityDefinitionRepos.findOne(activityDefinitionId);
        log.info("查询活动场次耗时 {} ms", System.currentTimeMillis() - getActivityDefinitionTime);

        //根据活动定义对象查出所有商品
        List<ActivityVo> activityVoList = new ArrayList<>();
        if (!Lang.isEmpty(activityDefinition)) {
            if (!Lang.isEmpty(activityDefinition.getActivityGoodses())) {
                List<String> goodsIds = activityDefinition.getActivityGoodses().stream().map(ActivityGoods::getGoodsId).collect(Collectors.toList());
                List<GoodsPictureDto> goodsPictureDTOs = batchQueryGoodsPictureForActivityGoodsList(goodsIds);
                goodsPictureDTOs.forEach(goodsPictureDTO -> {
                            ActivityVo activityVo = new ActivityVo();
                    activityVo.setGoodsPicture(goodsPictureDTO);
                            int stockNum = 0;
                            int soldNum = 0;
                            for (ActivityGoods activityGoods : activityDefinition.getActivityGoodses()) {
                        if (activityGoods.getGoodsId().equals(goodsPictureDTO.getGoods().getId())) {
                                    stockNum = Lang.isEmpty(activityGoods.getStockNum()) ? 0 : activityGoods.getStockNum();
                                    soldNum = Lang.isEmpty(activityGoods.getSoldNum()) ? 0 : activityGoods.getSoldNum();
                                    break;
                                }
                            }
                            //计算已售百分比
                            String soldPercent = calcStockPercent(stockNum, soldNum);
                            activityVo.setStockPercent(soldPercent);
                            activityVoList.add(activityVo);
                });
            }
            log.info("活动商品列表大小 {}", activityVoList.size());
            if(!Lang.isEmpty(activityVoList)) {
                // 批量获取活动商品售价
                long setActivityPriceTime = System.currentTimeMillis();
                activityGoodsApi.setActivityPrice(activityVoList, activityDefinition.getOrganizationCode(), storeCode);
                log.info("批量设置活动售价耗时 {} ms", System.currentTimeMillis() - setActivityPriceTime);
            }
        }
        return activityVoList;
    }

    @Override
    public Map<String, Object> getSystemTime(Integer activityIndex, String organizationCode, String activityCode) {
        //返回结果
        Map<String, Object> map = new HashMap<>();
        if (!Lang.isEmpty(activityIndex) && !Lang.isEmpty(activityCode)) {

            //所有活动已结束
            boolean noActivity = false;
            //本场次已开始
            boolean hasStart = false;
            //本场次已结束
            boolean hasEnd = false;
            //得到当前系统时间
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            Date currentTime = new Date();

            Date nextSecKillStartTime = null;
            Date currentSecKillEndTime = null;
            String currentActivityDefinitionId;
            String oldActivityDefinitionId;

            if (!activityBeginOrEnd(organizationCode, activityCode, ACTIVITY_HAS_END)) {
                String activityInitTimeKey = buildKeyForActivityInitTime(organizationCode, activityCode);
                BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(activityInitTimeKey);
                setActivityKey(organizationCode, activityInitTimeKey);
                if (!Lang.isEmpty(boundValueOperations) && !Lang.isEmpty(boundValueOperations.get())) {
                    List<Map<String, Object>> result = (List<Map<String, Object>>) boundValueOperations.get();
                    if (result.size() > 0) {
                        Date startTime = (Date) result.get(activityIndex).get("beginTime");
                        hasStart = (currentTime.after(startTime) || sdf.format(currentTime).equals(sdf.format(startTime)));
                        nextSecKillStartTime = (Date) result.get(activityIndex).get("beginTime");
                        currentSecKillEndTime = (Date) result.get(activityIndex).get("endTime");
                        if (hasStart) {
                            //本场次已开始
                            currentActivityDefinitionId = (String) result.get(0).get("id");
                            //将当前场次放入redis
                            String currentActivityIdKey = buildKeyForCurrentActivityId(organizationCode, activityCode);
                            BoundValueOperations boundValueOperationsId = redisTemplate.boundValueOps(currentActivityIdKey);
                            setActivityKey(organizationCode, currentActivityIdKey);

                            if (!Lang.isEmpty(boundValueOperationsId) && !Lang.isEmpty(boundValueOperationsId.get())) {
                                oldActivityDefinitionId = (String) boundValueOperationsId.get();
                                if (!currentActivityDefinitionId.equals(oldActivityDefinitionId)) {
                                    boundValueOperationsId.set(currentActivityDefinitionId);
                                }
                            } else {
                                boundValueOperationsId.set(currentActivityDefinitionId);
                            }
                            //查看本场次是否结束
                            Date endTime = (Date) result.get(activityIndex).get("endTime");
                            if (currentTime.after(endTime) || sdf.format(currentTime).equals(sdf.format(endTime))) {
                                //将刚结束的场次置为失效
                                activityApi.setActivityDefinitionIsOnline((String) result.get(activityIndex).get("id"));
                                //从当前redis中删除刚结束的场次
                                result.remove((int) activityIndex);
                                boundValueOperations.set(result);
                                hasEnd = true;
                            }
                        }
                    }
                }
            } else {
                String currentActivityIdKey = buildKeyForCurrentActivityId(organizationCode, activityCode);
                BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(currentActivityIdKey);
                if (!Lang.isEmpty(boundValueOperations) && !Lang.isEmpty(boundValueOperations.get())) {
                    activityApi.setActivityDefinitionIsOnline((String) boundValueOperations.get());
                    redisTemplate.delete(currentActivityIdKey);
                }
                //活动结束
                noActivity = true;
            }

            if (activityIndex > 0) {
                map.put("nextSecKillStartTime", nextSecKillStartTime);
            } else {
                map.put("isStart", hasStart);
                map.put("nextSecKillStartTime", nextSecKillStartTime);
                map.put("currentSecKillEndTime", currentSecKillEndTime);
            }
            map.put("isEnd", hasEnd);
            map.put("systemTime", currentTime);
            map.put("noActivity", noActivity);
        }
        return map;
    }

    @Override
    public String getCurrentActivityId(String organizationCode, String activityCode) {
        String key = buildKeyForCurrentActivityId(organizationCode, activityCode);
        String currentActivityId = (String) redisTemplate.boundValueOps(key).get();
        return currentActivityId;
    }

    @Override
    public String getCurrentActivityId(String organizationCode, String goodsId, String activityCode) {
        String key = buildKeyForCurrentActivityId(organizationCode, activityCode);
        String currentActivityId = (String) redisTemplate.boundValueOps(key).get();
        if (!Lang.isEmpty(currentActivityId) && !Lang.isEmpty(goodsId)) {
            List<ActivityGoodsDto> activityGoodsList = activityGoodsApi.findActivityGoodsByGoodsIdAndActivityId(goodsId, currentActivityId);
            if (activityGoodsList.size() != 1) {
                currentActivityId = null;
            }
        } else {
            currentActivityId = null;
        }
        return currentActivityId;
    }

    /**
     * 查询redis是否存在
     *
     * @param key
     * @return 活动id
     */
    @Override
    public boolean redisHas(String key) {
        boolean hasValue = false;
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
        if (!Lang.isEmpty(boundValueOperations) && !Lang.isEmpty(boundValueOperations.get())) {
            hasValue = true;
        }
        return hasValue;
    }

    /**
     * 存储redis
     *
     * @param key
     * @return 活动id
     */
    @Override
    public void redisSave(String key, Object obj) {
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
        boundValueOperations.set(obj);
    }

    /**
     * 得到redis
     *
     * @param key
     * @return 活动id
     */
    public Object redisReturn(String key) {
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
        return boundValueOperations.get();
    }

    /**
     * 删除redis
     *
     * @param key
     */
    @Override
    public void deleteRedisKey(String key) {
        redisTemplate.delete(key);
    }

    /**
     * 从缓存中获取活动商品库存数据
     *
     * @param organizationCode 机构代码
     * @return map集合 对应缓存 redis 的 key value
     */
    @Override
    public Map<String, Integer> getStockFromRedis(String organizationCode, String activityCode) {
        Map<String, Integer> stockMap = new HashMap<>();
        String key = buildKeyForActivityStock(organizationCode, activityCode);
        Set<String> keySet = redisTemplate.keys(key);
        for (String keyTemp : keySet) {
            stockMap.put(keyTemp, (Integer) redisTemplate.boundValueOps(keyTemp).get());
        }
        return stockMap;
    }

    @Override
    public String getStockPercent(String activityDefinitionId, String activityGoodsId) {
        List<ActivityGoodsDto> activityGoodsList = activityGoodsApi.findActivityGoodsByGoodsIdAndActivityId(activityGoodsId, activityDefinitionId);
        //计算已售百分比
        return calcStockPercent(activityGoodsList.get(0).getStockNum(), activityGoodsList.get(0).getSoldNum());
    }

    /**
     * 计算已售百分比
     *
     * @param stockNum 库存剩余数量
     * @param soldNum  已售数量
     * @return String
     */
    private String calcStockPercent(int stockNum, int soldNum) {
        String soldPercent = "100%";
        BigDecimal originalStock = new BigDecimal(stockNum).add(new BigDecimal(soldNum));
        if (!originalStock.equals(BigDecimal.ZERO)) {
            BigDecimal sold = new BigDecimal(soldNum).divide(originalStock, 2, BigDecimal.ROUND_HALF_UP);
            NumberFormat num = NumberFormat.getPercentInstance();
            num.setMaximumIntegerDigits(3);
            num.setMaximumFractionDigits(2);
            soldPercent = num.format(sold);
        }
        return soldPercent;
    }

    public String buildKeyForActivity(String organizationCode) {
        // 格式样例：chengtongrenli:activity:keys
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append("*");
        return key.toString();
    }

    public String buildKeyForActivityKeys(String organizationCode) {
        // 格式样例：chengtongrenli:activity:keys
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(KEYS);
        return key.toString();
    }

    @Override
    public String buildKeyForCurrentActivityId(String organizationCode, String activityCode) {
        // 格式样例：chengtongrenli:activity:seckill:current-activity-id
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(CURRENT_ACTIVITY_ID);
        return key.toString();
    }

    public String buildKeyForActivityInitTime(String organizationCode, String activityCode) {
        // 格式样例：chengtongrenli:activity:seckill:init-time
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(INIT_TIME);
        return key.toString();

    }

    @Override
    public String buildKeyForActivityStock(String organizationCode, String activityCode) {
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(STOCK);
        key.append("*");
        return key.toString();
    }

    @Override
    public String buildKeyForActivityStock(String organizationCode, String goodsId, String activityId, String activityCode) {
        // 格式样例：chengtongrenli:activity:seckill:stock:商品ID_场次ID
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(STOCK);
        key.append(SPLITER);
        key.append(goodsId).append("_").append(activityId);
        return key.toString();
    }

    /**
     * 批量构建库存缓存 KEY
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @param activityId 活动场次ID
     * @param goodsIds 商品ID列表
     * @return Map<String, String> KEY = goodsId, VALUE = 库存缓存 KEY
     */
    @Override
    public Map<String, String> buildKeyForActivityStock(String organizationCode, String activityCode, String activityId, List<String> goodsIds) {
        // 格式样例：chengtongrenli:activity:seckill:stock:商品ID_场次ID
        Map<String, String> keyMap = new HashMap<>(goodsIds.size());
        goodsIds.forEach(goodsId -> {
            StringBuilder key = new StringBuilder();
            key.append(organizationCode);
            key.append(SPLITER);
            key.append(ACTIVITY);
            key.append(SPLITER);
            key.append(activityCode);
            key.append(SPLITER);
            key.append(STOCK);
            key.append(SPLITER);
            key.append(goodsId).append("_").append(activityId);
            keyMap.put(goodsId, key.toString());
        });
        return keyMap;
    }

    public String buildKeyForActivityGoods(String organizationCode, String activityId, String activityCode) {
        // 格式样例：chengtongrenli:activity:seckill:goods:场次ID
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(GOODS);
        key.append(SPLITER);
        key.append(activityId);
        return key.toString();
    }

    public String buildKeyForActivityBeginTime(String organizationCode, String activityCode) {
        // 格式样例：chengtongrenli:activity:seckill:begin-time
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(BEGIN_TIME);
        return key.toString();
    }

    public String buildKeyForActivityEndTime(String organizationCode, String activityCode) {
        // 格式样例：chengtongrenli:activity:seckill:end-time
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(END_TIME);
        return key.toString();
    }

    /**
     * 构建加载活动商品列表并发锁 KEY
     *
     * @param organizationCode 机构代码
     * @param activityId 活动ID
     * @param activityCode 活动代码
     * @param storeCode 店铺代码
     * @return String
     */
    private String buildKeyForLoadActivityGoodsLock(String organizationCode, String activityId, String activityCode, String storeCode) {
        // 格式样例：chengtongrenli:activity:lock:店铺代码:seckill:goods:场次ID
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(LOCK);
        key.append(SPLITER);
        key.append(storeCode);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(GOODS);
        key.append(SPLITER);
        key.append(activityId);
        return key.toString();
    }

    /**
     * 构建加载活动商品列表并发锁 KEY
     *
     * @param organizationCode 机构代码
     * @param activityId 活动ID
     * @param activityCode 活动代码
     * @param storeCode 店铺代码
     * @return String
     */
    @Override
    public boolean lockForLoadActivityGoods(String organizationCode, String activityId, String activityCode, String storeCode) {
        String lockKey = buildKeyForLoadActivityGoodsLock(organizationCode, activityId, activityCode, storeCode);
        BoundValueOperations loadActivityGoodsLocker = redisTemplate.boundValueOps(lockKey);
        Boolean isLocked = loadActivityGoodsLocker.setIfAbsent(true);
        if(isLocked) {
            loadActivityGoodsLocker.expire(1, TimeUnit.MINUTES);
        }
        return isLocked;
    }

    /**
     * 为当前加载活动线程解锁
     *
     * @param organizationCode 机构代码
     * @param activityId 活动ID
     * @param activityCode 活动代码
     * @param storeCode 店铺代码
     */
    @Override
    public void unlockForLoadActivityGoods(String organizationCode, String activityId, String activityCode, String storeCode){
        String lockKey = buildKeyForLoadActivityGoodsLock(organizationCode, activityId, activityCode, storeCode);
        redisTemplate.delete(lockKey);
    }

    /**
     * 构建当前场次活动信息 key
     *
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return String
     */
    @Override
    public String buildKeyForCurrentActivityInfo(String organizationCode, String activityCode) {
        // 格式样例：itaipingMobile:activity:vourcherCenter:activity-info
        StringBuilder key = new StringBuilder();
        key.append(organizationCode);
        key.append(SPLITER);
        key.append(ACTIVITY);
        key.append(SPLITER);
        key.append(activityCode);
        key.append(SPLITER);
        key.append(ACTIVITY_INFO);
        return key.toString();
    }

    /**
     * 批量设置缓存数据
     *
     * @param map Map<Object, Object>
     */
    @Override
    public void redisSave(Map<Object, Object> map) {
        map.keySet().forEach(key -> redisTemplate.boundValueOps(key).set(map.get(key)));
    }

    /**
     * 清除指定机构代码下指定活动代码缓存数据
     *
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     */
    @Override
    public void clearActivityKeys(String organizationCode, String activityCode) {
        StringBuilder keyPattern = new StringBuilder();
        keyPattern.append(organizationCode);
        keyPattern.append(SPLITER);
        keyPattern.append(ACTIVITY);
        keyPattern.append(SPLITER);
        keyPattern.append(activityCode);
        keyPattern.append("*");
        Set keys = redisTemplate.keys(keyPattern.toString());
        redisTemplate.delete(keys);
    }

    /**
     * 从缓存中获取活动信息
     *
     * @param organizationCode 机构代码
     * @param activityCode 活动代码
     * @return ActivityInfoVO
     */
    @Override
    public ActivityInfoVO getActivityInfo(String organizationCode, String activityCode) {
        return (ActivityInfoVO) redisTemplate.boundValueOps(buildKeyForCurrentActivityInfo(organizationCode, activityCode)).get();
    }

}
