package activities.service.apiImpl;

import activities.api.ActivityApi;
import activities.api.ActivityConfigurationApi;
import activities.api.ActivityGoodsApi;
import activities.definition.ActivityDefinitionType;
import activities.definition.ActivityPriceType;
import activities.dto.activity.request.ActivityGoodsReqDto;
import activities.dto.activity.response.ActivityGoodsRespDto;
import activities.dto.modeldto.ActivityDefinitionDto;
import activities.dto.modeldto.ActivityGoodsDto;
import activities.model.ActivityDefinition;
import activities.model.ActivityGoods;
import activities.model.repository.ActivityDefinitionRepos;
import activities.model.repository.ActivityGoodsRepos;
import activities.service.service.ActivityGoodsService;
import activities.service.service.ActivityService;
import activities.vo.ActivityVo;
import cms.api.OrganizationApi;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import goods.api.GoodsApi;
import goods.api.SalesPriceRateQueryApi;
import goods.dto.goods.GoodsDto;
import goods.dto.goods.GoodsPictureDto;
import goods.model.Goods;
import goods.model.repository.GoodsRepos;
import goods.vo.SalesPriceQueryVo;
import ma.glasnost.orika.MapperFacade;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.transaction.annotation.Transactional;
import store.api.StoreApi;
import store.api.dto.modeldto.core.StoreDto;
import utils.Lang;
import utils.data.BeanMapper;
import utils.lang.Copys;
import utils.log.Log;
import utils.log.Logs;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by oracle on 2017-07-10.
 */
@MotanService
@Transactional
public class ActivityGoodsApiImpl implements ActivityGoodsApi {

    private final static Log log = Logs.getLog(ActivityGoodsApiImpl.class.getName());

    @Autowired
    ActivityGoodsRepos activityGoodsRepos;
    @MotanReferer
    SalesPriceRateQueryApi salesPriceRateQueryApi;
    @MotanReferer
    GoodsApi goodsApi;
    @MotanReferer
    StoreApi storeApi;
    @Autowired
    JdbcTemplate jdbcTemplate;
    @Autowired
    ActivityDefinitionRepos activityDefinitionRepos;

    @Autowired
    ActivityApi activityApi;

    @Autowired
    MapperFacade mapperFacade;

    @Autowired
    ActivityGoodsService activityGoodsService;

    @Autowired
    ActivityService activityService;
    @Autowired
    ActivityConfigurationApi activityConfigurationApi;

    @Value("${itaipingMobile.organization.code}")
    private String itaipingMobileOrganizationCode;

    @Autowired
    GoodsRepos goodsRepos;

    @MotanReferer
    OrganizationApi organizationApi;

    private RedisTemplate redisTemplate;

    @Autowired
    public void setRedisTemplate(RedisTemplate redisTemplate) {
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        this.redisTemplate = redisTemplate;
    }

    @Override
    public List<Map<String, Object>> isActivityGoodsAndGetActivityPriceBatch(List<activities.dto.goods.GoodsDto> goodsTemps, String organizationCode) {
        if (Lang.isEmpty(goodsTemps)) {
            return null;
        }
        List<Map<String, Object>> prices = new ArrayList<>();
        for (activities.dto.goods.GoodsDto goodsTemp : goodsTemps) {
            Map<String, Object> activityPrice = isActivityGoodsAndGetActivityPrice(goodsTemp, organizationCode);
            prices.add(activityPrice);
        }
        return prices;
    }


    @Override
    public Map<String, Object> isActivityGoodsAndGetActivityPrice(activities.dto.goods.GoodsDto goodsParam, String organizationCode) {
        Map<String, Object> activitiesGoodsInfo = new HashMap<>();
        if (Lang.isEmpty(goodsParam)) {
            activitiesGoodsInfo.put("isActivityGoods", false);
            activitiesGoodsInfo.put("activityPrice", new BigDecimal(0));
        } else {
            Goods goods = mapperFacade.map(goodsParam, Goods.class);
            activitiesGoodsInfo.put("goodsId", goods.getId());
            List<Map<String, Object>> pricesInfo = activityGoodsRepos.findActivityPrice(goods.getId(), organizationCode);
            if (pricesInfo == null || pricesInfo.isEmpty()) {
                activitiesGoodsInfo.put("isActivityGoods", false);
                return activitiesGoodsInfo;
            }
            Map<String, Object> objectMap = pricesInfo.get(0);
            activitiesGoodsInfo.put("isActivityGoods", true);
            activitiesGoodsInfo.put("activityId", objectMap.get("activityId"));
            if (ActivityGoodsDto.ACTIVITY_PRICE_TYPE_WITHOUTRATE.equals(objectMap.get("type"))) {  //不走费率
                activitiesGoodsInfo.put("activityPrice", goods.getCostPrice());
            } else if (ActivityGoodsDto.ACTIVITY_PRICE_TYPE_ACPRICE.equals(objectMap.get("type"))) {  //按活动价
                activitiesGoodsInfo.put("activityPrice", new BigDecimal(objectMap.get("activityPrice") + ""));
            } else if (ActivityGoodsDto.ACTIVITY_PRICE_TYPE_WITHRATE.equals(objectMap.get("type"))) {  //走费率
                goods = goodsRepos.findOne(goods.getId());
                /**Akers Liang 2017-04-21修改： 多店铺适配，根据产品所属店铺获取StoreCode START */
                StoreDto store = storeApi.getStoreInfo(goods.getStoreId());
                String storeCode = store.getStoreExt().getCode();
                /**Akers Liang 2017-04-21修改： 多店铺适配，根据产品所属店铺获取StoreCode END */
                GoodsDto goodsDTO = new GoodsDto();
                Copys.create().from(goods).excludes("product", "goodsAttrs", "goodsPictures", "goodsFashions", "goodsParams", "productFashion").to(goodsDTO);
                SalesPriceQueryVo salesPriceQueryVo = new SalesPriceQueryVo(organizationCode, storeCode, goodsDTO);
                List<Map> mapList = salesPriceRateQueryApi.getSalesPriceListByParams(Arrays.asList(salesPriceQueryVo));
                if (!Lang.isEmpty(mapList)) {
                    BigDecimal salePrice = (BigDecimal) mapList.get(0).get("salePrice");
                    if (!Lang.isEmpty(salePrice)) {
                        activitiesGoodsInfo.put("activityPrice", salePrice);
                    }
                }
            }
        }
        return activitiesGoodsInfo;
    }


    public ActivityDefinitionType getActivityDefinitionType(ActivityDefinition activityDefinition) {
        Date now = new Date();
        ActivityDefinitionType activityDefinitionType = null;
        if (now.before(activityDefinition.getBeginTime())) {
            activityDefinitionType = ActivityDefinitionType.beforeStart;
        } else if (now.after(activityDefinition.getBeginTime()) && now.before(activityDefinition.getEndTime())) {
            activityDefinitionType = ActivityDefinitionType.onGoing;
        } else if (now.after(activityDefinition.getEndTime())) {
            activityDefinitionType = ActivityDefinitionType.end;
        }
        return activityDefinitionType;
    }

    @Override
    public ActivityGoodsRespDto isActivityGoodsAndGetActivityPrice(ActivityGoodsReqDto activityGoodsReqDto) {
        ActivityGoods activityGoods = activityGoodsRepos.findByGoodsIdAndActivityDefinitionIdAndIsOnlineAndIsDelete(activityGoodsReqDto.getGoodsId(), activityGoodsReqDto.getActivityInfoId(), true, false);
        ActivityGoodsRespDto activityGoodsRespDto = new ActivityGoodsRespDto();
        if (activityGoods != null) {
            GoodsDto goods = null;
            BigDecimal activityPrice = null;
            ActivityDefinition activityDefinition = activityGoods.getActivityDefinition();
            switch (ActivityPriceType.activityPriceTypeMap.get(activityGoods.getType())) {
                case ratePrcie:
                    Goods goodsPO = goodsRepos.findOne(activityGoods.getGoodsId());
                    goods = new GoodsDto();
                    Copys.create().from(goodsPO).excludes("product", "goodsAttrs", "goodsPictures", "goodsFashions", "goodsParams", "productFashion").to(goods);
                    StoreDto store = storeApi.getStoreInfo(goods.getStoreId());
                    String storeCode = store.getStoreExt().getCode();
                    SalesPriceQueryVo salesPriceQueryVo = new SalesPriceQueryVo(activityDefinition.getOrganizationCode(), storeCode, goods);
                    List<Map> mapList = salesPriceRateQueryApi.getSalesPriceListByParams(Arrays.asList(salesPriceQueryVo));
                    if (!Lang.isEmpty(mapList)) {
                        activityPrice = (BigDecimal) mapList.get(0).get("salePrice");
                    }
                    break;
                case negotiatedPrice:
                    activityPrice = goodsApi.findCostPriceById(activityGoods.getGoodsId());
                    break;
                case configurationPrice:
                    activityPrice = activityGoods.getGoodsPrice();
                    break;
                    default:
                        break;
            }
            activityGoodsRespDto.setActivitylPrice(activityPrice);
            ActivityDefinitionType activityDefinitionType = getActivityDefinitionType(activityDefinition);
            activityGoodsRespDto.setActivityDefinitionType(activityDefinitionType);
        } else {
            activityGoodsRespDto.setActivityDefinitionType(ActivityDefinitionType.noExist);
        }
        return activityGoodsRespDto;
    }


    @Override
    public void increamentSaleCount(String activityDefinitionId, String goodsId, Integer buyCount) {
        String sql = "update Activity_Goods set sold_Num=sold_Num + ? where activity_Definition_id=? and goods_Id=?";
        jdbcTemplate.update(sql, buyCount, activityDefinitionId, goodsId);

    }


    /**
     * 根据商品ID获取活动商品
     *
     * @param goodsId    商品ID
     * @param activityId 活动ID
     * @return 活动商品
     */
    @Override
    public List<ActivityGoodsDto> findActivityGoodsByGoodsIdAndActivityId(String goodsId, String activityId) {
        ActivityDefinition activityDefinition = new ActivityDefinition();
        activityDefinition.setId(activityId);
        List<ActivityGoods> activityGoodsList = activityGoodsRepos.findByGoodsIdAndActivityDefinitionAndIsDelete(goodsId, activityDefinition, false);
        return BeanMapper.mapList(activityGoodsList, ActivityGoods.class, ActivityGoodsDto.class);
    }

    /**
     * 活动商品减库存
     *
     * @param goodsId    当前活动商品ID
     * @param activityId 当前活动ID
     * @param count      占用库存数
     * @return 成功返回true, 失败返回false
     */
    @Override
    public boolean changeActivityGoodsStock(String goodsId, String activityId, int count) {
        List<ActivityGoodsDto> activityGoodsList = findActivityGoodsByGoodsIdAndActivityId(goodsId, activityId);
        if (activityGoodsList.size() == 1) {
            ActivityGoodsDto activityGoods = activityGoodsList.get(0);
            String key = activityConfigurationApi.buildKeyForActivityStock(activityGoods.getActivityDefinition().getOrganizationCode(), goodsId, activityId, activityGoods.getActivityDefinition().getActivityCode());
            BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
            if (!Lang.isEmpty(boundValueOperations.get())) {
                log.info("=====================活动商品减库存goodsId={}，activityId={}，count={}", goodsId, activityId, count);
                Long number = boundValueOperations.increment(-count);
                log.info("=====================活动商品减库存后剩余goodsId={}，activityId={}，count={}，number={}", goodsId, activityId, count, number);
                if (number >= 0) {
                    //todo 高并发场景下不能直接写sql
                    int updateRow = changeActivityGoodsStockFromDataBase(activityGoodsList.get(0).getId(), count);
                    log.info("=====================活动商品更新数据库后影响行数goodsId={}，activityId={}，count={}，updateRow={}", goodsId, activityId, count, updateRow);
                    if (updateRow > 0) {
                        return true;
                    } else {
                        boundValueOperations.increment(count);
                    }
                } else {
                    boundValueOperations.increment(count);
                }
            } else {
                log.error("缓存中没有找到匹配商品ID{}与场次ID{}的库存数据！", goodsId, activityId);
            }
        } else {
            log.error("根据商品ID{}与场次ID{}找到数据集合大小不是期望值[1]实际大小[{}]", goodsId, activityId, activityGoodsList.size());
        }
        return false;
    }

    /**
     * 从缓存中获取当前活动商品库存
     *
     * @param goodsId 当前参与活动的商品ID
     * @return 参与活动的商品库存
     */
    @Override
    public int getCurrentActivityGoodsStockFromCache(String organizationCode, String goodsId, String activityId, String activityCode) {
        int stock = 0;
        String key = activityConfigurationApi.buildKeyForActivityStock(organizationCode, goodsId, activityId, activityCode);
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
        if (!Lang.isEmpty(boundValueOperations.get())) {
            stock = (int) boundValueOperations.get();
        }
        return stock;
    }

    /**
     * 数据库减库存
     *
     * @param activityGoodsId 活动商品ID
     * @param count           削减数量
     * @return
     */
    private int changeActivityGoodsStockFromDataBase(String activityGoodsId, int count) {
        String sql = "UPDATE activity_goods ag SET ag.sold_num = (ag.sold_num + ?), ag.stock_num = (ag.stock_num - ?) WHERE ag.id = ? AND ag.stock_num > 0";
        return jdbcTemplate.update(sql, new Object[]{count, count, activityGoodsId});
    }


    /**
     * 获得活动商品价格
     *
     * @param goodsId          参与活动的商品ID
     * @param organizationCode 机构代码
     * @param storeCode        商店代码
     * @return 如果传入的goodsId是活动商品ID则返回活动价格，否则返回普通售卖价格
     */
    @Override
    public BigDecimal getActivityPrice(String goodsId, String organizationCode, String storeCode) {
       /* BigDecimal activityPrice = null;
        GoodsDto goodsDto = goodsApi.findById(goodsId);
        if (!Lang.isEmpty(goodsDto)) {
            List<Map> Price = salesPriceRateQueryApi.getSalesPriceListByParams(Arrays.asList(new SalesPriceQueryVo(organizationCode, storeCode, goodsDto)));
            if (!Lang.isEmpty(Price) && Price.size() > 0) {
                activityPrice = (BigDecimal) Price.get(0).get("salePrice");
            }
        }
        activities.dto.goods.GoodsDto goodsDtoTemp = mapperFacade.map(goodsDto, activities.dto.goods.GoodsDto.class);
        Map activityPriceMap = isActivityGoodsAndGetActivityPrice(goodsDtoTemp, organizationCode);
        Boolean isActivityGoods = (Boolean) activityPriceMap.get("isActivityGoods");
        if (isActivityGoods && activityPriceMap.containsKey("activityPrice")) {
            activityPrice = (BigDecimal) activityPriceMap.get("activityPrice");
        }
        return activityPrice;*/

        // 20180322 因速度太慢直接使用goodsRepos
        Long getSalePriceTime = System.currentTimeMillis();
        BigDecimal salePrice = getActivityPriceNew(goodsId, organizationCode, storeCode);
        log.info("获取商品价格 goodsId = {}, organizationCode = {}, storeCode = {}, SalePrice = {} 耗时 {} ms", goodsId, organizationCode, storeCode, salePrice, System.currentTimeMillis() - getSalePriceTime);
        return salePrice;
    }

    /**
     * 获得活动场次有效订单数
     *
     * @param memberId   用户ID
     * @param activityId 活动场次ID
     * @return 有效订单数量
     */
    @Override
    public BigDecimal getEffectiveOrderCount(String memberId, String activityId) {
        return activityDefinitionRepos.hasEffectiveOrder(memberId, activityId);
    }

    /**
     * 改变活动商品库存以及售出数量
     *
     * @param activityId 活动ID
     * @param goodsId    参与活动的商品ID
     * @param buyNum     购买数量
     */
    @Override
    public void increamentSoldNumAndStockNum(String activityId, String goodsId, Integer buyNum) {
        String sql = "update Activity_Goods set sold_Num = (sold_Num + ?), stock_Num = (stock_Num - ?) where activity_Definition_id = ? and goods_Id = ?";
        jdbcTemplate.update(sql, buyNum, buyNum, activityId, goodsId);
    }

    /**
     * 反活动商品库存
     *
     * @param goodsId          参与活动的商品ID
     * @param activityId       活动ID
     * @param count            返还库存数量
     * @param organizationCode 机构代码
     */
    @Override
    public void backActivityGoodsStockAndSolcNum(String goodsId, String activityId, Integer count, String organizationCode) {
        increamentSoldNumAndStockNum(activityId, goodsId, -count);
        ActivityDefinitionDto activityDefinition = activityApi.findById(activityId);
        String key = activityConfigurationApi.buildKeyForActivityStock(organizationCode, goodsId, activityId, activityDefinition.getActivityCode());
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
        if (!Lang.isEmpty(boundValueOperations.get())) {
            boundValueOperations.increment(count);
        }
    }

    /**
     * 是否场次限制
     *
     * @param memberId     用户ID
     * @param activityId   活动场次ID
     * @param limitedCount 限制数量
     * @return
     */
    @Override
    public boolean isActivityLimited(String memberId, String activityId, int limitedCount) {
        BigDecimal count = getEffectiveOrderCount(memberId, activityId);
        return count.intValue() >= limitedCount;
    }

    @Override
    public ActivityDefinitionType getActivityDefinitionType(ActivityDefinitionDto activityDefinition) {
        Date now = new Date();
        ActivityDefinitionType activityDefinitionType = null;
        if (now.before(activityDefinition.getBeginTime())) {
            activityDefinitionType = ActivityDefinitionType.beforeStart;
        } else if (now.after(activityDefinition.getBeginTime()) && now.before(activityDefinition.getEndTime())) {
            activityDefinitionType = ActivityDefinitionType.onGoing;
        } else if (now.after(activityDefinition.getEndTime())) {
            activityDefinitionType = ActivityDefinitionType.end;
        }
        return activityDefinitionType;
    }

    @Override
    public List<ActivityGoodsDto> findByGiftClass(String giftClass,String activityDefinitionId) {
        List<ActivityGoods> activityGoods = activityGoodsRepos.findByGiftClassAndActivityDefinitionId(giftClass, activityDefinitionId);
        activityGoods = activityGoods.stream().filter(activityGoods1 -> !activityGoods1.getIsDelete() && activityGoods1.getIsOnline()).sorted(Comparator.comparing(ActivityGoods::getShowIndex)).collect(Collectors.toList());
        return mapperFacade.mapAsList(activityGoods, ActivityGoodsDto.class);
    }

    @Override
    public ActivityGoodsDto saveActivityGoods(ActivityGoodsDto activityGoodsDto) {
        if (Lang.isEmpty(activityGoodsDto)) {
            return null;
        }
        ActivityGoods activityGoods = activityGoodsRepos.save(mapperFacade.map(activityGoodsDto, ActivityGoods.class));
        if (Lang.isEmpty(activityGoods)) {
            return null;
        }
        return mapperFacade.map(activityGoods, ActivityGoodsDto.class);

    }

    /**
     * 是否商品当前处于活动期间
     *
     * @param goodsId          商品ID
     * @param organizationCode 机构代码
     * @return 如果处于活动期间, 则返回活动场次ID, 否则返回 null
     */
    @Override
    public String isInActivityNow(String goodsId, String organizationCode) {
        // 活动信息查询
        List<Map<String, Object>> activityMaps = activityGoodsRepos.findActivityPrice(goodsId, organizationCode);
        if (Lang.isEmpty(activityMaps)) {
            return null;
        }
        return (String) activityMaps.get(0).get("activityId");
    }

    /**
     * 批量设置活动商品售价
     *
     * @param activityVoList   List<ActivityVo>
     * @param organizationCode 机构代码
     * @param storeCode        店铺代码
     */
    @Override
    public void setActivityPrice(List<ActivityVo> activityVoList, String organizationCode, String storeCode) {
        // 需要费率查询集合, 应为之前已经查出 GoodsDTO 集合, 没必要让底服务层再去查一遍
        List<GoodsDto> ratePriceQueryGoodsList = new ArrayList<>();
        // 批量查询费率请求参数
        List<String> goodsIds = activityVoList.stream().map(ActivityVo::getGoodsPicture).map(activities.dto.goods.GoodsPictureDto::getGoods).map(activities.dto.goods.GoodsDto::getId).collect(Collectors.toList());

        long activityMapsTime = System.currentTimeMillis();
        List<Map<String, Object>> activityMaps = activityGoodsRepos.findActivityPrice(goodsIds, organizationCode);
        log.info("活动信息查询耗时 {} ms", System.currentTimeMillis() - activityMapsTime);

        for (ActivityVo activityVO : activityVoList) {
            GoodsDto goodsDTO = BeanMapper.map(activityVO.getGoodsPicture().getGoods(), GoodsDto.class);
            if (!Lang.isEmpty(activityMaps)) {
                for (Map<String, Object> activityMap : activityMaps) {
                    String goodsId = (String) activityMap.get("goodsId");
                    if(goodsDTO.getId().equals(goodsId)) {
                        // 售价方案
                        String keyForPriceType = "type";
                        String priceType = (String) activityMap.get(keyForPriceType);
                        // 活动配置价格
                        if (ActivityGoodsDto.ACTIVITY_PRICE_TYPE_ACPRICE.equals(priceType)) {
                            String keyForActivitySetPrice = "activityPrice";
                            Object activityPrice = activityMap.get(keyForActivitySetPrice);
                            if (!Lang.isEmpty(activityPrice)) {
                                activityVO.getGoodsPicture().getGoods().setSalePrice(new BigDecimal(String.valueOf(activityPrice)));
                                continue;
                            }
                            log.warn("使用活动配置价格方案失败 : 活动配置价格为空, 将使用默认费率价格方案");
                            continue;
                        } else if(ActivityGoodsDto.ACTIVITY_PRICE_TYPE_WITHOUTRATE.equals(priceType)) {
                            activityVO.getGoodsPicture().getGoods().setSalePrice(goodsDTO.getCostPrice());
                            continue;
                        }
                        // 默认费率查询
                        ratePriceQueryGoodsList.add(goodsDTO);
                    }
                }
            }
        }

        // 如果有调用批量查询费率售价接口
        if (!Lang.isEmpty(ratePriceQueryGoodsList)) {
            String organizationId = organizationApi.findOrganizationIdByOrgCode(organizationCode);
            // 查询默认使用的费率价格
            long salesPriceRateQueryTime = System.currentTimeMillis();
            Map<String, BigDecimal> salePriceMap = salesPriceRateQueryApi.getSaleRatePriceBatchByGoodsList(ratePriceQueryGoodsList, storeCode, organizationId);
            log.info("批量查询费率价格 (GoodsDTO集合方式) 耗时 {} ms", System.currentTimeMillis() - salesPriceRateQueryTime);

            salePriceMap.keySet().forEach(goodsId -> {
                for (ActivityVo activityVo : activityVoList) {
                    if(goodsId.equals(activityVo.getGoodsPicture().getGoods().getId())) {
                        activityVo.getGoodsPicture().getGoods().setSalePrice(salePriceMap.get(goodsId).setScale(2, BigDecimal.ROUND_UP));
                        break;
                    }
                }
            });
        }

    }

    /**
     * 根据样例数据获取活动商品列表
     *
     * @param activityGoodsExample ActivityGoodsDto
     * @return List<ActivityGoodsDto>
     */
    @Override
    public List<ActivityGoodsDto> findByExample(ActivityGoodsDto activityGoodsExample) {
        return activityGoodsService.findByExample(activityGoodsExample);
    }

    @Override
    public boolean checkCacheStock(String organizationCode, String activityId, String activityCode, String goodsId, Integer count) {
        String key = activityConfigurationApi.buildKeyForActivityStock(organizationCode, goodsId, activityId, activityCode);
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
        Long stock = (Long) boundValueOperations.get();
        if (!Lang.isEmpty(stock) && stock - count > 0) {
            return true;
        } else {
            log.error("缓存中没有找到匹配商品ID{}与场次ID{}的库存数据！", goodsId, activityId);
        }
        return false;
    }

    private BigDecimal getActivityPriceNew(String goodsId, String organizationCode, String storeCode) {
        // 活动信息查询
        long activityMapsTime = System.currentTimeMillis();
        List<Map<String, Object>> activityMaps = activityGoodsRepos.findActivityPrice(goodsId, organizationCode);
        log.info("查询活动信息耗时 {} ms", System.currentTimeMillis() - activityMapsTime);

        // 获得商品对象
        long goodsTime = System.currentTimeMillis();
        Goods goods = goodsRepos.findOne(goodsId);
        log.info("查询 goods 耗时 {} ms", System.currentTimeMillis() - goodsTime);

        if (!Lang.isEmpty(activityMaps)) {

            // 活动价格信息
            Map<String, Object> activityMap = activityMaps.get(0);

            // 售价方案
            String keyForPriceType = "type";
            String priceType = (String) activityMap.get(keyForPriceType);

            // 不走费率直接使用协议价
            if (ActivityGoodsDto.ACTIVITY_PRICE_TYPE_WITHOUTRATE.equals(priceType)) {
                return goods.getCostPrice();
            } else if (ActivityGoodsDto.ACTIVITY_PRICE_TYPE_ACPRICE.equals(priceType)) {
                String keyForActivitySetPrice = "activityPrice";
                Object activityPrice = activityMap.get(keyForActivitySetPrice);
                if (!Lang.isEmpty(activityPrice)) {
                    return new BigDecimal(String.valueOf(activityPrice));
                }
                log.warn("使用活动配置价格方案失败 : 活动配置价格为空, 将使用默认费率价格方案");
            }
        }

        // 默认使用费率价格
        // 封装价格查询参数并剔除懒加载对象
        GoodsDto goodsDto = new GoodsDto();
        Copys.create().from(goods).excludes("product", "goodsAttrs", "goodsPictures", "goodsFashions", "goodsParams", "productFashion").to(goodsDto);

        //String organizationId = organizationApi.findOrganizationIdByOrgCode(organizationCode);

        // 查询默认使用的费率价格
        long salePriceTime = System.currentTimeMillis();
        SalesPriceQueryVo salesPriceQueryVo = new SalesPriceQueryVo(organizationCode, storeCode, goodsDto);
        List<Map> rateSalePriceMaps = salesPriceRateQueryApi.getSalesPriceListByParams(Collections.singletonList(salesPriceQueryVo));
        BigDecimal salePrice = (BigDecimal) rateSalePriceMaps.get(0).get("salePrice");
        //Map<String, BigDecimal> salePriceMap = salesPriceRateQueryApi.getSaleRatePriceBatchByGoodsList(Collections.singletonList(goodsDto), storeCode, organizationId);
        log.info("批量查询费率价格耗时 {} ms", System.currentTimeMillis() - salePriceTime);

        return salePrice;
    }


}
