package activities.service.service;

import activities.api.ActivityConfigurationApi;
import activities.dto.modeldto.ActivityGoodsDto;
import activities.model.ActivityGoods;
import activities.model.repository.ActivityGoodsRepos;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.ExampleMatcher;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import utils.Lang;
import utils.data.BeanMapper;
import utils.log.Log;
import utils.log.Logs;

import java.util.List;

@Component
@Transactional(readOnly = true)
public class ActivityGoodsService {

    private final static Log log = Logs.getLog(ActivityGoodsService.class.getName());

    @Autowired
    private RedisTemplate redisTemplate;

    @Autowired
    ActivityGoodsRepos activityGoodsRepos;

    @Autowired
    ActivityConfigurationApi activityConfigurationApi;

    /**
     * 加载所有活动商品配置库存至缓存中
     */
    public void loadStockToRedis(String organizationCode, String activityCode) {
        int success = 0;
        log.info("活动信息 organizationCode : {}, activityCode: {}", organizationCode, activityCode);
        List<ActivityGoods> activityGoodsList =  activityGoodsRepos.findActivityGoods(organizationCode, activityCode);
        log.info("对应活动商品数量：{}", activityGoodsList.size());
        for (ActivityGoods ag : activityGoodsList) {
            //goodsId 以及 activityId 作为key
            if(!Lang.isEmpty(ag.getStockNum())) {
                String key = activityConfigurationApi.buildKeyForActivityStock(organizationCode, ag.getGoodsId(), ag.getActivityDefinition().getId(), activityCode);
                activityConfigurationApi.setActivityKey(organizationCode, key);
                BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(key);
                log.info("===初始化活动商品库存key={}，stockNum={}", key, ag.getStockNum());
                boundValueOperations.set(ag.getStockNum());
                success ++;
            } else {
                log.error("缓存活动商品配置库存失败，配置库存为空，请检查 activity_goods 表主键为 {} 的数据", ag.getId());
            }
        }
        log.info("成功缓存活动商品数量：{}", success);
    }

    /**
     * 根据样例数据获取活动商品列表
     *
     * @param activityGoodsDto ActivityGoodsDto
     * @return List<ActivityGoodsDto>
     */
    public List<ActivityGoodsDto> findByExample(ActivityGoodsDto activityGoodsDto) {
        ActivityGoods activityGoods = BeanMapper.map(activityGoodsDto, ActivityGoods.class);
        List<ActivityGoods> activityDefinitions = activityGoodsRepos.findAll(createExample(activityGoods));
        return BeanMapper.mapList(activityDefinitions, ActivityGoods.class, ActivityGoodsDto.class);
    }

    private Example<ActivityGoods> createExample(ActivityGoods entity) {
        ExampleMatcher matcher = ExampleMatcher.matching().withIgnoreNullValues();
        return Example.of(entity, matcher);
    }
}
