package activities.web.controller;

import activities.api.ActivityGoodsApi;
import activities.vo.ActivityGoodsVo;
import activities.vo.ActivityInfoVO;
import activities.web.req.ct.score.activity.MakeOrderReq;
import activities.web.service.ActivitiesService;
import activities.web.service.ActivityDefinitionService;
import cart.api.OrderMainApi;
import cart.api.vo.FashionDetailVo;
import cart.api.vo.TempOrderStoreVo;
import cart.api.vo.TempOrderVo;
import com.alibaba.fastjson.JSON;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.GoodsConfigApi;
import lombok.extern.slf4j.Slf4j;
import member.api.dto.core.CoreUserDto;
import member.api.dto.shop.MemberAddressDto;
import member.api.vo.MemberVo;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import sinomall.global.common.response.BaseResponse;
import utils.GlobalContants;
import utils.Lang;
import utils.data.BeanMapper;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author oracle
 * @date 2017-05-15
 */
@Slf4j
@Controller
public class CtScoreActivityController {

    @MotanReferer
    private OrderMainApi orderMainApi;

    @MotanReferer
    private ActivityGoodsApi activityGoodsApi;

    private final RedisTemplate redisTemplate;

    private final ActivityDefinitionService activityDefinitionService;

    private final RedissonClient redisClient;

    private final ActivitiesService activitiesService;

    /**
     * 合作企业代码
     */
    @Value("${organization.code}")
    private String organizationCode;

    /**
     * 活动编码
     */
    @Value("${ct-score.activity.code}")
    private String activityCode;

    /**
     * 用户活动ID
     */
    private static final String MEMBER_ACTIVITY_ID = "MEMBER_ACTIVITY_ID";

    /**
     * 下单请求锁前缀
     */
    private static final String MAKE_ORDER_LOCK_REQUEST_KEY_PREFIX = "CT-SCORE-ACTIVITY:MAKE-ORDER:REQUEST-ID:";

    /**
     * 下单用户锁前缀
     */
    private static final String MAKE_ORDER_LOCK_USER_KEY_PREFIX = "CT-SCORE-ACTIVITY:MAKE-ORDER:MEMBER-ID:";

    /**
     * MEMBER-VO SESSION KEY
     */
    private static final String MEMBER_VO_SESSION_KEY = "member_vo_json_itaiping";

    /**
     * 活动对象缓存KEY
     */
    private static final String ACTIVITY_INFO_CACHE_KEY = "ACTIVITY-WEB:CT-SCORE:ACTIVITY-INFO";

    /**
     * 下单规格缓存 KEY
     */
    private static final String FASHION_DETAIL_VO_LIST_BY_GOODS_ID_CACHE_KEY_PREFIX = "ACTIVITY-WEB:CT-SCORE:FASHION-DETAIL-VOS:GOODS-ID:";

    /**
     * 清除活动缓存 KEY
     */
    private static final String CLEAR_ACTIVITY_CACHE_KEY = "clearCache";

    @Autowired
    public CtScoreActivityController(RedisTemplate redisTemplate, ActivityDefinitionService activityDefinitionService, RedissonClient redisClient, ActivitiesService activitiesService) {
        this.redisTemplate = redisTemplate;
        this.activityDefinitionService = activityDefinitionService;
        this.redisClient = redisClient;
        this.activitiesService = activitiesService;
    }

    @RequestMapping("control.html")
    public String control(Map<String, Object> map, HttpServletRequest request) {
        long methodStartTime = System.currentTimeMillis();
        log.info("### 诚通领取活动加载页面开始...");

        if (!Lang.isEmpty(request.getParameter(CLEAR_ACTIVITY_CACHE_KEY))) {
            redisTemplate.delete(ACTIVITY_INFO_CACHE_KEY);
            log.info("活动缓存清除成功");
        }

        try {
            // 从session拿去数据避免数据库io
            MemberVo memberVo = JSON.parseObject((String) request.getSession().getAttribute(MEMBER_VO_SESSION_KEY), MemberVo.class);
            if (Lang.isEmpty(memberVo)) {
                log.info("未登录跳转诚通登录页面");
                return "redirect:../score";
            }

            BoundValueOperations activityInfoCache = redisTemplate.boundValueOps(ACTIVITY_INFO_CACHE_KEY);
            ActivityInfoVO activityInfoVO = (ActivityInfoVO) activityInfoCache.get();
            if (Lang.isEmpty(activityInfoVO)) {
                activityInfoVO = activityDefinitionService.getActivityData(organizationCode, activityCode);
                if (!Lang.isEmpty(activityInfoVO)) {
                    activityInfoCache.set(activityInfoVO, 1, TimeUnit.DAYS);
                }
            }

            Date currentTime = new Date();
            if (Lang.isEmpty(activityInfoVO)) {
                log.info("当前没有活动数据");
                return "ctScoreActivity/common";
            } else if (!Lang.isEmpty(activityInfoVO.getBeginTime()) && currentTime.before(activityInfoVO.getBeginTime())) {
                log.info("活动暂未开始");
                map.put("status", "HAS_NOT_STARTED");
                return "ctScoreActivity/common";
            } else if (!Lang.isEmpty(activityInfoVO.getEndTime()) && currentTime.after(activityInfoVO.getEndTime())) {
                log.info("活动已结束");
                return "ctScoreActivity/common";
            } else {
                BigDecimal orderCount = activityGoodsApi.getEffectiveOrderCount(memberVo.getMember().getId(), activityInfoVO.getActivityId());
                if (orderCount.intValue() > 0) {
                    log.info("用户已参与活动, memberId {}", memberVo.getMember().getId());
                    map.put("status", "SUCCESS");
                    return "ctScoreActivity/common";
                }
                request.getSession().setAttribute(MEMBER_ACTIVITY_ID, activityInfoVO.getActivityId());

                //进行中
                List<ActivityGoodsVo> listActivityGoodsVo = null;
                if (CoreUserDto.USER_TYPE_CT_SCORE_ACTIVITY.equals(memberVo.getUser().getUserType())) {
                    if (activityInfoVO.getActivityGoodsVOs().stream().anyMatch(activityGoodsVo -> activityGoodsVo.getGiftClass().equals(memberVo.getUser().getSource()))) {
                        listActivityGoodsVo = activityInfoVO.getActivityGoodsVOs().stream().filter(activityGoodsVo -> activityGoodsVo.getGiftClass().equals(memberVo.getUser().getSource())).sorted(Comparator.comparing(activities.vo.ActivityGoodsVo::getShowIndex)).collect(Collectors.toList());
                    }
                } else {
                    log.info("非法用户, userSource {}, memberId {}", memberVo.getUser().getSource(), memberVo.getMember().getId());
                    return "redirect:../score";
                }
                map.put("listActivityGoodsVo", listActivityGoodsVo);
                return "ctScoreActivity/goodss";
            }
        } catch (Exception e) {
            log.error("加载活动页面异常", e);
            return "ctScoreActivity/common";
        } finally {
            log.info("### 诚通领取活动加载总耗时 {} ms", System.currentTimeMillis() - methodStartTime);
        }
    }

    @RequestMapping("success.html")
    public String success(Map<String, Object> map) {
        map.put("status", "SUCCESS");
        return "ctScoreActivity/common";
    }

    @RequestMapping("end.html")
    public String end() {
        return "ctScoreActivity/common";
    }

    @RequestMapping("goods.html")
    public String goods() {
        return "duanwu/goodss";
    }

    @RequestMapping("noBegin.html")
    public String noBegin(Map<String, Object> map) {
        map.put("status", "HAS_NOT_STARTED");
        return "ctScoreActivity/common";
    }

    @ResponseBody
    @RequestMapping("/ct-score-activity/order")
    public BaseResponse makeOrder(@RequestBody MakeOrderReq makeOrderReq, HttpServletRequest request) {
        log.info("诚通礼包领取活动请求下单 参数 {}", JSON.toJSONString(makeOrderReq));
        // 从缓存获取活动数据, 避免数据库io
        ActivityInfoVO activityInfoVO = (ActivityInfoVO) redisTemplate.boundValueOps(ACTIVITY_INFO_CACHE_KEY).get();
        // 从session拿去数据, 避免数据库io
        MemberVo memberVo = JSON.parseObject((String) request.getSession().getAttribute(MEMBER_VO_SESSION_KEY), MemberVo.class);
        if (Lang.isEmpty(makeOrderReq) || Lang.isEmpty(makeOrderReq.getRequestId())) {
            log.error("请求id不能为空");
            return new BaseResponse(GlobalContants.ResponseStatus.ERROR, "请求参数 [requestId] 请求id 不能为空");
        } else if (Lang.isEmpty(makeOrderReq.getGoodsId())) {
            log.error("未选择商品");
            return new BaseResponse(GlobalContants.ResponseStatus.ERROR, "请求参数 [goodsId] 商品id 不能为空");
        } else if (Lang.isEmpty(activityInfoVO)) {
            log.error("无法获取活动数据");
            return new BaseResponse("info", "活动数据获取失败, 请稍后再试");
        } else if (Lang.isEmpty(memberVo)) {
            log.info("从 session 获取 memberVo 失败");
            return new BaseResponse("info", "登录失效, 请重新登录");
        } else if(!CoreUserDto.USER_TYPE_CT_SCORE_ACTIVITY.equals(memberVo.getUser().getUserType())) {
            log.error("无效的用户, memberID {}", memberVo.getMember().getId());
            return new BaseResponse("info", "您没有资格参与本次活动");
        }
        // 检查 goodsId 是否为活动, 用户组限定内
        if (activityInfoVO.getActivityGoodsVOs().stream().noneMatch(activityGoodsVo -> activityGoodsVo.getGoodsId().equals(makeOrderReq.getGoodsId()) && activityGoodsVo.getGiftClass().equals(memberVo.getUser().getSource()))) {
            log.warn("非法的 goodsId {}, memberId {}", makeOrderReq.getGoodsId(), memberVo.getMember().getId());
            return new BaseResponse("info", "您选择的商品无效, 请更换商品后");
        }
        String requestLockKey = MAKE_ORDER_LOCK_REQUEST_KEY_PREFIX + memberVo.getMember().getId() + "_" + makeOrderReq.getRequestId();
        BoundValueOperations requestLocker = redisTemplate.boundValueOps(requestLockKey);
        if (requestLocker.setIfAbsent(true)) {
            requestLocker.expire(30L, TimeUnit.MINUTES);
            RLock userLocker = null;
            long orderTime = System.currentTimeMillis();
            try {
                String userLockKey = MAKE_ORDER_LOCK_USER_KEY_PREFIX + memberVo.getMember().getId();
                userLocker = redisClient.getLock(userLockKey);
                userLocker.lock(10L, TimeUnit.MINUTES);
                if (userLocker.isLocked()) {
                    BigDecimal orderCount = activityGoodsApi.getEffectiveOrderCount(memberVo.getMember().getId(), activityInfoVO.getActivityId());
                    if (orderCount.intValue() > 0) {
                        log.info("用户已参与活动, memberId {}", memberVo.getMember().getId());
                        return new BaseResponse("info", "您已参与活动");
                    }
                    BoundValueOperations fashionDetailVoListCache = redisTemplate.boundValueOps(FASHION_DETAIL_VO_LIST_BY_GOODS_ID_CACHE_KEY_PREFIX + makeOrderReq.getGoodsId());
                    List<FashionDetailVo> fashionDetailVoList = (List<FashionDetailVo>) fashionDetailVoListCache.get();
                    if(Lang.isEmpty(fashionDetailVoList)) {
                        fashionDetailVoList = activitiesService.getFashionDetailVoListByGoodsId(makeOrderReq.getGoodsId());
                        if(!Lang.isEmpty(fashionDetailVoList)) {
                            fashionDetailVoListCache.set(fashionDetailVoList, 10, TimeUnit.MINUTES);
                        }
                    }
                    if (Lang.isEmpty(fashionDetailVoList)) {
                        log.error("goodsId {} 无法找到 ProductFashion 数据", makeOrderReq.getGoodsId());
                        return new BaseResponse("info", "该商品无法购买, 请选择其他商品");
                    }
                    MemberAddressDto address = BeanMapper.map(makeOrderReq, MemberAddressDto.class);
                    address.setMember(memberVo.getMember());
                    String minAddressCode = Lang.isEmpty(address.getTownCode()) ? address.getAreaCode() : address.getTownCode();
                    if (Lang.isEmpty(minAddressCode)) {
                        log.error("地址数据有误, MemberAddress {}", JSON.toJSONString(address));
                        return new BaseResponse("info", "您选择的地址暂不可用, 请更换其他地址");
                    }
                    long createTempOrderTime = System.currentTimeMillis();
                    log.info("正在创建临时订单, memberId {}", memberVo.getMember().getId());
                    TempOrderVo tempOrder = orderMainApi.newTempOrderByFasions(fashionDetailVoList, organizationCode, minAddressCode);
                    tempOrder.setBuyType(request);
                    log.info("创建临时订单耗时 {} ms, memberId {}, 结果 {}", System.currentTimeMillis() - createTempOrderTime, memberVo.getMember().getId(), JSON.toJSONString(tempOrder));
                    tempOrder.getStoreList().get(0).getOrderItems().get(0).setActivityId(activityInfoVO.getActivityId());
                    return activitiesService.makeOrder(tempOrder, address, memberVo, fashionDetailVoList, minAddressCode);
                } else {
                    userLocker = null;
                    log.error("用户锁获取失败, memberId {}", memberVo.getMember().getId());
                    return new BaseResponse("info", "您的请求正在处理, 请勿重复提交");
                }
            } catch (Exception e) {
                log.error("诚通活动商品免费领取下单异常, memberId {}", memberVo.getMember().getId(), e);
                return new BaseResponse(GlobalContants.ResponseStatus.ERROR, "请联系诚意通客服专线 400-820-2095");
            } finally {
                log.info("用户下单耗时 {} ms, memberId {}", System.currentTimeMillis() - orderTime, memberVo.getMember().getId());
                if (!Lang.isEmpty(userLocker)) {
                    userLocker.unlock();
                }
            }
        } else {
            log.error("用户重复请求拦截, memberId {}", memberVo.getMember().getId());
            return new BaseResponse("info", "您的请求正在处理, 请勿重复提交");
        }
    }

}
