package activities.web.service;

import activities.api.ActivityDefinitionApi;
import activities.dto.modeldto.ActivityDefinitionDto;
import activities.dto.modeldto.ActivityGoodsDto;
import activities.vo.ActivityGoodsVo;
import activities.vo.ActivityInfoVO;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.GoodsConfigApi;
import goods.model.Goods;
import goods.model.GoodsPicture;
import goods.model.repository.GoodsPictureRepos;
import goods.model.repository.GoodsRepos;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import utils.Lang;
import utils.data.BeanMapper;

import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by Admin on 2017/5/17.
 */
@Slf4j
@Service
public class ActivityDefinitionService {

    @MotanReferer
    private GoodsConfigApi goodsConfigApi;

    @MotanReferer
    private ActivityDefinitionApi activityDefinitionApi;

    private final GoodsRepos goodsRepos;

    private final GoodsPictureRepos goodsPictureRepos;

    @Autowired
    public ActivityDefinitionService(GoodsRepos goodsRepos, GoodsPictureRepos goodsPictureRepos) {
        this.goodsRepos = goodsRepos;
        this.goodsPictureRepos = goodsPictureRepos;
    }

    /**
     * 获取活动相关数据
     *
     * @param organizationCode 机构代码
     * @param activityCode     活动代码
     * @return ActivityInfoVO
     */
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public ActivityInfoVO getActivityData(String organizationCode, String activityCode) {
        List<ActivityDefinitionDto> activityDefinitions = activityDefinitionApi.findByOrganizationCodeAndActivityCode(organizationCode, activityCode);
        if (Lang.isEmpty(activityDefinitions)) {
            return null;
        } else if (activityDefinitions.size() > 1) {
            log.error("在线的活动数据不是唯一");
            return null;
        }
        ActivityDefinitionDto activityDefinition = activityDefinitions.get(0);
        List<ActivityGoodsVo> activityGoodsVos = null;
        if(Lang.isEmpty(activityDefinition.getEndTime()) || new Date().before(activityDefinition.getEndTime())) {
            List<ActivityGoodsDto> activityGoodsList = activityDefinition.getActivityGoodses().stream().filter(activityGoods -> !activityGoods.getIsDelete() && activityGoods.getIsOnline() && goodsConfigApi.isUpGoodsId(activityGoods.getGoodsId(), organizationCode)).collect(Collectors.toList());
            activityGoodsVos = BeanMapper.mapList(activityGoodsList, ActivityGoodsDto.class, ActivityGoodsVo.class);
            List<String> goodsIds = activityGoodsVos.stream().map(ActivityGoodsVo::getGoodsId).collect(Collectors.toList());

            long batchQueryGoodsTime = System.currentTimeMillis();
            List<Goods> goodsListNew = new Vector<>();
            goodsIds.parallelStream().forEach(goodsId -> {
                Goods goodsTemp = goodsRepos.findOne(goodsId);
                goodsTemp.setGoodsPictures(Collections.singletonList(goodsPictureRepos.findMinIndexPictrueByGoodsId(goodsId)));
                goodsListNew.add(goodsTemp);
            });
            log.info("批量查询 goods 耗时 {} ms", System.currentTimeMillis() - batchQueryGoodsTime);

            Map<String, Goods> goodsIdMap = goodsListNew.stream().collect(Collectors.toMap(Goods::getId, goods -> goods));

            activityGoodsVos.parallelStream().forEach(activityGoodsVo -> {
                Goods goodsPO = goodsIdMap.get(activityGoodsVo.getGoodsId());
                activityGoodsVo.setGoodsName(goodsPO.getName());
                if (!Lang.isEmpty(goodsPO.getGoodsPictures())) {
                    GoodsPicture firstPic = goodsPO.getGoodsPictures().stream().sorted(Comparator.comparing(GoodsPicture::getShowIndex)).collect(Collectors.toList()).get(0);
                    activityGoodsVo.setGoodsPicture(firstPic.getPicturePath());
                }
            });
            log.info("加载 {} 条 活动商品数据", activityGoodsVos.size());
        }

        ActivityInfoVO activityInfoVO = new ActivityInfoVO();
        activityInfoVO.setActivityId(activityDefinition.getId());
        activityInfoVO.setBeginTime(activityDefinition.getBeginTime());
        activityInfoVO.setEndTime(activityDefinition.getEndTime());
        activityInfoVO.setActivityGoodsVOs(activityGoodsVos);
        return activityInfoVO;
    }
}
