package activities.web.service.Impl;

import activities.api.ActivityGoodsApi;
import activities.vo.ActivityInfoVO;
import activities.web.req.ct.score.activity.TestMakeOrderReq;
import activities.web.service.ActivitiesService;
import activities.web.service.ActivityDefinitionService;
import activities.web.service.TestCtScoreActivityService;
import cart.api.OrderMainApi;
import cart.api.vo.FashionDetailVo;
import cart.api.vo.TempOrderVo;
import com.alibaba.fastjson.JSON;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import lombok.extern.slf4j.Slf4j;
import member.api.MemberApi;
import member.api.dto.core.CoreUserDto;
import member.api.dto.shop.MemberAddressDto;
import member.api.vo.MemberVo;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import sinomall.global.common.response.BaseResponse;
import utils.GlobalContants;
import utils.Lang;

import java.math.BigDecimal;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 测试诚通活动服务接口实现
 *
 * @author FatAss
 * @date 2018-08-11
 */
@Slf4j
@Service
public class TestCtScoreActivityServiceImpl implements TestCtScoreActivityService {

    /**
     * 合作企业代码
     */
    @Value("${organization.code}")
    private String organizationCode;

    /**
     * 活动编码
     */
    @Value("${ct-score.activity.code}")
    private String activityCode;

    @MotanReferer
    private OrderMainApi orderMainApi;

    @MotanReferer
    private MemberApi memberApi;

    @MotanReferer
    private ActivityGoodsApi activityGoodsApi;

    private final ActivitiesService activitiesService;

    private final RedisTemplate redisTemplate;

    private final RedissonClient redisClient;

    private final ActivityDefinitionService activityDefinitionService;

    @Autowired
    public TestCtScoreActivityServiceImpl(ActivitiesService activitiesService, RedisTemplate redisTemplate, RedissonClient redisClient, ActivityDefinitionService activityDefinitionService) {
        this.activitiesService = activitiesService;
        this.redisTemplate = redisTemplate;
        this.redisClient = redisClient;
        this.activityDefinitionService = activityDefinitionService;
    }

    /**
     * 活动对象缓存KEY
     */
    private static final String ACTIVITY_INFO_CACHE_KEY = "ACTIVITY-WEB:CT-SCORE:ACTIVITY-INFO";

    /**
     * 下单规格缓存 KEY
     */
    private static final String FASHION_DETAIL_VO_LIST_BY_GOODS_ID_CACHE_KEY_PREFIX = "ACTIVITY-WEB:CT-SCORE:FASHION-DETAIL-VOS:GOODS-ID:";

    /**
     * 测试下单请求锁前缀
     */
    private static final String TEST_MAKE_ORDER_LOCK_REQUEST_KEY_PREFIX = "CT-SCORE-ACTIVITY:TEST:MAKE-ORDER:REQUEST-ID:";

    /**
     * 测试下单用户锁前缀
     */
    private static final String TEST_MAKE_ORDER_LOCK_USER_KEY_PREFIX = "CT-SCORE-ACTIVITY:TEST:MAKE-ORDER:MEMBER-ID:";

    @Override
    public BaseResponse<String> testMakeOrder(TestMakeOrderReq testMakeOrderReq) {
        BoundValueOperations activityInfoCache = redisTemplate.boundValueOps(ACTIVITY_INFO_CACHE_KEY);
        ActivityInfoVO activityInfoVO = (ActivityInfoVO) activityInfoCache.get();
        if (Lang.isEmpty(activityInfoVO)) {
            activityInfoVO = activityDefinitionService.getActivityData(organizationCode, activityCode);
            if (!Lang.isEmpty(activityInfoVO)) {
                activityInfoCache.set(activityInfoVO, 1, TimeUnit.DAYS);
            }
        }
        if(Lang.isEmpty(testMakeOrderReq) || Lang.isEmpty(testMakeOrderReq.getRequestId())) {
            return new BaseResponse<>(GlobalContants.ResponseStatus.ERROR, "请求参数 [requestId] - 请求 id 不能为空");
        } else if(Lang.isEmpty(testMakeOrderReq.getGoodsId())) {
            return new BaseResponse<>(GlobalContants.ResponseStatus.ERROR, "请求参数 [goodsId] - 用户 goodsId 不能为空");
        } else if(Lang.isEmpty(testMakeOrderReq.getMemberId())) {
            return new BaseResponse<>(GlobalContants.ResponseStatus.ERROR, "请求参数 [memberId] - 用户 memberId 不能为空");
        }
        long memberVoTime = System.currentTimeMillis();
        MemberVo memberVo = memberApi.getMemberInfo(testMakeOrderReq.getMemberId());
        log.info("查询 memberVO 耗时 {} ms, memberId {}", System.currentTimeMillis() - memberVoTime,testMakeOrderReq.getMemberId());

        if(Lang.isEmpty(memberVo) || !CoreUserDto.USER_TYPE_CT_SCORE_ACTIVITY.equals(memberVo.getUser().getUserType())) {
            log.error("无效的 memberId {}", testMakeOrderReq.getMemberId());
            return new BaseResponse<>("info", "您没有资格参与本次活动");
        }
        if (activityInfoVO.getActivityGoodsVOs().stream().noneMatch(activityGoodsVo -> activityGoodsVo.getGoodsId().equals(testMakeOrderReq.getGoodsId()) && activityGoodsVo.getGiftClass().equals(memberVo.getUser().getSource()))) {
            log.warn("非法的 goodsId {}, memberId {}", testMakeOrderReq.getGoodsId(), testMakeOrderReq.getMemberId());
            return new BaseResponse<>("info", "您选择的商品无效, 请更换商品后");
        }
        String requestLockKey = TEST_MAKE_ORDER_LOCK_REQUEST_KEY_PREFIX + testMakeOrderReq.getMemberId() + "_" + testMakeOrderReq.getRequestId();
        BoundValueOperations requestLocker = redisTemplate.boundValueOps(requestLockKey);
        if (requestLocker.setIfAbsent(true)) {
            requestLocker.expire(30L, TimeUnit.MINUTES);
            RLock userLocker = null;
            long orderTime = System.currentTimeMillis();
            try {
                long lockMemberTime = System.currentTimeMillis();
                String userLockKey = TEST_MAKE_ORDER_LOCK_USER_KEY_PREFIX + testMakeOrderReq.getMemberId();
                userLocker = redisClient.getLock(userLockKey);
                userLocker.lock(3L, TimeUnit.MINUTES);
                if (userLocker.isLocked()) {
                log.info("锁用户 memberId {}, 耗时 {} ms", memberVo.getMember().getId(),System.currentTimeMillis() - lockMemberTime);
                    long getEffectiveOrderCountTime = System.currentTimeMillis();
                    BigDecimal orderCount = activityGoodsApi.getEffectiveOrderCount(testMakeOrderReq.getMemberId(), activityInfoVO.getActivityId());
                    log.info("查询用户有效订单耗时 {} ms, memberId {}", System.currentTimeMillis() - getEffectiveOrderCountTime, memberVo.getMember().getId());
                    if (orderCount.intValue() > 0) {
                        log.info("用户已参与活动");
                        return new BaseResponse<>("info", "您已参与活动");
                    }

                    long getFashionDetailVoListTime = System.currentTimeMillis();
                    BoundValueOperations fashionDetailVoListCache = redisTemplate.boundValueOps(FASHION_DETAIL_VO_LIST_BY_GOODS_ID_CACHE_KEY_PREFIX + testMakeOrderReq.getGoodsId());
                    List<FashionDetailVo> fashionDetailVoList = (List<FashionDetailVo>) fashionDetailVoListCache.get();
                    if(Lang.isEmpty(fashionDetailVoList)) {
                        fashionDetailVoList = activitiesService.getFashionDetailVoListByGoodsId(testMakeOrderReq.getGoodsId());
                        if(!Lang.isEmpty(fashionDetailVoList)) {
                            fashionDetailVoListCache.set(fashionDetailVoList, 10, TimeUnit.MINUTES);
                        }
                    }
                    log.info("获取下单规格耗时 {} ms", System.currentTimeMillis() - getFashionDetailVoListTime);
                    if (Lang.isEmpty(fashionDetailVoList)) {
                        log.error("下单失败 : goodsId {} 无法找到 ProductFashion 数据", testMakeOrderReq.getGoodsId());
                        return new BaseResponse<>("info", "该商品无法购买, 请选择其他商品");
                    }
                    MemberAddressDto address = new MemberAddressDto();
                    address.setMember(memberVo.getMember());
                    address.setUsername("测试人员");
                    address.setPhone("13888888888");
                    address.setDetailaddress("测试地址请勿发货");
                    address.setAddressDetail("测试地址请勿发货");
                    address.setProvinceCode("110000");
                    address.setProvinceName("北京");
                    address.setCityCode("110100");
                    address.setCityName("北京");
                    address.setAreaCode("110101");
                    address.setAreaName("东城区");
                    address.setTownCode("d8a15c70-9c8a-44b6-bc34-06b457ecfaaf");
                    address.setTownName("内环到三环里");
                    address.setEmail("test@test.com");
                    long createTempOrderTime = System.currentTimeMillis();
                    log.info("正在创建临时订单, memberId {}", testMakeOrderReq.getMemberId());
                    TempOrderVo tempOrder = orderMainApi.newTempOrderByFasions(fashionDetailVoList, organizationCode, address.getAreaCode());
                    log.info("创建临时订单耗时 {} ms, memberId {}, 结果 {}", System.currentTimeMillis() - createTempOrderTime, testMakeOrderReq.getMemberId(), JSON.toJSONString(tempOrder));
                    tempOrder.getStoreList().get(0).getOrderItems().get(0).setActivityId(activityInfoVO.getActivityId());
                    return activitiesService.makeOrder(tempOrder, address, memberVo, fashionDetailVoList, address.getTownCode());
                } else {
                    log.error("用户锁获取失败, memberId {}", testMakeOrderReq.getMemberId());
                    return new BaseResponse<>("info", "您的请求正在处理, 请勿重复提交");
                }
            } catch (Exception e) {
                log.error("下单异常", e);
                return new BaseResponse<>(GlobalContants.ResponseStatus.ERROR, "请联系管理员或稍后重试");
            } finally {
                log.info("用户下单耗时 {} ms, memberId {}", System.currentTimeMillis() - orderTime, testMakeOrderReq.getMemberId());
                if (!Lang.isEmpty(userLocker)) {
                    userLocker.unlock();
                }
            }
        } else {
            log.error("用户重复请求拦截, memberId {}", testMakeOrderReq.getMemberId());
            return new BaseResponse<>("info", "您的请求正在处理, 请勿重复提交");
        }
    }

}
