package bidding.service.service.impl;

import bidding.model.po.*;
import bidding.model.repository.*;
import bidding.model.specification.*;
import bidding.service.service.BiddingConformityReviewService;
import bidding.service.service.BiddingConformityReviewTransactionalService;
import com.alibaba.fastjson.JSON;
import definition.BiddingConformAuditStatus;
import definition.BiddingProviderAddRespStatus;
import definition.BiddingStatus;
import definition.ProviderConformanceRespStatus;
import dto.*;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import query.BiddingConformityReviewAuditListQuery;
import sinomall.global.common.response.BaseResponse;
import utils.Lang;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 竞价符合性审查业务逻辑层
 *
 * @author Drury
 * @date 2018/3/19
 */
@Slf4j
@Service
public class BiddingConformityReviewServiceImpl implements BiddingConformityReviewService {
    @Autowired
    JcbdConformanCheckRepo jcbdConformanCheckRepo;

    /**
     * 分页获取符合性审查列表
     *
     * @param biddingConformityReviewAuditListQuery BiddingConformityReviewAuditListQuery
     * @return BaseResponse<Page<BiddingConformReviewAuditListDTO>>
     */
    @Override
    public BaseResponse<Page<BiddingConformReviewAuditListDTO>> getBiddingConformityReviewAuditList(BiddingConformityReviewAuditListQuery biddingConformityReviewAuditListQuery) {
        BaseResponse<Page<BiddingConformReviewAuditListDTO>> baseResponse;

        log.info("获取符合性审查列表-请求参数 {}", JSON.toJSONString(biddingConformityReviewAuditListQuery));

        // 请求参数校验
        if (Lang.isEmpty(biddingConformityReviewAuditListQuery)) {
            baseResponse = new BaseResponse<>("获取符合性审查列表失败-请求参数不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(biddingConformityReviewAuditListQuery.getAuditorId())) {
            baseResponse = new BaseResponse<>("获取符合性审查列表失败-审核人ID不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(biddingConformityReviewAuditListQuery.getBiddingCode())) {
            baseResponse = new BaseResponse<>("获取符合性审查列表失败-竞价编号不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(biddingConformityReviewAuditListQuery.getPageable())) {
            baseResponse = new BaseResponse<>("获取符合性审查列表失败-分页信息不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }

        try {
            // 根据 竞价编号 得到 竞价对象
            JcbdBidding bidding;
            BaseResponse<JcbdBidding> biddingResponse = findAndCheckBidding(biddingConformityReviewAuditListQuery.getBiddingCode());
            if (biddingResponse.isSuccess()) {
                bidding = biddingResponse.getResult();
            } else {
                return new BaseResponse<>(biddingResponse.getResultMessage());
            }

            JcbdBiddingDemand biddingDemand = new JcbdBiddingDemand();
            biddingDemand.setJcbdBidding(bidding);
            Page<JcbdBiddingDemand> biddingDemandPage = biddingDemandRepo.findAll(new BiddingDemandSpecification<>(biddingDemand), biddingConformityReviewAuditListQuery.getPageable());

            // PO 转 DTO
            List<JcbdBiddingDemand> biddingDemands = biddingDemandPage.getContent();
            List<BiddingConformReviewAuditListDTO> biddingConformReviewAuditList = mapperFacade.mapAsList(biddingDemands, BiddingConformReviewAuditListDTO.class);

            // 其他数据封装
            for (int i = 0; i < biddingDemands.size(); i++) {
                // 查询竞价需求符合性审查数量
                JcbdBiddingConformance biddingConformance = new JcbdBiddingConformance();
                biddingConformance.setJcbdBiddingDemand(biddingDemands.get(i));
                Long checkItemCount = biddingConformanceRepo.count(new BiddingConformanceSpecification<>(biddingConformance));
                biddingConformReviewAuditList.get(i).setCheckItemCount(checkItemCount.intValue());

                // 查询竞价需求供应商响应数量
                JcbdBiddingDemandResponse biddingDemandResponse = new JcbdBiddingDemandResponse();
                biddingDemandResponse.setJcbdBiddingDemand(biddingDemands.get(i));
                Long demandResponseCount = biddingDemandRespRepo.count(new BiddingDemandResponseSpecification<>(biddingDemandResponse));
                biddingConformReviewAuditList.get(i).setDemandResponseCount(demandResponseCount.intValue());

                // 查询已审查供应商数
                // 当前竞价需求所有符合性检查项
                Map<String, Integer> map = new HashMap<>(demandResponseCount.intValue());
                List<JcbdBiddingConformance> biddingConformances = biddingConformanceRepo.findAll(new BiddingConformanceSpecification<>(biddingConformance));
                for (JcbdBiddingConformance biddingConformanceTemp : biddingConformances) {
                    // 查询 当前遍历符合性审查项下指定审核人的审核数据
                    JcbdConformanceCheck conformanceCheck = new JcbdConformanceCheck();
                    conformanceCheck.setBiddingConformance(biddingConformanceTemp);
                    conformanceCheck.setBiddingGroupMemberId(biddingConformityReviewAuditListQuery.getAuditorId());
                    List<JcbdConformanceCheck> conformanceChecks = biddingConformanCheckRepo.findAll(new BiddingConformanceCheckSpecification<>(conformanceCheck));
                    // 统计供应商数量
                    for (JcbdConformanceCheck conformanceCheckTemp : conformanceChecks) {
                        String key = conformanceCheckTemp.getProviderId();
                        if (map.containsKey(key)) {
                            map.put(key, map.get(key) + 1);
                        } else {
                            map.put(key, 1);
                        }
                    }
                }
                int auditSupplierCount = 0;
                // 供应商数量等于符合性检查项数量说明完成审核
                for (String key : map.keySet()) {
                    if (map.get(key) == biddingConformances.size()) {
                        auditSupplierCount++;
                    }
                }
                biddingConformReviewAuditList.get(i).setAuditSupplierCount(auditSupplierCount);
            }

            // 封装 PAGE
            Page<BiddingConformReviewAuditListDTO> biddingConformReviewAuditListDTOPage = new PageImpl<>(biddingConformReviewAuditList, biddingConformityReviewAuditListQuery.getPageable(), biddingConformReviewAuditList.size());

            log.info("获取符合性审查列表 DATA : {}", JSON.toJSONString(biddingConformReviewAuditListDTOPage));

            return new BaseResponse<>(true, "操作成功", biddingConformReviewAuditListDTOPage);
        } catch (Exception e) {
            log.error("获取符合性审查列表异常, 竞价编号 {} ", biddingConformityReviewAuditListQuery.getBiddingCode(), e);
            return new BaseResponse<>("获取符合性审查列表异常");
        }
    }

    /**
     * 获取竞价需求符合性审查数据
     *
     * @param demandCode 竞价需求编号
     * @param auditorId  审核人ID
     * @return BaseResponse<List<BiddingConformityReviewInfoDTO>>
     */
    @Override
    public BaseResponse<List<BiddingConformityReviewInfoDTO>> getBiddingConformityReviewInfo(String demandCode, String auditorId) {
        BaseResponse<List<BiddingConformityReviewInfoDTO>> baseResponse;
        log.info("获取竞价需求符合性审查数据-请求参数竞价需求编号 {}", demandCode);
        // 请求参数检查
        if (Lang.isEmpty(demandCode)) {
            baseResponse = new BaseResponse<>("获取竞价需求符合性审查数据失败-请求参数竞价需求编号不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }

        try {
            // 通过竞价需求编号找到
            JcbdBiddingDemand biddingDemand = new JcbdBiddingDemand();
            biddingDemand.setDemandCode(demandCode);
            List<JcbdBiddingDemand> biddingDemands = biddingDemandRepo.findAll(new BiddingDemandSpecification<>(biddingDemand));

            // 数据检查
            if (biddingDemands.size() != 1) {
                log.error("获取竞价需求符合性审查数据失败-竞价需求编号为 {} 的数据要求有且只有有 1 条, 实际 {} 条", biddingDemand, biddingDemands.size());
                return new BaseResponse<>("获取竞价需求符合性审查数据失败-数据检查未通过");
            }

            biddingDemand = biddingDemands.get(0);

            // 获得竞价需求下的地址需求
            JcbdDemandAddressNum biddingDemandAdd = new JcbdDemandAddressNum();
            biddingDemandAdd.setJcbdBiddingDemand(biddingDemand);
            List<JcbdDemandAddressNum> biddingDemandAdds = biddingDemandAddressRepo.findAll(new BiddingDemandAddressSpecification<>(biddingDemandAdd));

            // 竞价需求符合性审查项
            JcbdBiddingConformance biddingConformance = new JcbdBiddingConformance();
            biddingConformance.setJcbdBiddingDemand(biddingDemand);
            List<JcbdBiddingConformance> biddingConformances = biddingConformanceRepo.findAll(new BiddingConformanceSpecification<>(biddingConformance));

            // 根据 竞价ID 找到 响应供应商数据
            JcbdBidding bidding = new JcbdBidding();
            bidding.setId(biddingDemand.getJcbdBidding().getId());

            JcbdBdProvider biddingProvider = new JcbdBdProvider();
            biddingProvider.setJcbdBidding(bidding);

            List<JcbdBdProvider> biddingProviders = biddingProviderRepo.findAll(new BiddingProviderSpecification<>(biddingProvider));
            // 除去 未响应 以及 已放弃 状态
            biddingProviders = biddingProviders.stream().filter(provider -> !provider.getStatus().equals(BiddingStatus.SUPPLIER_RESPONSE_NOT.getValue())).collect(Collectors.toList());
            biddingProviders = biddingProviders.stream().filter(provider -> !provider.getStatus().equals(BiddingStatus.SUPPLIER_RESPONSE_GIVEUP.getValue())).collect(Collectors.toList());

            // 查询竞价需求供应商响应数据
            JcbdBiddingDemandResponse biddingDemandResponse = new JcbdBiddingDemandResponse();
            biddingDemandResponse.setJcbdBiddingDemand(biddingDemand);
            List<JcbdBiddingDemandResponse> biddingDemandResponses = biddingDemandRespRepo.findAll(new BiddingDemandResponseSpecification<>(biddingDemandResponse));

            // 遍历需求所属的竞价下拥有的供应商 封装页面数据
            List<BiddingConformityReviewInfoDTO> biddingConformityReviewInfos = new ArrayList<>();
            for (JcbdBdProvider biddingProviderTemp : biddingProviders) {
                // 封装供应商 TAB
                BiddingConformityReviewInfoDTO biddingConformityReviewInfo = new BiddingConformityReviewInfoDTO();
                biddingConformityReviewInfo.setSupplierId(biddingProviderTemp.getId());
                biddingConformityReviewInfo.setSupplierName(biddingProviderTemp.getProviderName());

                // 基本信息-竞价需求
                biddingConformityReviewInfo.setGoodsDescription(biddingDemand.getGoodsDesription());
                biddingConformityReviewInfo.setSpecification(biddingDemand.getSpecification());
                biddingConformityReviewInfo.setServiceRequirement(biddingDemand.getServiceRequirement());
                biddingConformityReviewInfo.setDateRequiredDelivery(biddingDemand.getDateRequiredDelivery());


                for (JcbdBiddingDemandResponse biddingDemandResponseTemp : biddingDemandResponses) {
                    // 匹配供应商
                    if (biddingDemandResponseTemp.getResponseProviderId().equals(biddingProviderTemp.getId())) {
                        // 基本信息-竞价响应
                        biddingConformityReviewInfo.setGoodsTitle(biddingDemandResponseTemp.getGoodsTitle());
                        biddingConformityReviewInfo.setGoodsParams(biddingDemandResponseTemp.getGoodsParams());
                        biddingConformityReviewInfo.setServerPromise(biddingDemandResponseTemp.getServicePromise());
                        biddingConformityReviewInfo.setSku(biddingDemandResponseTemp.getSku());
                        biddingConformityReviewInfo.setGoodsLink(biddingDemandResponseTemp.getGoodsLink());
                        // TODO 附件待陈圣融确认
                        /*biddingConformityReviewInfo.setAttachments();*/
                        biddingConformityReviewInfo.setPricePeriodOfValidity(biddingDemandResponseTemp.getQouteIndate());
                        biddingConformityReviewInfo.setSupplyPeriod(biddingDemandResponseTemp.getSupplyPeriod());
                        biddingConformityReviewInfo.setRemark(biddingDemandResponseTemp.getRemark());

                        // 获得当前供应商响应需求下地区数据反馈
                        JcbdProviderAddressResponse providerAddressResponse = new JcbdProviderAddressResponse();
                        providerAddressResponse.setJcbdBiddingDemandResponse(biddingDemandResponseTemp);
                        List<JcbdProviderAddressResponse> providerAddressResponses = providerAddressRespRepo.findAll(new BiddingProviderAddRespSpecification<>(providerAddressResponse));

                        // 竞价数量及地址
                        List<BiddingConformityReviewDemandAddDTO> demandAdds = new ArrayList<>();
                        for (JcbdProviderAddressResponse providerAddressResponseTemp : providerAddressResponses) {
                            for (JcbdDemandAddressNum biddingDemandAddress : biddingDemandAdds) {
                                if (providerAddressResponseTemp.getDemandAddressId().equals(biddingDemandAddress.getId())) {
                                    String address = biddingDemandAddress.getProvinceName();
                                    address += biddingDemandAddress.getCityName();
                                    address += biddingDemandAddress.getAreaName();
                                    address += biddingDemandAddress.getTownName();
                                    BiddingConformityReviewDemandAddDTO biddingConformityReviewDemandAdd = new BiddingConformityReviewDemandAddDTO();
                                    biddingConformityReviewDemandAdd.setAddress(address);
                                    biddingConformityReviewDemandAdd.setNumberNeed(biddingDemandAddress.getNumberNeed());
                                    biddingConformityReviewDemandAdd.setRemark(providerAddressResponseTemp.getRemark());
                                    biddingConformityReviewDemandAdd.setStatus(BiddingProviderAddRespStatus.bddingProviderAddRespStatusMap.get(providerAddressResponseTemp.getStatus()).getDescription());
                                    demandAdds.add(biddingConformityReviewDemandAdd);
                                }
                            }
                        }
                        biddingConformityReviewInfo.setDemandAdds(demandAdds);

                        // 符合性审查
                        List<BiddingConformityReviewDTO> conformityReviews = new ArrayList<>();
                        JcbdProviderConformanceResp providerConformanceResp = new JcbdProviderConformanceResp();
                        providerConformanceResp.setJcbdBiddingDemandResponse(biddingDemandResponseTemp);
                        List<JcbdProviderConformanceResp> providerConformanceResps = providerConformanceRespRepo.findAll(new BiddingProviderConformanceRespSpec<>(providerConformanceResp));
                        // 统计审核结果数量
                        int accordCount = 0;
                        int notAccordCount = 0;
                        for (JcbdProviderConformanceResp providerConformanceRespTemp : providerConformanceResps) {
                            for (JcbdBiddingConformance biddingConformanceTemp : biddingConformances) {
                                if (providerConformanceRespTemp.getBiddingConformanceId().equals(biddingConformanceTemp.getId())) {
                                    BiddingConformityReviewDTO biddingConformityReview = new BiddingConformityReviewDTO();
                                    biddingConformityReview.setConformance(biddingConformanceTemp.getConformance());
                                    biddingConformityReview.setStatus(ProviderConformanceRespStatus.providerConformanceRespStatusMap.get(providerConformanceRespTemp.getStatus()).getDescription());
                                    biddingConformityReview.setRemark(providerConformanceRespTemp.getRemark());

                                    // 查询审核人对该供应商响应的符合性审查项数据的审核状态
                                    JcbdConformanceCheck biddingConformanceCheck = new JcbdConformanceCheck();
                                    biddingConformanceCheck.setBiddingGroupMemberId(auditorId);
                                    biddingConformanceCheck.setJcbdProviderConformanceResp(providerConformanceRespTemp);
                                    biddingConformanceCheck.setProviderId(biddingDemandResponseTemp.getResponseProviderId());
                                    biddingConformanceCheck.setBiddingConformance(biddingConformanceTemp);
                                    List<JcbdConformanceCheck> biddingConformanceChecks = biddingConformanCheckRepo.findAll(new BiddingConformanceCheckSpecification<>(biddingConformanceCheck));
                                    if (biddingConformanceChecks.size() > 1) {
                                        log.error("获取竞价需求符合性审查数据失败-同一竞价小组成员同一符合性审查项供应商响应审核的数据要求最多只有 1 条, 实际 {} 条", biddingDemands.size());
                                        return new BaseResponse<>("获取竞价需求符合性审查数据失败-数据检查未通过");
                                    } else if (biddingConformanceChecks.size() == 1) {
                                        JcbdConformanceCheck conformanceCheck = biddingConformanceChecks.get(0);
                                        if (BiddingConformAuditStatus.ACCORD.equals(BiddingConformAuditStatus.biddingConformAuditStatusMap.get(conformanceCheck.getStatus()))) {
                                            // 符合
                                            accordCount++;
                                        } else if (BiddingConformAuditStatus.NOT_ACCORD.equals(BiddingConformAuditStatus.biddingConformAuditStatusMap.get(conformanceCheck.getStatus()))) {
                                            // 不符合
                                            notAccordCount++;
                                        }
                                        biddingConformityReview.setAuditStatus(conformanceCheck.getStatus());
                                    }
                                    conformityReviews.add(biddingConformityReview);
                                }
                            }
                        }
                        // 完成审核判断
                        if (accordCount == biddingConformances.size()) {
                            biddingConformityReviewInfo.setSupplierResponseAuditStatus(BiddingConformAuditStatus.ACCORD.getDescription());
                        } else if (notAccordCount == biddingConformances.size()) {
                            biddingConformityReviewInfo.setSupplierResponseAuditStatus(BiddingConformAuditStatus.NOT_ACCORD.getDescription());
                        }
                        biddingConformityReviewInfo.setConformityReviews(conformityReviews);
                    }
                }
                biddingConformityReviewInfos.add(biddingConformityReviewInfo);
            }
            return new BaseResponse<>(true, "操作成功", biddingConformityReviewInfos);
        } catch (Exception e) {
            log.error("获取竞价需求符合性审查数据异常, 竞价需求编号 {} ", demandCode, e);
            return new BaseResponse<>("获取竞价需求符合性审查数据异常");
        }
    }

    /**
     * 保存符合性审查审核数据
     *
     * @param conformityAudit BiddingConformityAuditDTO
     * @return BaseResponse
     */
    @Override
    public BaseResponse saveBiddingConformityReviewAuditData(BiddingConformityAuditDTO conformityAudit) {
        BaseResponse baseResponse;
        log.info("保存符合性审查审核数据-请求参数 {}", JSON.toJSONString(conformityAudit));
        // 请求参数检查
//        if (Lang.isEmpty(conformityAudit)) {
//            baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-请求参数不能为空");
//            log.error(baseResponse.getResultMessage());
//            return baseResponse;
//        } else if (Lang.isEmpty(conformityAudit.getDemandCode())) {
//            baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-竞价需求编号不能为空");
//            log.error(baseResponse.getResultMessage());
//            return baseResponse;
//        } else if (Lang.isEmpty(conformityAudit.getAuditorId())) {
//            baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核人(竞价小组成员)ID不能为空");
//            log.error(baseResponse.getResultMessage());
//            return baseResponse;
//        } else if (Lang.isEmpty(conformityAudit.getSuppliers())) {
//            baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核供应商不能为空");
//            log.error(baseResponse.getResultMessage());
//            return baseResponse;
//        } else {
//            for (BiddingConformAuditSupplierDTO biddingConformAuditSupplier : conformityAudit.getSuppliers()) {
//                if (Lang.isEmpty(biddingConformAuditSupplier.getSupplierId())) {
//                    baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核供应商ID不能为空");
//                    log.error(baseResponse.getResultMessage());
//                    return baseResponse;
//                }
//                if (Lang.isEmpty(biddingConformAuditSupplier.getAuditInfos())) {
//                    baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核供应商的审核数据不能为空");
//                    log.error(baseResponse.getResultMessage());
//                    return baseResponse;
//                }
//                for (BiddingConformAuditInfoDTO biddingConformAuditInfo : biddingConformAuditSupplier.getAuditInfos()) {
//                    if (Lang.isEmpty(biddingConformAuditInfo.getAuditStatus())) {
//                        baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核结果不能为空");
//                        log.error(baseResponse.getResultMessage());
//                        return baseResponse;
//                    }
//                    // 检查审核状态
//                    if (!BiddingConformAuditStatus.biddingConformAuditStatusMap.containsKey(biddingConformAuditInfo.getAuditStatus())) {
//                        baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-无效的审核状态");
//                        log.error("保存符合性审查审核数据失败-无效的审核状态 {}, 请参考状态枚举 {}", biddingConformAuditInfo.getAuditStatus(), JSON.toJSONString(BiddingConformAuditStatus.biddingConformAuditStatusMap));
//                        return baseResponse;
//                    }
//                    if (Lang.isEmpty(biddingConformAuditInfo.getComplianceId())) {
//                        baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核的符合性检查项ID不能为空");
//                        log.error(baseResponse.getResultMessage());
//                        return baseResponse;
//                    }
//                    if (Lang.isEmpty(biddingConformAuditInfo.getComplianceIdSupplierResponse())) {
//                        baseResponse = new BaseResponse<>("保存符合性审查审核数据失败-审核的符合性检查项供应商响应ID不能为空");
//                        log.error(baseResponse.getResultMessage());
//                        return baseResponse;
//                    }
//                }
//            }
//        }
        try {
            biddingConformityReviewTransactionalService.saveBiddingConformityReviewAuditData(conformityAudit);
            return new BaseResponse<>(true, "操作成功");
        } catch (Exception e) {
            log.error("获取竞价需求符合性审查数据异常, 请求参数 {}", JSON.toJSONString(conformityAudit), e);
            return new BaseResponse<>("保存符合性审查审核数据异常");
        }
    }

    /**
     * 获取竞价小组成员符合性审查审核是否完成
     *
     * @param biddingCode 竞价编号
     * @param auditorId   审核人 ID
     * @return BaseResponse<Boolean>
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse<Boolean> isBiddingConformityReviewAuditComplished(String biddingCode, String auditorId,String email) {
        BaseResponse<Boolean> baseResponse;
        log.info("获取竞价小组成员符合性审查审核是否完成-请求参数 : 竞价编号 {}, 审核人ID {},审核人邮箱", biddingCode, auditorId,email);
        // 请求参数检查
        if (Lang.isEmpty(biddingCode)) {
            baseResponse = new BaseResponse<>("获取竞价小组成员符合性审查审核是否完成失败-竞价编号不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(auditorId) && Lang.isEmpty(email)) {
            baseResponse = new BaseResponse<>("获取竞价小组成员符合性审查审核是否完成失败-审核人ID或者邮件不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }

        try {
            JcbdBidding bidding;
            BaseResponse<JcbdBidding> biddingResponse = findAndCheckBidding(biddingCode);
            if (biddingResponse.isSuccess()) {
                bidding = biddingResponse.getResult();
            } else {
                return new BaseResponse<>(biddingResponse.getResultMessage());
            }

            boolean auditComplished = isBiddingConformityReviewAuditComplished(bidding, auditorId,email);

            return new BaseResponse<>(true, "操作成功", auditComplished);
        } catch (Exception e) {
            log.info("获取竞价小组成员符合性审查审核是否完成异常-请求参数 : 竞价编号 {}, 审核人ID {},审核人邮箱 {}", biddingCode, auditorId,email, e);
            return new BaseResponse<>("获取竞价小组成员符合性审查审核是否完成异常");
        }
    }

    /**
     * 获取竞价小组成员符合性审查审核提交是否最后一个
     *
     * @param biddingCode 竞价编号
     * @return BaseResponse<Boolean>
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse<Boolean> isLastBiddingConformityReviewSubmit(String biddingCode) {
        BaseResponse<Boolean> baseResponse;
        log.info("获取竞价小组成员符合性审查审核提交是否最后一个-请求参数 竞价编号 : {}", biddingCode);
        // 请求参数检查
        if (Lang.isEmpty(biddingCode)) {
            baseResponse = new BaseResponse<>("取竞价小组成员符合性审查审核提交是否最后一个-竞价编号不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }
        try {
            // 根据 竞价编号 得到 竞价对象
            JcbdBidding bidding;
            BaseResponse<JcbdBidding> biddingResponse = findAndCheckBidding(biddingCode);
            if (biddingResponse.isSuccess()) {
                bidding = biddingResponse.getResult();
            } else {
                return new BaseResponse<>(biddingResponse.getResultMessage());
            }

            List<JcbdBiddingGroupMember> biddingGroupMembers = bidding.getJcbdBiddingGroupMembers().stream().filter(biddingGroupMember -> !biddingGroupMember.getIsDelete()).collect(Collectors.toList());

            int auditComplishedCount = 0;
            for (JcbdBiddingGroupMember biddingGroupMember : biddingGroupMembers) {
                boolean auditComplished = isBiddingConformityReviewAuditComplished(bidding, biddingGroupMember.getGroupMemberId(),biddingGroupMember.getEmail());
                if (auditComplished) {
                    auditComplishedCount++;
                }
            }

            log.info("全部数量 {}, 完成审核数量 {}", biddingGroupMembers.size(), auditComplishedCount);

            // 全部审核完成
            if (auditComplishedCount == biddingGroupMembers.size()) {
                return new BaseResponse<>(true, "操作成功", true);
            }
            return new BaseResponse<>(true, "操作成功", false);
        } catch (Exception e) {
            log.error("获取竞价小组成员符合性审查审核提交是否最后一个异常", e);
            return new BaseResponse<>(false, "获取竞价小组成员符合性审查审核提交是否最后一个异常");
        }
    }

    /**
     * 获取符合竞价需求的供应商数量 (要求竞价小组全部审核完成)
     *
     * @param biddingCode 竞价编号
     * @return BaseResponse<Integer>
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse<Integer> getBiddingComformanceAuditAvailableCount(String biddingCode) {
        BaseResponse<Integer> baseResponse;
        log.info("获取符合竞价需求的供应商数量-请求参数 竞价编号 : {}", biddingCode);
        // 请求参数检查
        if (Lang.isEmpty(biddingCode)) {
            baseResponse = new BaseResponse<>("获取符合竞价需求的供应商数量失败-竞价编号不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }
        try {
            // 根据 竞价编号 得到 竞价对象
            JcbdBidding bidding;
            BaseResponse<JcbdBidding> biddingResponse = findAndCheckBidding(biddingCode);
            if (biddingResponse.isSuccess()) {
                bidding = biddingResponse.getResult();
            } else {
                return new BaseResponse<>(biddingResponse.getResultMessage());
            }

            List<JcbdBiddingGroupMember> biddingGroupMembers = bidding.getJcbdBiddingGroupMembers().stream().filter(biddingGroupMember -> !biddingGroupMember.getIsDelete()).collect(Collectors.toList());

            int auditComplishedCount = 0;
            for (JcbdBiddingGroupMember biddingGroupMember : biddingGroupMembers) {
                boolean auditComplished = isBiddingConformityReviewAuditComplished(bidding, biddingGroupMember.getGroupMemberId(),biddingGroupMember.getEmail());
                if (auditComplished) {
                    auditComplishedCount++;
                }
            }

            // 全部审核完成
            if (auditComplishedCount != biddingGroupMembers.size()) {
                baseResponse = new BaseResponse<>("获取符合竞价需求的供应商数量失败-竞价小组成员尚未完成符合性审查审核");
                log.error(baseResponse.getResultMessage());
                return baseResponse;
            }
            // 统计符合竞价需求的供应商数量
            int auditYesCount = getBiddingComformanceAuditYesSuppliersCount(bidding);
            log.info("符合竞价需求的供应商数量 {}", auditYesCount);
            return new BaseResponse<>(true, "操作成功", auditYesCount);
        } catch (Exception e) {
            log.error("获取符合竞价需求的供应商数量异常", e);
            return new BaseResponse<>(false, "获取符合竞价需求的供应商数量异常");
        }
    }

    @Override
    public Page<JcbdBiddingQueryResultDto> supplierFindJcbdBiddingByParams(JcbdBiddingQueryDto jcbdBiddingQueryDto) {
        return null;
    }

    /**
     * 获取并检查竞价数据
     *
     * @param biddingCode 竞价编号
     * @return BaseResponse<JcbdBidding>
     */
    private BaseResponse<JcbdBidding> findAndCheckBidding(String biddingCode) {
        JcbdBidding bidding = new JcbdBidding();
        bidding.setBiddingCode(biddingCode);
        List<JcbdBidding> biddings = biddingRepo.findAll(new BiddingSpecification<>(bidding));
        if (biddings.size() != 1) {
            log.error("获取竞价小组成员符合性审查审核是否完成-竞价编号为 {} 的数据要求有且只有有 1 条, 实际 {} 条", biddingCode, biddings.size());
            return new BaseResponse<>("获取竞价小组成员符合性审查审核是否完成失败-数据检查未通过");
        }
        return new BaseResponse<>(true, "操作成功", biddings.get(0));
    }

    /**
     * 获取竞价小组成员符合性审查审核是否完成
     *
     * @param bidding   竞价 ID
     * @param auditorId 审核人 ID
     * @return boolean
     */
    public boolean isBiddingConformityReviewAuditComplished(JcbdBidding bidding, String auditorId,String email) {
        // 获得所有需求
        List<JcbdBiddingDemand> biddingDemands = bidding.getJcbdBiddingDemands().stream().filter(biddingDemand -> !biddingDemand.getIsDelete()).collect(Collectors.toList());
        // 获取所有供应商
        List<JcbdBdProvider> biddingProviders = bidding.getJcbdBdProviders().stream().filter(biddingProvider -> !biddingProvider.getIsDelete()).collect(Collectors.toList());
        // 过滤未响应
        biddingProviders = biddingProviders.stream().filter(biddingProvider -> !BiddingStatus.SUPPLIER_RESPONSE_NOT.getValue().equals(biddingProvider.getStatus())).collect(Collectors.toList());
        // 过滤已放弃
        biddingProviders = biddingProviders.stream().filter(biddingProvider -> !BiddingStatus.SUPPLIER_RESPONSE_GIVEUP.getValue().equals(biddingProvider.getStatus())).collect(Collectors.toList());
        // 供应商
        for (JcbdBdProvider biddingProvider : biddingProviders) {
            // 需求
            for (JcbdBiddingDemand biddingDemand : biddingDemands) {
                // 需求符合性审查项
                List<JcbdBiddingConformance> biddingConformances = biddingDemand.getJcbdBiddingConformances().stream().filter(biddingConformance -> !biddingConformance.getIsDelete()).collect(Collectors.toList());
                int conformanceAuditCount = 0;
                for (JcbdBiddingConformance biddingConformance : biddingConformances) {
                    // 需求符合性审查项审核结果
                    List<JcbdConformanceCheck> biddingConformanceChecks = jcbdConformanCheckRepo.findByBiddingConformanceId(biddingConformance.getId()).stream().filter(biddingConformanceCheck -> !biddingConformanceCheck.getIsDelete()).collect(Collectors.toList());
                    for (JcbdConformanceCheck biddingConformanceCheck : biddingConformanceChecks) {
                        // 找到审核记录
                        if(!Lang.isEmpty(biddingConformanceCheck.getBiddingGroupMemberId())){
                            if (biddingProvider.getProviderId().equals(biddingConformanceCheck.getProviderId()) && biddingConformanceCheck.getBiddingGroupMemberId().equals(auditorId)) {
                                JcbdBiddingGroupMember jcbdMember = jcbdBiddingGroupMemberRepo.findByGroupMemberIdAndJcbdBiddingIdAndIsDelete(auditorId, bidding.getId(), false);
                                if(!Lang.isEmpty(jcbdMember.getConfomanceCheck())){
                                    conformanceAuditCount++;
                                }
                            }
                        }else {
                            if (biddingProvider.getProviderId().equals(biddingConformanceCheck.getProviderId()) && biddingConformanceCheck.getEmail().equals(email)) {
                                JcbdBiddingGroupMember jcbdMember = jcbdBiddingGroupMemberRepo.findByEmailAndJcbdBiddingIdAndIsDelete(email, bidding.getId(), false);
                                if(!Lang.isEmpty(jcbdMember.getConfomanceCheck())) {
                                    conformanceAuditCount++;
                                }
                            }
                        }
                    }
                }
                // 符合性检查项审核数量 与 符合性检查项数量 比较一致说明完成审核
                if (conformanceAuditCount != biddingConformances.size()) {
                    return false;
                }
            }
        }
        return true;
    }



    @Autowired
    JcbdBdProviderRepo jcbdBdProviderRepo;

    /**
     * 获得符合性审查审核通过的供应商数量
     *
     * @param bidding 竞价对象
     * @return int
     */
    @Transactional
    public int getBiddingComformanceAuditYesSuppliersCount(JcbdBidding bidding) {
        List<JcbdBdProvider> biddingProviders = bidding.getJcbdBdProviders().stream().filter(biddingProvider -> !biddingProvider.getIsDelete()).collect(Collectors.toList());
        // 符合性审查审核通过的供应商ID
        List<String> auditYesBiddingProviderIds = new ArrayList<>();
        for (JcbdBdProvider biddingProvider : biddingProviders) {
            auditYesBiddingProviderIds.add(biddingProvider.getProviderId());
        }

        List<JcbdBiddingDemand> biddingDemands = bidding.getJcbdBiddingDemands().stream().filter(biddingDemand -> !biddingDemand.getIsDelete()).collect(Collectors.toList());
        for (JcbdBiddingDemand biddingDemand : biddingDemands) {
            List<JcbdBiddingConformance> biddingConformances = biddingDemand.getJcbdBiddingConformances().stream().filter(biddingConformance -> !biddingConformance.getIsDelete()).collect(Collectors.toList());
            for (JcbdBiddingConformance biddingConformance : biddingConformances) {
                List<JcbdConformanceCheck> biddingConformanceChecks = jcbdConformanCheckRepo.findByBiddingConformanceId(biddingConformance.getId()).stream().filter(biddingConformanceCheck -> !biddingConformanceCheck.getIsDelete()).collect(Collectors.toList());
                for (JcbdConformanceCheck biddingConformanceCheck : biddingConformanceChecks) {
                    // 供应商仍然处于审核通过列表继续检查是否有未通过审核记录
                    if (auditYesBiddingProviderIds.contains(biddingConformanceCheck.getProviderId())) {
                        // 有审核未通过记录,将此供应商从审核通过列表剔除
                        if (BiddingConformAuditStatus.NOT_ACCORD.getValue().equals(biddingConformanceCheck.getStatus())) {
                            auditYesBiddingProviderIds.remove(biddingConformanceCheck.getProviderId());
                        }
                    }
                }
            }
        }
        ArrayList<JcbdBdProvider> jcbdBdProviders = new ArrayList<>();
        for (String providerId:auditYesBiddingProviderIds) {
            JcbdBdProvider provider = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(bidding.getId(),providerId,false);
            provider.setConformanceCheckStatus(BiddingConformAuditStatus.ACCORD.getValue());
            jcbdBdProviders.add(provider);
        }
        jcbdBdProviderRepo.save(jcbdBdProviders);
        return auditYesBiddingProviderIds.size();
    }

    /**
     * bean 转换工具
     */
    private MapperFacade mapperFacade;

    /**
     * 竞价小组
     */
    @Autowired
    private JcbdBiddingGroupMemberRepo jcbdBiddingGroupMemberRepo;

    /**
     * 竞价数据操作对象
     */
    private JcbdBiddingRepo biddingRepo;

    /**
     * 竞价需求数据操作对象
     */
    private JcbdBiddingDemandRepo biddingDemandRepo;

    /**
     * 竞价需求符合性检查项数据操作对象
     */
    private JcbdBiddingConformanceRepo biddingConformanceRepo;

    /**
     * 竞价需求供应商响应数据操作对象
     */
    private JcbdBiddingDemandRespRepo biddingDemandRespRepo;

    /**
     * 竞价小组符合性审查数据操作对象
     */
    private JcbdConformanCheckRepo biddingConformanCheckRepo;

    /**
     * 竞价供应商数据操作对象
     */
    private JcbdBdProviderRepo biddingProviderRepo;

    /**
     * 竞价需求地区数据操作对象
     */
    private JcbdDemandAddressNumRepo biddingDemandAddressRepo;

    /**
     * 竞价需求地区供应商反馈数据操作对象
     */
    private JcbdProviderAddressRespRepo providerAddressRespRepo;

    /**
     * 竞价需求符合性审查项供应商响应数据操作对象
     */
    private JcbdProviderConformanceRespRepo providerConformanceRespRepo;

    /**
     * 符合性审查事务层
     */
    private BiddingConformityReviewTransactionalService biddingConformityReviewTransactionalService;

    @Autowired
    private void setMapperFacade(MapperFacade mapperFacade) {
        this.mapperFacade = mapperFacade;
    }

    @Autowired
    private void setBiddingRepo(JcbdBiddingRepo biddingRepo) {
        this.biddingRepo = biddingRepo;
    }

    @Autowired
    private void setBiddingDemandRepo(JcbdBiddingDemandRepo biddingDemandRepo) {
        this.biddingDemandRepo = biddingDemandRepo;
    }

    @Autowired
    private void setBiddingConformanceRepo(JcbdBiddingConformanceRepo biddingConformanceRepo) {
        this.biddingConformanceRepo = biddingConformanceRepo;
    }

    @Autowired
    private void setBiddingDemandRespRepo(JcbdBiddingDemandRespRepo biddingDemandRespRepo) {
        this.biddingDemandRespRepo = biddingDemandRespRepo;
    }

    @Autowired
    private void setBiddingConformanCheckRepo(JcbdConformanCheckRepo biddingConformanCheckRepo) {
        this.biddingConformanCheckRepo = biddingConformanCheckRepo;
    }

    @Autowired
    private void setBiddingProviderRepo(JcbdBdProviderRepo biddingProviderRepo) {
        this.biddingProviderRepo = biddingProviderRepo;
    }

    @Autowired
    private void setBiddingDemandAddressRepo(JcbdDemandAddressNumRepo biddingDemandAddressRepo) {
        this.biddingDemandAddressRepo = biddingDemandAddressRepo;
    }

    @Autowired
    private void setProviderAddressRespRepo(JcbdProviderAddressRespRepo providerAddressRespRepo) {
        this.providerAddressRespRepo = providerAddressRespRepo;
    }

    @Autowired
    private void setProviderConformanceRespRepo(JcbdProviderConformanceRespRepo providerConformanceRespRepo) {
        this.providerConformanceRespRepo = providerConformanceRespRepo;
    }

    @Autowired
    public void setBiddingConformityReviewTransactionalService(BiddingConformityReviewTransactionalService biddingConformityReviewTransactionalService) {
        this.biddingConformityReviewTransactionalService = biddingConformityReviewTransactionalService;
    }
}
