package bidding.service.service.impl;

import bidding.model.po.*;
import bidding.model.repository.*;
import bidding.model.specification.BiddingSpecification;
import bidding.model.specification.BiddingStatusTransformSpecification;
import bidding.service.service.BiddingMainService;
import com.alibaba.fastjson.JSON;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import definition.*;
import dto.*;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import sinomall.global.common.response.BaseResponse;
import uninogift.api.protocol.SupplierLoginApi;
import utils.Lang;
import utils.data.BeanMapper;
import utils.lock.RedisLockUtil;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 竞价主管业务逻辑层
 *
 * @author Drury
 * @date 2018/3/19
 */
@Slf4j
@Service
public class BiddingMainServiceImpl implements BiddingMainService {

    @Autowired
    MapperFacade mapperFacade;

    @Autowired
    JcbdBiddingRepo jcbdBiddingRepo;

    @Autowired
    JcbdBdGroupCheckRepo jcbdBdGroupCheckRepo;

    @Autowired
    JcbdBiddingDemandRepo jcbdBiddingDemandRepo;

    @Autowired
    BiddingStatusTransformRepo biddingStatusTransformRepo;
    @Autowired
    BiddingStatusCategoryRepo biddingStatusCategoryRepo;

    @Autowired
    JcbdBdProviderRepo jcbdBdProviderRepo;

    @Autowired
    JcbdBiddingGroupMemberRepo jcbdBiddingGroupMemberRepo;

    @Autowired
    JcbdBdSkuCheckRepo jcbdBdSkuCheckRepo;
    @Autowired
    JcbdOrderRepo jcbdOrderRepo;
    @Autowired
    JcbdMallOrderRepo jcbdMallOrderRepo;

    @Autowired
    JcbdTempOrderRepo jcbdTempOrderRepo;
    @Autowired
    JcbdTempGoodsRepo jcbdTempGoodsRepo;
//    @MotanReferer
//    OrderMainApi orderMainApi;
    @Autowired
    RedisLockUtil redisLockUtil;
    @Autowired
    RedisTemplate redisTemplate;
    @Autowired
    VoteDetailRepo voteDetailRepo;
    @Autowired
    JdbcTemplate jdbcTemplate;
    @Autowired
    JcbdBiddingOperationLogRepo jcbdBiddingOperationLogRepo;

    @MotanReferer
    SupplierLoginApi supplierLoginApi;

    @Autowired
    JcbdBiddingDemandRespRepo jcbdBiddingDemandRespRepo;
    /**
     * 竞价投票编号KEY
     */
    public String VOTENO_CODE_REDIS_KEY = "BIDDING:VOTENO_CODE_REDIS_KEY:";

    /**
     * 竞价投票编号KEY
     */
    public String GROUPCHECKNO_CODE_REDIS_KEY = "BIDDING:GROUPCHECKNO_CODE_REDIS_KEY:";

    /**
     * 竞价保存更新接口
     *
     * @param biddingDto
     * @return
     */
    @Override
    @Transactional
    public BaseResponse<JcbdBiddingDto> saveOrUpdateBiddingBaseInfo(JcbdBiddingDto biddingDto) {
        JcbdBiddingDto jbdDto = null;
        if (Lang.isEmpty(biddingDto) || Lang.isEmpty(biddingDto.getBiddingCode())) {
            return new BaseResponse(false, "竞价数据为空", null);
        }
        JcbdBidding jb = null;

        try {
            List<JcbdBiddingOperationLog> operationLogs = new ArrayList<>();
            if (!Lang.isEmpty(biddingDto.getId())) {
                jb = jcbdBiddingRepo.findByIdAndIsDelete(biddingDto.getId(), false);
                if (!jb.getBiddingCode().equals(biddingDto.getBiddingCode())) {
                    return new BaseResponse(false, "竞价编号不允许修改", null);
                }
                operationLogs = jcbdBiddingOperationLogRepo.findByBiddingId(jb.getId());
                jcbdBiddingRepo.delete(jb);
                jcbdBiddingRepo.flush();
            }

            JcbdBidding jcbdBidding = mapperFacade.map(biddingDto, JcbdBidding.class);
            if (jb != null) {
                List<JcbdBiddingDemand> jcbdBiddingDemands = jb.getJcbdBiddingDemands();
                jcbdBidding.setJcbdBiddingDemands(jcbdBiddingDemands);
                jcbdBiddingDemands.forEach(demand -> demand.setJcbdBidding(jcbdBidding));
            }

            List<JcbdBiddingGroupMember> jcbdBiddingGroupMembers = jcbdBidding.getJcbdBiddingGroupMembers();
            if (!Lang.isEmpty(jcbdBiddingGroupMembers)) {
                for (int i = 0; i < jcbdBiddingGroupMembers.size(); i++) {
                    JcbdBiddingGroupMember jbgm = jcbdBiddingGroupMembers.get(i);
                    jbgm.setJcbdBidding(jcbdBidding);
                    jbgm.setStatus(BiddingStatus.OM_SUBMIT_WAIT.getValue());
                }
            }
            List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
            if (!Lang.isEmpty(jcbdBdProviders)) {
                for (int i = 0; i < jcbdBdProviders.size(); i++) {
                    JcbdBdProvider jbp = jcbdBdProviders.get(i);
                    jbp.setJcbdBidding(jcbdBidding);
                }
            }

            //待提交状态
            jcbdBidding.setStatus(BiddingStatus.OM_SUBMIT_WAIT.getValue());

            JcbdBidding jbd = jcbdBiddingRepo.saveAndFlush(jcbdBidding);
            if (!Lang.isEmpty(operationLogs)) {
                operationLogs.forEach(operationLog -> {
                    operationLog.setBiddingId(jbd.getId());
                });
                jcbdBiddingOperationLogRepo.save(operationLogs);
            }
            jbdDto = mapperFacade.map(jbd, JcbdBiddingDto.class);

        } catch (Exception e) {
            log.error("数据库操作异常", e);
            throw e;
//            return new BaseResponse(false, "竞价保存失败", null);
        }
        return new BaseResponse(true, "操作成功", jbdDto);
    }

    /**
     * 竞价需求保存更新接口
     *
     * @param biddingDemandDto
     * @return
     */
    @Override
    public BaseResponse saveOrUpdateBiddingDemand(JcbdBiddingDemandDto biddingDemandDto) {

        if (Lang.isEmpty(biddingDemandDto)) {
            return new BaseResponse(false, "竞价需求数据为空");
        }
        if (Lang.isEmpty(biddingDemandDto.getBiddingId())) {
            return new BaseResponse(false, "竞价id为空");
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByIdAndIsDelete(biddingDemandDto.getBiddingId(), false);
        if (Lang.isEmpty(jcbdBidding)) {
            return new BaseResponse(false, "竞价不存在");
        }
        JcbdBiddingDemand jcbdBiddingDemand = null;
        //查询竞价需求是否已存在
        if (!Lang.isEmpty(biddingDemandDto.getId())) {
            jcbdBiddingDemand = jcbdBiddingDemandRepo.findByIdAndIsDelete(biddingDemandDto.getId(), false);
        }
//        try {
        if (!Lang.isEmpty(jcbdBiddingDemand)) {
            if (!jcbdBiddingDemand.getDemandCode().equals(biddingDemandDto.getDemandCode())) {
                return new BaseResponse(false, "竞价需求编号不允许修改");
            }
            //竞价需求编号已存在，先删除
            jcbdBiddingDemandRepo.delete(jcbdBiddingDemand.getId());

        }
        jcbdBiddingDemand = mapperFacade.map(biddingDemandDto, JcbdBiddingDemand.class);

        List<JcbdDemandAddressNum> jcbdDemandAddressNums = jcbdBiddingDemand.getJcbdDemandAddressNums();
        if (!Lang.isEmpty(jcbdDemandAddressNums)) {
            for (int i = 0; i < jcbdDemandAddressNums.size(); i++) {
                JcbdDemandAddressNum jdan = jcbdDemandAddressNums.get(i);
                jdan.setJcbdBiddingDemand(jcbdBiddingDemand);
            }
        }
        List<JcbdBiddingConformance> jcbdBiddingConformances = jcbdBiddingDemand.getJcbdBiddingConformances();
        if (!Lang.isEmpty(jcbdBiddingConformances)) {
            for (int i = 0; i < jcbdBiddingConformances.size(); i++) {
                JcbdBiddingConformance jbc = jcbdBiddingConformances.get(i);
                jbc.setJcbdBiddingDemand(jcbdBiddingDemand);
            }
        }

        jcbdBiddingDemand.setJcbdBidding(jcbdBidding);
        jcbdBiddingDemandRepo.save(jcbdBiddingDemand);

//        } catch (Exception e) {
//            log.error("数据库操作异常", e);
//            return new BaseResponse(false, "竞价需求保存失败");
//        }

        return new BaseResponse(true, "操作成功");
    }

    /**
     * 竞价删除接口（逻辑删除）
     *
     * @param biddingNo
     * @return
     */
    @Override
    @Transactional
    public BaseResponse deleteBidding(String biddingNo) {
        if (Lang.isEmpty(biddingNo)) {
            return new BaseResponse(false, "竞价编号为空");
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (Lang.isEmpty(jcbdBidding)) {
            return new BaseResponse(false, "竞价数据异常");
        }
        List<JcbdBiddingGroupMember> jcbdBiddingGroupMembers = jcbdBidding.getJcbdBiddingGroupMembers();
        jcbdBiddingGroupMembers.forEach(member -> {
            member.setIsDelete(true);
        });
        jcbdBiddingGroupMemberRepo.save(jcbdBiddingGroupMembers);
        List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
        jcbdBdProviders.forEach(jcbdBdProvider -> {
            jcbdBdProvider.setIsDelete(true);
        });
        jcbdBdProviderRepo.save(jcbdBdProviders);

        jcbdBidding.setIsDelete(true);
        try {
            jcbdBiddingRepo.saveAndFlush(jcbdBidding);
        } catch (Exception e) {
            log.error("数据库操作异常", e);
            return new BaseResponse(false, "竞价删除失败");
        }
        return new BaseResponse(true, "操作成功");
    }


    /**
     * 竞价需求删除接口（逻辑删除）
     *
     * @param biddingDemandNo
     * @return
     */
    @Override
    public BaseResponse deleteBiddingDemand(String biddingDemandNo) {

        if (Lang.isEmpty(biddingDemandNo)) {
            return new BaseResponse(false, "竞价需求编号为空");
        }
        JcbdBiddingDemand jcbdBiddingDemand = jcbdBiddingDemandRepo.findByDemandCodeAndIsDelete(biddingDemandNo, false);
        if (Lang.isEmpty(jcbdBiddingDemand)) {
            return new BaseResponse(false, "竞价需求数据不存在");
        }
        jcbdBiddingDemand.setIsDelete(true);
        try {
            jcbdBiddingDemandRepo.saveAndFlush(jcbdBiddingDemand);
        } catch (Exception e) {
            log.error("数据库操作异常", e);
            return new BaseResponse(false, "竞价删除失败");
        }
        return new BaseResponse(true, "操作成功");
    }

    @Transactional
    @Override
    public BiddingStateResponseDto changeBiddingState(BiddingChangeStateDto changeStateDto) {
        JcbdBidding bidding = null;
        BiddingStatusTransform biddingStatusTransform = null;
        String statusCategoryCode = BiddingMainService.BIDDING_STATUS_CATEGORY_BIDDING;
        BiddingStatusCategory biddingStatusCategory = biddingStatusCategoryRepo.findByBiddingCategoryCode(statusCategoryCode);
        try {
            if (StringUtils.isEmpty(changeStateDto.getBiddingNo()) || StringUtils.isEmpty(changeStateDto.getCurrentState()) || StringUtils.isEmpty(changeStateDto.getOpcode())) {
                throw new IllegalArgumentException("更新竞价状态参数不正确");
            }
            bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(changeStateDto.getBiddingNo(), false);
            if (bidding == null) {
                throw new IllegalArgumentException("不存在该竞价");
            }
            if (!StringUtils.equals(bidding.getStatus(), changeStateDto.getCurrentState())) {
                throw new IllegalArgumentException("当前竞价状态有误");
            }
            biddingStatusTransform = getBiddingStatusTransform(changeStateDto, biddingStatusCategory);
        } catch (Exception e) {
            log.error("竞价状态转换参数错误,changeStateDto={}", JSON.toJSONString(changeStateDto), e);
            return new BiddingStateResponseDto(false, e.getMessage());
        }
        bidding.setStatus(biddingStatusTransform.getNextStatus());
        jcbdBiddingRepo.saveAndFlush(bidding);
        log.info("竞价状态修改成功，同步竞价小组成员状态 changeStateDto={}", JSON.toJSONString(changeStateDto));
        List<JcbdBiddingGroupMember> jcbdBiddingGroupMembers = bidding.getJcbdBiddingGroupMembers();
        for (JcbdBiddingGroupMember jcbdBiddingGroupMember : jcbdBiddingGroupMembers) {
            jcbdBiddingGroupMember.setStatus(bidding.getStatus());
        }
        jcbdBiddingGroupMemberRepo.save(jcbdBiddingGroupMembers);
        return new BiddingStateResponseDto(true, "操作成功", bidding.getStatus(), bidding.getId());
    }

    @Override
    public BiddingStatusTransform getBiddingStatusTransform(BiddingChangeStateDto changeStateDto, BiddingStatusCategory biddingStatusCategory) {
        BiddingStatusTransform biddingStatusTransform;
        biddingStatusTransform = new BiddingStatusTransform();
        biddingStatusTransform.setBiddingStatusCategoryId(biddingStatusCategory.getId());
        biddingStatusTransform.setCurrentStatus(changeStateDto.getCurrentState());
        biddingStatusTransform.setOperationCode(changeStateDto.getOpcode());
        List<BiddingStatusTransform> biddingStatusTransforms = biddingStatusTransformRepo.findAll(new BiddingStatusTransformSpecification<BiddingStatusTransform>(biddingStatusTransform));
        if (biddingStatusTransforms == null || biddingStatusTransforms.isEmpty()) {
            throw new IllegalArgumentException("操作码有误，没有对应的转换状态");
        }
        biddingStatusTransform = biddingStatusTransforms.get(0);
        return biddingStatusTransform;
    }

    /**
     * 更新竞价开始及结束响应时间
     *
     * @param biddingNo
     * @param timeStartResponse
     * @param timeEndResponse
     * @return
     */
    @Override
    public BaseResponse updateBiddingTime(String biddingNo, Date timeStartResponse, Date timeEndResponse) {
        if (Lang.isEmpty(biddingNo) || Lang.isEmpty(timeStartResponse) || Lang.isEmpty(timeEndResponse)) {
            return new BaseResponse(false, "更新竞价响应时间参数不正确");
        }
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (bidding == null) {
            return new BaseResponse(false, "不存在该竞价");
        }
        bidding.setTimeStartResponse(timeStartResponse);
        bidding.setTimeEndResponse(timeEndResponse);
        try {
            jcbdBiddingRepo.saveAndFlush(bidding);
        } catch (Exception e) {
            log.error("数据库操作异常", e);
            return new BaseResponse(false, "更新竞价响应时间失败");
        }
        return new BaseResponse(true, "操作成功");
    }

    /**
     * 重选竞价小组成员
     *
     * @param chooseBiddingMemberAgainDtoList
     * @return
     */
    @Override
    public BaseResponse chooseBiddingGroupMemberAgain(List<ChooseBiddingMemberAgainDto> chooseBiddingMemberAgainDtoList) {

        try {
            if (!Lang.isEmpty(chooseBiddingMemberAgainDtoList)) {
                for (ChooseBiddingMemberAgainDto cbmad : chooseBiddingMemberAgainDtoList) {
                    if (!Lang.isEmpty(cbmad.getId())) {
                        JcbdBiddingGroupMember member = jcbdBiddingGroupMemberRepo.findOne(cbmad.getId());
                        if (!Lang.isEmpty(cbmad.getEffective())) {
                            member.setEffective(cbmad.getEffective());
                        }
                        if (!Lang.isEmpty(cbmad.getEmail())) {
                            member.setEmail(cbmad.getEmail());
                        }
                        jcbdBiddingGroupMemberRepo.save(member);
                    }
                }
            }
            return new BaseResponse(true, "操作成功");
        } catch (Exception e) {
            log.error("数据库操作异常", e);
            return new BaseResponse(false, "重选竞价小组成员");
        }
    }

    @Autowired
    BiddingMainService biddingMainService;

    @Override
    @Transactional
    public BaseResponse auditBiddingDemand(BiddingAuditDTO biddingAudit) {

        final String noPass = "0";
        String biddingNo = biddingAudit.getBiddingNo();
        if (Lang.isEmpty(biddingNo)) {
            return new BaseResponse(false, "biddingNo不为空");
        }
        if (Lang.isEmpty(biddingAudit.getStatus())) {
            return new BaseResponse(false, "Status不为空");
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        JcbdBiddingGroupMember groupMember = null;
        if (!Lang.isEmpty(biddingAudit.getAuditorId())) {
            groupMember = jcbdBiddingGroupMemberRepo.findByGroupMemberIdAndJcbdBiddingIdAndIsDelete(biddingAudit.getAuditorId(), jcbdBidding.getId(), false);
        } else if (!Lang.isEmpty(biddingAudit.getEmail())) {
            groupMember = jcbdBiddingGroupMemberRepo.findByEmailAndJcbdBiddingIdAndIsDelete(biddingAudit.getEmail(), jcbdBidding.getId(), false);
        }

        if (Lang.isEmpty(groupMember)) {
            return new BaseResponse("用户信息为空");
        }
        JcbdBdGroupCheck groupCheck = jcbdBdGroupCheckRepo.findByGroupMemberIdAndJcbdBiddingBiddingCode(groupMember.getId(), biddingNo);
        if (!Lang.isEmpty(groupCheck)) {
            return new BaseResponse("已审核，不能重复审核");
        }

        JcbdBdGroupCheck jcbdBdGroupCheck = new JcbdBdGroupCheck();
        jcbdBdGroupCheck.setStatus(biddingAudit.getStatus());
        jcbdBdGroupCheck.setContent(biddingAudit.getContent());
        jcbdBdGroupCheck.setGroupCheckNo(jcbdBidding.getGroupCheckNo());
        jcbdBdGroupCheck.setGroupMemberId(groupMember.getId());
        jcbdBdGroupCheck.setJcbdBidding(jcbdBidding);
        jcbdBdGroupCheckRepo.save(jcbdBdGroupCheck);

        if (jcbdBdGroupCheck.getStatus().equals(noPass)) {
            BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
            biddingChangeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_NOPASS.getCode());
            biddingChangeStateDto.setCurrentState(BiddingStatus.OM_AUDIT_WAIT.getValue());
            biddingChangeStateDto.setBiddingNo(biddingNo);
            biddingMainService.changeBiddingState(biddingChangeStateDto);
            return new BaseResponse(true, "审核意见保存成功");
        }

        Integer memberSize = jcbdBidding.getJcbdBiddingGroupMembers().size();
        List<JcbdBdGroupCheck> checks = jcbdBdGroupCheckRepo.findByJcbdBiddingBiddingCodeAndGroupCheckNo(biddingNo, jcbdBidding.getGroupCheckNo());
        if (memberSize.equals(checks.size())) {
            for (JcbdBdGroupCheck check : checks) {
                if (noPass.equals(check.getStatus())) {
                    BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
                    biddingChangeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_NOPASS.getCode());
                    biddingChangeStateDto.setCurrentState(BiddingStatus.OM_AUDIT_WAIT.getValue());
                    biddingChangeStateDto.setBiddingNo(biddingNo);
                    biddingMainService.changeBiddingState(biddingChangeStateDto);
                    return new BaseResponse(true, "审核意见保存成功");
                }
            }
            BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
            biddingChangeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_PASS.getCode());
            biddingChangeStateDto.setCurrentState(BiddingStatus.OM_AUDIT_WAIT.getValue());
            biddingChangeStateDto.setBiddingNo(biddingNo);
            biddingMainService.changeBiddingState(biddingChangeStateDto);
        } else {
            groupMember.setStatus(BiddingStatus.M_Bidding_AUDIT_OK.getValue());
            jcbdBiddingGroupMemberRepo.save(groupMember);
        }


        return new BaseResponse(true, "审核意见保存成功");


    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse openPrice(String biddingNo) {
        List<JcbdBdProvider> jcbdBdProviderList = jcbdBdProviderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(biddingNo, false);
        jcbdBdProviderList = jcbdBdProviderList.stream().filter(provider -> provider.getStatus().equals(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue())).collect(Collectors.toList());
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        //最小价格
        BigDecimal minPrice = null;
        List<JcbdBdProvider> bidProviders = new ArrayList<>();
        //满足
        final String accord = "1";
        for (JcbdBdProvider provider : jcbdBdProviderList) {
            BigDecimal totalPrice = new BigDecimal(0);
            Boolean isAccord = true;
            for (JcbdBiddingDemand demand : jcbdBiddingDemands) {
                Long minPurchaseAmount = demand.getMinPurchaseAmount();
                if (minPurchaseAmount == null) {
                    minPurchaseAmount = Long.parseLong("1");
                }
                List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses = demand.getJcbdBiddingDemandResponses();
                for (JcbdBiddingDemandResponse demandResponse : jcbdBiddingDemandResponses) {
                    if (provider.getId().equals(demandResponse.getResponseProviderId())) {
                        if (!accord.equals(demandResponse.getIsAllAccord())) {
                            isAccord = false;
                        }
                        BigDecimal price = demandResponse.getPrice();
                        totalPrice = totalPrice.add((price.multiply(new BigDecimal(minPurchaseAmount))).setScale(2, BigDecimal.ROUND_UP));
                    }
                }
            }
            if (isAccord) {
                if (minPrice == null) {
                    minPrice = totalPrice;
                    bidProviders.add(provider);
                } else {
                    if (totalPrice.compareTo(minPrice) < 0) {
                        bidProviders.clear();
                        minPrice = totalPrice;
                        bidProviders.add(provider);
                    } else if (totalPrice.compareTo(minPrice) == 0) {
                        bidProviders.add(provider);
                    }
                }
            }
        }
        if (!Lang.isEmpty(bidProviders)) {
            if (bidProviders.size() == 1) {
                BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
                biddingChangeStateDto.setBiddingNo(biddingNo);
                biddingChangeStateDto.setCurrentState(BiddingStatus.OM_CONFORM_AUDIT_YES.getValue());
                biddingChangeStateDto.setOpcode(BiddingOperateCode.START_QUOTED_PRICE_EQUALS_ONE.getCode());
                changeBiddingState(biddingChangeStateDto);
                for (JcbdBdProvider jcbdBdProvider : jcbdBdProviderList) {
                    if (!jcbdBdProvider.getId().equals(bidProviders.get(0).getId())) {
                        jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_NOT.getValue());
                        jcbdBdProviderRepo.save(jcbdBdProvider);
                    } else {
                        jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue());
                        jcbdBdProvider.setDealStatus("已成交");
                        jcbdBdProviderRepo.save(jcbdBdProvider);
                    }
                }
            } else {
                BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
                biddingChangeStateDto.setBiddingNo(biddingNo);
                biddingChangeStateDto.setCurrentState(BiddingStatus.OM_CONFORM_AUDIT_YES.getValue());
                if (WayforConfirmBiddingResult.BIDDING_RADIO_VOTE.getValue().equals(jcbdBidding.getWayForConfirmBiddingResult())) {
                    biddingChangeStateDto.setOpcode(BiddingOperateCode.START_QUOTED_PRICE_OVER_ONE_RANDOM.getCode());
                } else if (WayforConfirmBiddingResult.BIDDING_RADIO_RANDOM.getValue().equals(jcbdBidding.getWayForConfirmBiddingResult())) {
                    biddingChangeStateDto.setOpcode(BiddingOperateCode.START_QUOTED_PRICE_OVER_ONE_VOTE.getCode());
                } else {
                    throw new RuntimeException("没有已经选择的确认方式");
                }
                changeBiddingState(biddingChangeStateDto);

                for (JcbdBdProvider best : bidProviders) {
                    best.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue());
                    jcbdBdProviderRepo.save(best);
                }
                jcbdBdProviderList.removeAll(bidProviders);
                for (JcbdBdProvider other : jcbdBdProviderList) {
                    other.setStatus(BiddingStatus.AUDITOR_RESULT_AUDIT_NOT.getValue());
                    jcbdBdProviderRepo.save(other);
                }

            }
        }
        return new BaseResponse(true, "开启报价操作成功");
    }

    /**
     * 竞价结束校验定时任务
     *
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<List<JcbdBiddingDto>> biddingResponseFinishValidateAll() {
        final Integer[] providerResponseNum = {0};
        // 获取当前时间
        Date currentTime = Calendar.getInstance().getTime();
        JcbdBidding jcbdBidding = new JcbdBidding();
        jcbdBidding.setStatus(BiddingStatus.OM_RESPONSE_WAIT.getValue());
        BiddingSpecification<JcbdBidding> biddingSpecification = new BiddingSpecification<>();
        biddingSpecification.setCurrentTime(currentTime);
        biddingSpecification.setBidding(jcbdBidding);
        biddingSpecification.setBiddingEndFlag(true);
        List<JcbdBidding> jcbdBiddingList = jcbdBiddingRepo.findAll(biddingSpecification);
        if (Lang.isEmpty(jcbdBiddingList)) {
            log.info("BiddingMainServiceImpl.biddingResponseFinishValidateAll:未扫描到竞价数据!");
            return new BaseResponse<>(true, "该时间段无过期竞价!");
        } else {
            log.info("本次扫描共有{}条竞价截止", jcbdBiddingList.size());
        }
        jcbdBiddingList.forEach(bidding -> {
            BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
            List<JcbdBdProvider> jcbdBdProviderList = jcbdBdProviderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(bidding.getBiddingCode(), false);
            biddingChangeStateDto.setBiddingNo(bidding.getBiddingCode());
            biddingChangeStateDto.setCurrentState(BiddingStatus.OM_RESPONSE_WAIT.getValue());
            List<JcbdBdProvider> providers = new ArrayList<>();
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviderList) {
                if (jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_RESPONSE_YES.getValue())) {
                    providers.add(jcbdBdProvider);
                }
            }
            boolean isEnd = false;
            if (providers.size() < 2) {
                // 供应商数小于2 ，状态流转至竞价终止
                biddingChangeStateDto.setOpcode(BiddingOperateCode.ABANDON_RESPONSE.getCode());
                isEnd = true;
            } else {
                // 供应商数大于等于2，状态流转至竞价响应
                biddingChangeStateDto.setOpcode(BiddingOperateCode.RESPOND_BIDDING.getCode());
            }
            // 改变竞价状态
            this.changeBiddingState(biddingChangeStateDto);
            // 改变供应商状态
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviderList) {
                if (isEnd) {
                    if (BiddingStatus.SUPPLIER_RESPONSE_YES.getValue().equals(jcbdBdProvider.getStatus())) {
                        providerResponseNum[0] = providerResponseNum[0] + 1;
                    }
                    jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_BIDDING_END.getValue());

                } else {
                    if (BiddingStatus.SUPPLIER_RESPONSE_YES.getValue().equals(jcbdBdProvider.getStatus())) {
                        providerResponseNum[0] = providerResponseNum[0] + 1;
                        jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue());
                    } else if (BiddingStatus.SUPPLIER_RESPONSE_NOT.getValue().equals(jcbdBdProvider.getStatus()) || BiddingStatus.SUPPLIER_RESPONSE_YES_GIVEUP.getValue().equals(jcbdBdProvider.getStatus())) {
                        jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_RESPONSE_GIVEUP.getValue());
                    }
                }

                jcbdBdProviderRepo.save(jcbdBdProvider);
            }
            bidding.setProviderResponseNum(providerResponseNum[0]);
            jcbdBiddingRepo.save(bidding);
            log.info("竞价编号--{}--已处理", bidding.getBiddingCode());
        });

        return new BaseResponse<>(true, "已处理相关过期数据!");
    }

    /**
     * 竞价结束校验定时任务
     *
     * @param biddingNo 竞价编号
     * @return
     */
    @Transactional
    @Override
    public BaseResponse biddingResponseFinishValidate(String biddingNo) {
        Integer providerResponseNum = 0;
        if (Lang.isEmpty(biddingNo)) {
            log.info("BiddingMainServiceImpl.biddingResponseFinishValidate() parm is empty!");
            return new BaseResponse(false, "竞价编号为空!");
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (Lang.isEmpty(jcbdBidding)) {
            log.info("BiddingMainServiceImpl.biddingResponseFinishValidate():未扫描到过期竞价!");
            return new BaseResponse(false, "未扫描到过期竞价");
        }
        // 当前时间
        Date currentTime = Calendar.getInstance().getTime();
        // 结束响应时间
        Date timeEndResponse = jcbdBidding.getTimeEndResponse();
        //&& (timeEndResponse.getTime() < currentTime.getTime())
        if (!Lang.isEmpty(timeEndResponse)) {
            BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
            List<JcbdBdProvider> jcbdBdProviderList = jcbdBdProviderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(jcbdBidding.getBiddingCode(), false);
            biddingChangeStateDto.setBiddingNo(jcbdBidding.getBiddingCode());
            biddingChangeStateDto.setCurrentState(BiddingStatus.OM_RESPONSE_WAIT.getValue());
            List<JcbdBdProvider> providers = new ArrayList<>();
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviderList) {
                if (jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_RESPONSE_YES.getValue())) {
                    providers.add(jcbdBdProvider);
                }
            }
            boolean isEnd = false;
            if (providers.size() < 2) {
                // 供应商数小于2 ，状态流转至竞价终止
                biddingChangeStateDto.setOpcode(BiddingOperateCode.ABANDON_RESPONSE.getCode());
                isEnd = true;
            } else {
                // 供应商数大于等于2，状态流转至竞价响应
                biddingChangeStateDto.setOpcode(BiddingOperateCode.RESPOND_BIDDING.getCode());
            }
            // 改变竞价状态
            this.changeBiddingState(biddingChangeStateDto);
            // 改变供应商状态
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviderList) {
                if (isEnd) {
                    if (BiddingStatus.SUPPLIER_RESPONSE_YES.getValue().equals(jcbdBdProvider.getStatus())) {
                        providerResponseNum = providerResponseNum + 1;
                    }
                    jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_BIDDING_END.getValue());
                } else {
                    if (BiddingStatus.SUPPLIER_RESPONSE_YES.getValue().equals(jcbdBdProvider.getStatus())) {
                        providerResponseNum = providerResponseNum + 1;
                        jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue());
                    } else if (BiddingStatus.SUPPLIER_RESPONSE_NOT.getValue().equals(jcbdBdProvider.getStatus()) || BiddingStatus.SUPPLIER_RESPONSE_YES_GIVEUP.getValue().equals(jcbdBdProvider.getStatus())) {
                        jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_RESPONSE_GIVEUP.getValue());
                        jcbdBdProvider.setConformanceCheckStatus(BiddingConformAuditStatus.NOT_ACCORD.getValue());
                    }
                }

                jcbdBdProviderRepo.save(jcbdBdProvider);
            }
            jcbdBidding.setProviderResponseNum(providerResponseNum);
            jcbdBiddingRepo.save(jcbdBidding);
            return new BaseResponse("001", "竞价已过期!");
        }

        return new BaseResponse(false, "未找到相关数据!");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse auditBiddingSku(BiddingAuditDTO biddingAuditDTO) {
        final String pass = "1";
        String biddingNo = biddingAuditDTO.getBiddingNo();
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        JcbdBdSkuCheck jcbdBdSkuCheck = BeanMapper.map(biddingAuditDTO, JcbdBdSkuCheck.class);
        jcbdBdSkuCheck.setJcbdBidding(jcbdBidding);
        jcbdBdSkuCheckRepo.save(jcbdBdSkuCheck);
        String providerId="";
        if (pass.equals(biddingAuditDTO.getStatus())) {
            BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
            biddingChangeStateDto.setBiddingNo(biddingNo);
            biddingChangeStateDto.setCurrentState(BiddingStatus.OM_SKU_AUDIT_WAIT.getValue());
            biddingChangeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_SKU_PASS.getCode());
            changeBiddingState(biddingChangeStateDto);

            List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviders) {
                if (BiddingStatus.SUPPLIER_SKU_AUDIT_WAIT.getValue().equals(jcbdBdProvider.getStatus())) {
                    jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_BIDDING_YES_AGAIN.getValue());
                    jcbdBdProviderRepo.save(jcbdBdProvider);
                    providerId=jcbdBdProvider.getId();
                }
            }
        } else {
            BiddingChangeStateDto biddingChangeStateDto = new BiddingChangeStateDto();
            biddingChangeStateDto.setBiddingNo(biddingNo);
            biddingChangeStateDto.setCurrentState(BiddingStatus.OM_SKU_AUDIT_WAIT.getValue());
            biddingChangeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_SKU_NO_PASS.getCode());
            changeBiddingState(biddingChangeStateDto);

            List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviders) {
                if (BiddingStatus.SUPPLIER_SKU_AUDIT_WAIT.getValue().equals(jcbdBdProvider.getStatus())) {
                    jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_SKU_AUDIT_NOT.getValue());
                    jcbdBdProviderRepo.save(jcbdBdProvider);
                    providerId=jcbdBdProvider.getId();
                }
            }
        }
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        for(JcbdBiddingDemand jcbdBiddingDemand:jcbdBiddingDemands){
            JcbdBiddingDemandResponse jcbdBiddingDemandResponse = jcbdBiddingDemandRespRepo.findByJcbdBiddingDemandIdAndResponseProviderIdAndIsDelete(jcbdBiddingDemand.getId(), providerId, false);
            jcbdBiddingDemandResponse.setSkuIsModify(false);
            jcbdBiddingDemandRespRepo.save(jcbdBiddingDemandResponse);
        }
        return new BaseResponse(true, "审核SKU通过");
    }

    @Override
    @Transactional
    public BaseResponse biddingResultNoPass(String biddingNo) {
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (bidding == null) {
            log.error("竞价结果审查不通过回调失败，竞价不存在biddingNo={}", biddingNo);
        }
        BiddingChangeStateDto changeStateDto = new BiddingChangeStateDto();
        changeStateDto.setBiddingNo(biddingNo);
        changeStateDto.setCurrentState(BiddingStatus.AUDITOR_RESULT_AUDIT_WAIT.getValue());
        changeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_RESULT_NO_PASS.getCode());
        BiddingStateResponseDto biddingState = changeBiddingState(changeStateDto);
        if (!biddingState.isSuccess()) {
            log.error("竞价结果审查不通过回调失败，更新竞价失败，biddingNo={}，biddingState={}", biddingNo, JSON.toJSONString(biddingState));
        }
        for (JcbdBdProvider jcbdBdProvider : bidding.getJcbdBdProviders()) {
            jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_BIDDING_NOT.getValue());
            String dealStatus = "未成交";
            jcbdBdProvider.setDealStatus(dealStatus);
        }
        return new BaseResponse(true, "竞价结果审核不通过回调成功");
    }

    @Transactional
    @Override
    public BaseResponse<BiddingResultNotifyDto> finishBiddingResultAudit(String biddingNo) {
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (bidding == null) {
            log.error("竞价结果审查通过回调失败，竞价不存在biddingNo={}", biddingNo);
        }
        BiddingChangeStateDto changeStateDto = new BiddingChangeStateDto();
        changeStateDto.setBiddingNo(biddingNo);
        changeStateDto.setCurrentState(BiddingStatus.OM_RESULT_AUDIT_WAIT.getValue());
        changeStateDto.setOpcode(BiddingOperateCode.AUDIT_BIDDING_RESULT_PASS.getCode());
        changeStateDto.setOperationTime(new Date());
        changeStateDto.setOperationContent(BiddingOperateCode.AUDIT_BIDDING_RESULT_PASS.getCodeName());
        BiddingStateResponseDto biddingState = changeBiddingState(changeStateDto);
        if (!biddingState.isSuccess()) {
            log.error("竞价结果审查通过回调失败，更新竞价失败，biddingNo={}，biddingState={}", biddingNo, JSON.toJSONString(biddingState));
            new BaseResponse(false, "更新竞价失败");
        }
        BiddingResultNotifyDto biddingResultNotifyDto = new BiddingResultNotifyDto();
        List<NotifyProvider> notDealProviders = new ArrayList<>();

        for (JcbdBdProvider jcbdBdProvider : bidding.getJcbdBdProviders()) {
            if (BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue().equals(jcbdBdProvider.getStatus())) {
                jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_BIDDING_YES.getValue());
                String dealStatus = "已成交";
                jcbdBdProvider.setDealStatus(dealStatus);
                NotifyProvider notifyProvider = new NotifyProvider();
                notifyProvider.setProviderId(jcbdBdProvider.getProviderId());
                notifyProvider.setName(jcbdBdProvider.getProviderName());
                biddingResultNotifyDto.setDealProvider(notifyProvider);
            } else {
                jcbdBdProvider.setStatus(BiddingStatus.SUPPLIER_BIDDING_NOT.getValue());
                String dealStatus = "未成交";
                jcbdBdProvider.setDealStatus(dealStatus);
                NotifyProvider notifyProvider = new NotifyProvider();
                notifyProvider.setProviderId(jcbdBdProvider.getProviderId());
                notifyProvider.setName(jcbdBdProvider.getProviderName());
                notDealProviders.add(notifyProvider);
            }
        }
        biddingResultNotifyDto.setOperatorName(bidding.getOperatorName());
        biddingResultNotifyDto.setOperatorPhone(bidding.getOperatorPhone());
        biddingResultNotifyDto.setNotDealProviders(notDealProviders);
        jcbdBdProviderRepo.save(bidding.getJcbdBdProviders());
        return new BaseResponse(true, "竞价结果审核通过回调成功", biddingResultNotifyDto);
    }

    @Transactional
    @Override
    public BaseResponse finishBiddingOrderAudit(String biddingOrderNo) {
        JcbdOrder jcbdOrder = jcbdOrderRepo.findByBdOrderNo(biddingOrderNo);
        if (jcbdOrder == null) {
            log.error("审核通过，找不到对应的竞价订单，biddingOrderNo={}", biddingOrderNo);
            return new BaseResponse(true, "找不到对应的竞价订单");
        }
        jcbdOrder.setStatus(BiddingOrderStatus.ORDER_STATUS_TOBECONFIRMED.getCode());
        List<JcbdMallOrder> mallOrders = jcbdOrder.getJcbdMallOrders();
        if (mallOrders != null && !mallOrders.isEmpty()) {
            for (JcbdMallOrder mallOrder : mallOrders) {
                mallOrder.setStatus(BiddingOrderStatus.ORDER_STATUS_TOBECONFIRMED.getCode());
                mallOrder.setSupplierOrderStatus(BiddingOrderStatus.SUPPLIER_ORDER_STATUS_TOBECONFIRMED.getCode());
                log.info("审核完成，更新竞价商城订单状态成功,mallOrderNo={}", mallOrder.getOrderNo());
            }
        }
        JcbdBdProvider jcbdBdProvider = jcbdBdProviderRepo.findOne(jcbdOrder.getSupplierId());
        BiddingOrderConfirmNotifyDto biddingOrderConfirmNotifyDto = new BiddingOrderConfirmNotifyDto();
        biddingOrderConfirmNotifyDto.setOperatorName(jcbdOrder.getJcbdBidding().getOperatorName());
        biddingOrderConfirmNotifyDto.setOperatorPhone(jcbdOrder.getJcbdBidding().getOperatorPhone());
        biddingOrderConfirmNotifyDto.setBiddingNo(jcbdOrder.getJcbdBidding().getBiddingCode());
        NotifyProvider notifyProvider = new NotifyProvider();
        notifyProvider.setProviderId(jcbdBdProvider.getProviderId());
        notifyProvider.setName(jcbdOrder.getSupplierName());
        biddingOrderConfirmNotifyDto.setNotifyProvider(notifyProvider);
        return new BaseResponse(true, "更新成功", biddingOrderConfirmNotifyDto);
    }

    @Override
    @Transactional
    public BaseResponse biddingOrderAuditNoPass(String biddingOrderNo) {
        JcbdOrder jcbdOrder = jcbdOrderRepo.findByBdOrderNo(biddingOrderNo);
        if (jcbdOrder == null) {
            log.error("审核不通过，找不到对应的竞价订单，biddingOrderNo={}", biddingOrderNo);
            return new BaseResponse(true, "找不到对应的竞价订单");
        }
        jcbdOrder.setStatus(BiddingOrderStatus.ORDER_STATUS_AUDITFAILURE.getCode());
        List<JcbdMallOrder> mallOrders = jcbdOrder.getJcbdMallOrders();
        if (mallOrders != null && !mallOrders.isEmpty()) {
            for (JcbdMallOrder mallOrder : mallOrders) {
                mallOrder.setStatus(BiddingOrderStatus.ORDER_STATUS_AUDITFAILURE.getCode());
                log.info("订单审核不通过，更新竞价商城订单状态成功,mallOrderNo={}", mallOrder.getOrderNo());
            }
        }
        return new BaseResponse(true, "更新成功");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse saveBiddingTempOrderInfo(JcbdTempOrderDto jcbdTempOrderDto) {
        if (!Lang.isEmpty(jcbdTempOrderDto.getId())) {
            JcbdTempOrder jcbdTempOrderOlder = jcbdTempOrderRepo.findOne(jcbdTempOrderDto.getId());
            if (!Lang.isEmpty(jcbdTempOrderOlder)) {
                jcbdTempOrderRepo.delete(jcbdTempOrderOlder);
            }
        }
        String biddingNo = jcbdTempOrderDto.getBiddingNo();
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
        String storeId = "";
        for (JcbdBdProvider jcbdBdProvider : jcbdBdProviders) {
            if (jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_BIDDING_YES_AGAIN.getValue())) {
                storeId = jcbdBdProvider.getProviderId();
            }
        }
        JcbdTempOrder jcbdTempOrder = BeanMapper.map(jcbdTempOrderDto, JcbdTempOrder.class);
        List<JcbdTempGoods> tempGoodses = jcbdTempOrder.getJcbdTempGoodses();
        for (JcbdTempGoods goods : tempGoodses) {
            goods.setJcbdTempOrder(jcbdTempOrder);
        }
        jcbdTempOrder.setJcbdTempGoodses(tempGoodses);
        jcbdTempOrder.setJcbdBidding(jcbdBidding);
        jcbdTempOrder.setStoreId(storeId);
        jcbdTempOrderRepo.save(jcbdTempOrder);
        return new BaseResponse(true, "保存成功");

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public JcbdOrderDto saveJcbdOrder(JcbdOrderDto jcbdOrderDto) {
        JcbdOrder jcbdOrder = mapperFacade.map(jcbdOrderDto, JcbdOrder.class);
        for (JcbdMallOrder jcbdMallOrder : jcbdOrder.getJcbdMallOrders()) {
            jcbdMallOrder.setJcbdOrder(jcbdOrder);
        }
        JcbdOrder order = jcbdOrderRepo.save(jcbdOrder);
        return mapperFacade.map(order, JcbdOrderDto.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse deleteTempOrderById(List<String> tempOrderId) {
        for (String orderId : tempOrderId) {
            jcbdTempOrderRepo.delete(orderId);
        }
        return new BaseResponse(true, "删除成功");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteTempOrderByBiddingCode(String biddingCode) {
        List<JcbdTempOrder> jcbdTempOrders = jcbdTempOrderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(biddingCode, false);
        jcbdTempOrderRepo.delete(jcbdTempOrders);
    }

    @Override
    @Transactional
    public JcbdTempOrderDto modifyTempOrderById(String tempOrderId) {
        JcbdTempOrder jcbdTempOrder = jcbdTempOrderRepo.findOne(tempOrderId);
        if (jcbdTempOrder != null) {
            return BeanMapper.map(jcbdTempOrder, JcbdTempOrderDto.class);
        } else {
            return null;
        }

    }

    @Override
    @Transactional
    public BaseResponse finishBidding(String biddingNo) {
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (bidding == null) {
            return new BaseResponse(false, "竞价不存在");
        }
        bidding.setStatus(BiddingStatus.OM_BIDDING_END.getValue());
        List<JcbdBdProvider> providers = bidding.getJcbdBdProviders();
        List<JcbdBiddingGroupMember> jcbdBiddingGroupMembers = bidding.getJcbdBiddingGroupMembers();
        for (JcbdBiddingGroupMember jcbdBiddingGroupMember : jcbdBiddingGroupMembers) {
            jcbdBiddingGroupMember.setStatus(BiddingStatus.OM_BIDDING_END.getValue());
        }
        for (JcbdBdProvider provider : providers) {
            provider.setStatus(BiddingStatus.SUPPLIER_BIDDING_END.getValue());
        }
        try {
            jcbdBiddingRepo.save(bidding);
            jcbdBdProviderRepo.save(providers);
            jcbdBiddingGroupMemberRepo.save(jcbdBiddingGroupMembers);
        } catch (Exception e) {
            log.error("数据库操作异常", e);
            return new BaseResponse(false, "更新竞价状态失败");
        }
        return new BaseResponse(true, "操作成功");
    }

    @Override
    public String updateBiddingVoteNo(String biddingNo) {

        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        String voteNo = jcbdBidding.getVoteNo();
        if (Lang.isEmpty(voteNo)) {
            //获取当前年月
            SimpleDateFormat format = new SimpleDateFormat("YYYYMMdd");
            String dateTime = format.format(new Date());
            BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(VOTENO_CODE_REDIS_KEY);
            Long sequenceNo = boundValueOperations.increment(1);
            if (sequenceNo >= 1000) {
                boundValueOperations.set(0);
            }
            voteNo = "V" + dateTime + (sequenceNo + 1000 + "").substring(1);
        } else {
            voteNo = voteNo.substring(0, 1) + String.valueOf(Long.parseLong(voteNo.substring(1)) + 1);
        }
        jcbdBidding.setVoteNo(voteNo);
        jcbdBiddingRepo.saveAndFlush(jcbdBidding);
        return voteNo;
    }

    @Override
    public String updateBiddingGroupCheckNo(String biddingNo) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        String groupCheckNo = jcbdBidding.getGroupCheckNo();
        if (Lang.isEmpty(groupCheckNo)) {
            //获取当前年月
            SimpleDateFormat format = new SimpleDateFormat("YYYYMMdd");
            String dateTime = format.format(new Date());
            BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(GROUPCHECKNO_CODE_REDIS_KEY);
            Long sequenceNo = boundValueOperations.increment(1);
            if (sequenceNo >= 1000) {
                boundValueOperations.set(0);
            }
            groupCheckNo = "C" + dateTime + (sequenceNo + 1000 + "").substring(1);
        } else {
            groupCheckNo = groupCheckNo.substring(0, 1) + String.valueOf(Long.parseLong(groupCheckNo.substring(1)) + 1);
        }
        jcbdBidding.setGroupCheckNo(groupCheckNo);
        jcbdBiddingRepo.saveAndFlush(jcbdBidding);
        return groupCheckNo;
    }

    @Override
    public BaseResponse updateBiddingResultResult(String biddingNo, String remark) {
        if (Lang.isEmpty(biddingNo)) {
            return new BaseResponse(false, "提交竞价结果审核参数不正确");
        }
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (bidding == null) {
            return new BaseResponse(false, "不存在该竞价");
        }
        if (Lang.isEmpty(remark)) {
            remark = " ";
        }
        bidding.setBdCommitContent(remark);
        try {
            jcbdBiddingRepo.saveAndFlush(bidding);
        } catch (Exception e) {
            log.error("数据库操作异常", e);
            return new BaseResponse(false, "提交竞价结果审核失败");
        }
        return new BaseResponse(true, "操作成功");
    }

    @Override
    public void changeBiddingGroupMemberStatus(String biddingNo, String auditorId, String status, String email) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        JcbdBiddingGroupMember member;
        if (!Lang.isEmpty(auditorId)) {
            member = jcbdBiddingGroupMemberRepo.findByGroupMemberIdAndJcbdBiddingIdAndIsDelete(auditorId, jcbdBidding.getId(), false);
            member.setStatus(status);
        } else {
            member = jcbdBiddingGroupMemberRepo.findByEmailAndJcbdBiddingIdAndIsDelete(email, jcbdBidding.getId(), false);
            member.setStatus(status);
        }
        jcbdBiddingGroupMemberRepo.save(member);
    }

    @Override
    @Transactional
    public BaseResponse<Boolean> getVoteStatus(String biddingNo, String memberId) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (jcbdBidding == null) {
            return new BaseResponse<>(true, "竞价不存在", false);
        }
        if (StringUtils.equals(WayforConfirmBiddingResult.BIDDING_RADIO_VOTE.getValue(), jcbdBidding.getWayForConfirmBiddingResult())) {
            return new BaseResponse<>(true, "竞价是随机确定供应商", false);
        }
        if (StringUtils.equals(BiddingStatus.OM_VOTING.getValue(), jcbdBidding.getStatus())) {
            return new BaseResponse<>(true, "投票中", true);
        }
        JcbdBiddingGroupMember groupMember = jcbdBiddingGroupMemberRepo.findByGroupMemberIdAndJcbdBiddingIdAndIsDelete(memberId, jcbdBidding.getId(), false);
        if (groupMember == null) {
            groupMember = jcbdBiddingGroupMemberRepo.findByEmailAndJcbdBiddingIdAndIsDelete(memberId, jcbdBidding.getId(), false);
        }
        if(groupMember==null){
            return new BaseResponse<>(true, "未找到竞价小组成员信息", false);
        }
        if(groupMember==null){
            return new BaseResponse<>(true, "未找到竞价小组成员信息", false);
        }
        //todo groupMember NullPointerException
        List<VoteDetail> voteDetails = voteDetailRepo.findByBusinessIdentifyingAndVoteMemberIdAndIsDelete(biddingNo, groupMember.getId(), false);
        if (voteDetails != null && !voteDetails.isEmpty()) {
            return new BaseResponse<>(true, "已存在投票记录", true);
        }
        return new BaseResponse<>(true, "竞价结果决定方式为投票，不需要投票", false);
    }


    @Override
    public BaseResponse<String> getBiddingNo() {
        String biddingNoRedisKey = "bidding-service:biddingNo:key";
        String biddingNoRedisTimeKey = "bidding-service:biddingNo:timeKey";
        String createBiddingNoSequenceSql = "create sequence mall_bidding_seq_newsId increment by 1 start with 1";
        String dropBiddingNoSequenceSql = "drop sequence mall_bidding_seq_newsId";
        String queryBiddingNoSql = "select mall_bidding_seq_newsId.nextval from dual";
        String monthSequence = null;

        redisLockUtil.lock(biddingNoRedisKey);
        try {
            Calendar calendar = Calendar.getInstance();
            BoundValueOperations biddingTimeBoundValueOperations = redisTemplate.boundValueOps(biddingNoRedisTimeKey);
            Object month = biddingTimeBoundValueOperations.get();
            if (month != null) {
                if (calendar.get(Calendar.MONTH) != (Integer) month) {
                    try {
                        jdbcTemplate.execute(dropBiddingNoSequenceSql);
                        jdbcTemplate.execute(createBiddingNoSequenceSql);
                    } catch (Exception e) {
                        log.error("重置竞价序列号出错", e);
                    }
                }
                biddingTimeBoundValueOperations.set(calendar.get(Calendar.MONTH));
            }
            try {
                monthSequence = jdbcTemplate.queryForObject(queryBiddingNoSql, String.class);
//                return new BaseResponse<>(true, "获取竞价编号成功", biddingNo);
            } catch (Exception e) {
                log.error("查询竞价序列出错", e);
                log.info("创建竞价序列");
                jdbcTemplate.execute(createBiddingNoSequenceSql);
                monthSequence = jdbcTemplate.queryForObject(queryBiddingNoSql, String.class);
//                return new BaseResponse<>(true, "获取竞价编号成功", monthSequence);
            }
            StringBuilder biddingBuilder = new StringBuilder("B").append(new SimpleDateFormat("YYYYMM").format(calendar.getTime()));
            if (monthSequence.length() < 4) {
                for (int t = 0; t < 4 - monthSequence.length(); t++) {
                    biddingBuilder.append("0");
                }
            }
            biddingBuilder.append(monthSequence);
            return new BaseResponse<>(true, "获取竞价编号成功", biddingBuilder.toString());
        } catch (Exception e) {
            log.error("生成竞价编号失败", e);
        } finally {
            redisLockUtil.unlock(biddingNoRedisKey);
        }
        return null;
    }

    /**
     * 根据sku列表查询临时商品
     *
     * @param skus
     * @return
     */
    @Override
    public Boolean findTempOrderBySkus(List<String> skus) {
        List<JcbdTempGoods> jcbdTempGoods = jcbdTempGoodsRepo.findBySkuIn(skus);
        if (jcbdTempGoods.size() > 0) {
            return true;
        }
        return false;
    }


//    @Override
//    public String getOperatorName(String operatorId) {
//        JcbdBiddingGroupMember jcbdBiddingGroupMember = jcbdBiddingGroupMemberRepo.findFirstByGroupMemberIdAndIsDelete(operatorId, false);
//        if (jcbdBiddingGroupMember != null) {
//            if (!Lang.isEmpty(jcbdBiddingGroupMember.getName())) {
//                return jcbdBiddingGroupMember.getName();
//            } else if (!Lang.isEmpty(jcbdBiddingGroupMember.getEmail())) {
//                return jcbdBiddingGroupMember.getEmail();
//            }
//        } else {
//            //供应商登录时 获取供应商及联系人信息
//            List<SupplierContactDto> supplierContact =  supplierLoginApi.findByDUserIdAndIsDelete(operatorId);
//            if(!Lang.isEmpty(supplierContact)){
//                if(!Lang.isEmpty(supplierContact.get(0).getdUserName()) && !Lang.isEmpty(supplierContact.get(0).getdVendorName())){
//                    return new StringBuffer(supplierContact.get(0).getdVendorName()).append("-").append(supplierContact.get(0).getdUserName()).toString();
//                }
//            }
//            JcbdBdProvider jcbdBdProvider = jcbdBdProviderRepo.findFirstByProviderIdAndIsDelete(operatorId, false);
//            if (jcbdBdProvider != null) {
//                if (jcbdBdProvider.getContactsId() != null) {
//                    JcbdStoreContacts storeContacts = jcbdStoreContactsRepo.findOne(jcbdBdProvider.getContactsId());
//                    if (storeContacts != null) {
//                        if (!Lang.isEmpty(storeContacts.getName())) {
//                            return storeContacts.getName();
//                        } else if (!Lang.isEmpty(storeContacts.getEmail())) {
//                            return storeContacts.getEmail();
//                        }
//                    }
//                } else {
//                    return jcbdBdProvider.getProviderName();
//                }
//            }
//        }
//        return operatorId;
//    }

    @Autowired
    JcbdStoreContactsRepo jcbdStoreContactsRepo;

    @Override
    public BaseResponse changeGroupMemberConfomanceCheck(String biddingNo, String auditorId, String email) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        JcbdBiddingGroupMember member;
        if (!Lang.isEmpty(auditorId)) {
            member = jcbdBiddingGroupMemberRepo.findByGroupMemberIdAndJcbdBiddingIdAndIsDelete(auditorId, jcbdBidding.getId(), false);
            member.setConfomanceCheck(true);
        } else {
            member = jcbdBiddingGroupMemberRepo.findByEmailAndJcbdBiddingIdAndIsDelete(email, jcbdBidding.getId(), false);
            member.setConfomanceCheck(true);
        }
        try {
            jcbdBiddingGroupMemberRepo.save(member);
        } catch (Exception e) {
            log.error("数据库操作错误");
            return new BaseResponse("更改竞价状态失败");
        }
        return new BaseResponse(true);
    }

    @Override
    public BaseResponse cancelGroupMemberConfomanceCheck(String biddingNo, String auditorId, String email) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        JcbdBiddingGroupMember member;
        if (!Lang.isEmpty(auditorId)) {
            member = jcbdBiddingGroupMemberRepo.findByGroupMemberIdAndJcbdBiddingIdAndIsDelete(auditorId, jcbdBidding.getId(), false);
            member.setConfomanceCheck(false);
        } else {
            member = jcbdBiddingGroupMemberRepo.findByEmailAndJcbdBiddingIdAndIsDelete(email, jcbdBidding.getId(), false);
            member.setConfomanceCheck(false);
        }
        try {
            jcbdBiddingGroupMemberRepo.save(member);
        } catch (Exception e) {
            log.error("数据库操作错误");
            return new BaseResponse("更改竞价状态失败");
        }
        return new BaseResponse(true);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse changeBiddingOrderState(List<String> orderNos, String state) {
        log.info("更改竞价订单状态-orderNo:{},status:{}",JSON.toJSONString(orderNos),state);
        orderNos.parallelStream().forEach(orderNo->{
           jcbdMallOrderRepo.updateOrderStatus(orderNo,state);
        });
        return new BaseResponse(true,"更改竞价订单状态成功");
    }

    @Override
    public JcbdBiddingGroupMemberDto findByMemberId(String operatorId) {
        JcbdBiddingGroupMember jcbdBiddingGroupMember = jcbdBiddingGroupMemberRepo.findFirstByGroupMemberIdAndIsDelete(operatorId, false);
        JcbdBiddingGroupMemberDto jcbdBiddingGroupMemberDto=null;
        if(!Lang.isEmpty(jcbdBiddingGroupMember)){
            jcbdBiddingGroupMemberDto = BeanMapper.map(jcbdBiddingGroupMember, JcbdBiddingGroupMemberDto.class);
        }
        return jcbdBiddingGroupMemberDto;
    }
}

