package bidding.service.service.impl.query;

import bidding.model.po.*;
import bidding.model.repository.*;
import bidding.model.specification.BiddingDemandSpecification;
import bidding.model.specification.BiddingProviderSpecification;
import bidding.service.service.query.BiddingMainQueryService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import definition.BiddingRole;
import definition.BiddingStatus;
import dto.*;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import query.BiddingDemandListQuery;
import sinomall.global.common.response.BaseResponse;
import utils.Lang;
import utils.data.BeanMapper;

import javax.persistence.criteria.Predicate;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * 竞价主管查询业务逻辑层
 *
 * @author bande
 * @date 2018/3/19
 */
@Slf4j
@Service
public class BiddingMainQueryServiceImpl implements BiddingMainQueryService {

    @Autowired
    JcbdBiddingGroupMemberRepo jcbdBiddingGroupMemberRepo;
    @Autowired
    JcbdRoleOperationRepo jcbdRoleOperationRepo;
    @Autowired
    JcbdBdProviderRepo jcbdBdProviderRepo;

    @Autowired
    JcbdBiddingRepo jcbdBiddingRepo;
    @Autowired
    JcbdBiddingDemandRepo jcbdBiddingDemandRepo;
    @Autowired
    MapperFacade mapperFacade;

    @Autowired
    JcbdTempOrderRepo jcbdTempOrderRepo;


    //已审查
    public static final String IS_CHECKED = "1";
    //未审查
    public static final String IS_CHECKEING = "0";


    @Override
    public BaseResponse<List<BiddingPurchaseGoodsDTO>> getPurchaseGoods(String biddingNo) {
        return null;
    }

    @Override
    public Page<JcbdBiddingQueryResultDto> operatorFindJcbdBiddingByParams(JcbdBiddingQueryDto jcbdBiddingQueryDto) {
        log.info("查询参数{}", JSONObject.toJSONString(jcbdBiddingQueryDto));
        Specification specification = getSpecificationForOperator(jcbdBiddingQueryDto);
        Page<JcbdBiddingGroupMember> biddingGroupMembers = jcbdBiddingGroupMemberRepo.findAll(specification, jcbdBiddingQueryDto.getPageable());
        List<JcbdBiddingGroupMember> jcbdBiddingGroupMemberList = biddingGroupMembers.getContent();
        //取出内容，重新封装成满足前台显示的dto
        List<JcbdBiddingQueryResultDto> jcbdBiddingQueryResultDtoList = new ArrayList<>();
        if (!Lang.isEmpty(jcbdBiddingGroupMemberList)) {
            for (JcbdBiddingGroupMember groupMember : jcbdBiddingGroupMemberList) {
                JcbdBiddingQueryResultDto jcbdBiddingQueryResultDto = new JcbdBiddingQueryResultDto();
                jcbdBiddingQueryResultDto.setJcbdBiddingNo(groupMember.getJcbdBidding().getBiddingCode());
                jcbdBiddingQueryResultDto.setDescribe(groupMember.getJcbdBidding().getBiddingDescription());
                jcbdBiddingQueryResultDto.setStartTime(groupMember.getJcbdBidding().getTimeStartResponse());
                jcbdBiddingQueryResultDto.setEndTime(groupMember.getJcbdBidding().getTimeEndResponse());
                jcbdBiddingQueryResultDto.setInvitationSupplierNum(groupMember.getJcbdBidding().getProviderInviteNum());
                jcbdBiddingQueryResultDto.setResponseSupplierNum(groupMember.getJcbdBidding().getProviderResponseNum());
                jcbdBiddingQueryResultDto.setState(groupMember.getJcbdBidding().getStatus());
                BiddingStatus biddingStatus = BiddingStatus.biddingStatusMap.get(groupMember.getJcbdBidding().getStatus());
                jcbdBiddingQueryResultDto.setStateName(biddingStatus.getDescription());
                //查找该用户可以对该记录进行的操作
                List<JcbdRoleOperationDto> jcbdRoleOperationDtoList = new ArrayList<>();
                //如果该用户是这条竞价的经办人则查找两种身份下拥有的操作

                if (groupMember.getJcbdBidding().getOperatorId().equals(jcbdBiddingQueryDto.getUserId())) {
                    List<JcbdRoleOperation> operationDtoForOperator = jcbdRoleOperationRepo.findByRoleCodeAndStatusOrderByShowIndex(BiddingRole.BIDDING_OPERATOR, groupMember.getStatus());
                    List<JcbdRoleOperation> operationDtosForGroupMember = jcbdRoleOperationRepo.findByRoleCodeAndStatusOrderByShowIndex(BiddingRole.BIDDING_GROUP_MEMBER, groupMember.getStatus());
                    if (!Lang.isEmpty(operationDtoForOperator)) {
                        jcbdRoleOperationDtoList.addAll(BeanMapper.mapList(operationDtoForOperator, JcbdRoleOperation.class, JcbdRoleOperationDto.class));
                    }
                    if (!Lang.isEmpty(operationDtosForGroupMember)) {
                        jcbdRoleOperationDtoList.addAll(BeanMapper.mapList(operationDtosForGroupMember, JcbdRoleOperation.class, JcbdRoleOperationDto.class));
                    }
                } else {
                    //否则只查找作为竞价小组成员才有的操作
                    List<JcbdRoleOperation> operationDtosForGroupMember = jcbdRoleOperationRepo.findByRoleCodeAndStatusOrderByShowIndex(BiddingRole.BIDDING_GROUP_MEMBER, groupMember.getStatus());
                    if (!Lang.isEmpty(operationDtosForGroupMember)) {
                        jcbdRoleOperationDtoList.addAll(BeanMapper.mapList(operationDtosForGroupMember, JcbdRoleOperation.class, JcbdRoleOperationDto.class));
                    }
                }
                //将操作链接拼接一个竞价编号参数
                for (JcbdRoleOperationDto jcbdRoleOperationDto : jcbdRoleOperationDtoList) {
                    String url = jcbdRoleOperationDto.getUrl();
                    if (url != null && !"".equals(url)) {
                        url = url + "?biddingNo=" + jcbdBiddingQueryResultDto.getJcbdBiddingNo();
                    }
                    jcbdRoleOperationDto.setUrl(url);
                }
                jcbdBiddingQueryResultDto.setOperations(jcbdRoleOperationDtoList);
                jcbdBiddingQueryResultDtoList.add(jcbdBiddingQueryResultDto);
            }
        }

        return new PageImpl(jcbdBiddingQueryResultDtoList, new PageRequest(biddingGroupMembers.getNumber(), biddingGroupMembers.getSize(), biddingGroupMembers.getSort()), biddingGroupMembers.getTotalElements());

    }

    @Override
    public Page<JcbdBiddingQueryResultDto> supplierFindJcbdBiddingByParams(JcbdBiddingQueryDto jcbdBiddingQueryDto) {
        log.info("查询参数{}", JSONObject.toJSONString(jcbdBiddingQueryDto));
        Specification specification = getSpecificationForSupplier(jcbdBiddingQueryDto);
        Page<JcbdBdProvider> jcbdBdProviders = jcbdBdProviderRepo.findAll(specification, jcbdBiddingQueryDto.getPageable());
        List<JcbdBdProvider> jcbdBdProviderList = jcbdBdProviders.getContent();
        //取出内容，重新封装成满足前台显示的dto
        List<JcbdBiddingQueryResultDto> jcbdBiddingQueryResultDtoList = new ArrayList<>();
        if (!Lang.isEmpty(jcbdBdProviderList)) {
            for (JcbdBdProvider bdProvider : jcbdBdProviderList) {
                JcbdBiddingQueryResultDto jcbdBiddingQueryResultDto = new JcbdBiddingQueryResultDto();
                jcbdBiddingQueryResultDto.setJcbdBiddingNo(bdProvider.getJcbdBidding().getBiddingCode());
                jcbdBiddingQueryResultDto.setDescribe(bdProvider.getJcbdBidding().getBiddingDescription());
                jcbdBiddingQueryResultDto.setStartTime(bdProvider.getJcbdBidding().getTimeStartResponse());
                jcbdBiddingQueryResultDto.setEndTime(bdProvider.getJcbdBidding().getTimeEndResponse());
                List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBiddingDemandRepo.findByJcbdBiddingIdAndIsDelete(bdProvider.getJcbdBidding().getId(), false);
                if (!Lang.isEmpty(jcbdBiddingDemands)) {
                    jcbdBiddingQueryResultDto.setNeedNum(jcbdBiddingDemands.size());
                }else {
                    jcbdBiddingQueryResultDto.setNeedNum(0);
                }
                jcbdBiddingQueryResultDto.setState(bdProvider.getStatus());
                BiddingStatus biddingStatus = BiddingStatus.biddingStatusMap.get(bdProvider.getStatus());
                if(biddingStatus!=null) {
                    jcbdBiddingQueryResultDto.setStateName(biddingStatus.getDescription());
                }
                //查找该用户可以对该记录进行的操作
                List<JcbdRoleOperationDto> jcbdRoleOperationDtoList = new ArrayList<>();
                List<JcbdRoleOperation> operationDtos = jcbdRoleOperationRepo.findByRoleCodeAndStatusOrderByShowIndex(BiddingRole.BIDDING_SUPPLIER, bdProvider.getStatus());
                if (!Lang.isEmpty(operationDtos)) {
                    jcbdRoleOperationDtoList.addAll(BeanMapper.mapList(operationDtos, JcbdRoleOperation.class, JcbdRoleOperationDto.class));
                }
                //将操作链接拼接一个竞价编号参数
                for (JcbdRoleOperationDto jcbdRoleOperationDto : jcbdRoleOperationDtoList) {
                    String url = jcbdRoleOperationDto.getUrl();
                    if (url != null && !"".equals(url)) {
                        url = url + "?biddingNo=" + jcbdBiddingQueryResultDto.getJcbdBiddingNo();
                    }
                    jcbdRoleOperationDto.setUrl(url);
                }
                jcbdBiddingQueryResultDto.setOperations(jcbdRoleOperationDtoList);
                jcbdBiddingQueryResultDtoList.add(jcbdBiddingQueryResultDto);
            }
        }
        return new PageImpl(jcbdBiddingQueryResultDtoList, new PageRequest(jcbdBdProviders.getNumber(), jcbdBdProviders.getSize(), jcbdBdProviders.getSort()), jcbdBdProviders.getTotalElements());
    }

    /**
     * 竞价需求查询（竞价编号）
     * @param biddingDemandListQuery
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public BaseResponse<Page<BiddingDemandQueryDto>> getBiddingDemandList(BiddingDemandListQuery biddingDemandListQuery) {

        BaseResponse<Page<BiddingDemandQueryDto>> baseResponse;
        log.info("经办人编辑审核-竞价需求查询-请求参数 {}", JSON.toJSONString(biddingDemandListQuery));
        // 请求参数校验
        if(Lang.isEmpty(biddingDemandListQuery)) {
            baseResponse = new BaseResponse<>(false, "经办人编辑审核-竞价需求查询-请求参数不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if(Lang.isEmpty(biddingDemandListQuery.getBiddingCode())) {
            baseResponse = new BaseResponse<>(false, "经办人编辑审核-竞价需求查询-竞价编号不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if(Lang.isEmpty(biddingDemandListQuery.getPageable())) {
            baseResponse = new BaseResponse<>(false, "经办人编辑审核-竞价需求查询-分页信息不能为空");
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }
        try {
            JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingDemandListQuery.getBiddingCode(),false);
            if(Lang.isEmpty(jcbdBidding)) {
                baseResponse = new BaseResponse<>(false, "竞价不存在");
                log.error("竞价{}不存在", biddingDemandListQuery.getBiddingCode());
                return baseResponse;
            }

            //查询竞价需求
            JcbdBiddingDemand biddingDemand = new JcbdBiddingDemand();
            biddingDemand.setJcbdBidding(jcbdBidding);
            Page<JcbdBiddingDemand> biddingDemandPage = jcbdBiddingDemandRepo.findAll(new BiddingDemandSpecification<>(biddingDemand), biddingDemandListQuery.getPageable());
            List<JcbdBiddingDemand> biddingDemands = biddingDemandPage.getContent();
            List<BiddingDemandQueryDto> list = new ArrayList<>();
            if(Lang.isEmpty(biddingDemands)) {
                baseResponse = new BaseResponse<>(true, "查询成功",new PageImpl<>(list, biddingDemandListQuery.getPageable(), list.size()));
                return baseResponse;
            }
            for (JcbdBiddingDemand demand:biddingDemands) {
                BiddingDemandQueryDto bdqd = mapperFacade.map(demand, BiddingDemandQueryDto.class);
                //需求符合性审查（个）
                if(!Lang.isEmpty(demand.getJcbdBiddingConformances())) {
                    bdqd.setComplianceAuditNum(demand.getJcbdBiddingConformances().size());
                }

                //需求地址列表
                if(!Lang.isEmpty(demand.getJcbdDemandAddressNums())) {
                    String addr ="";
                    for (int i = 0; i <demand.getJcbdDemandAddressNums().size() ; i++) {
                        if(i != demand.getJcbdDemandAddressNums().size() -1) {
                            addr = addr +demand.getJcbdDemandAddressNums().get(i).getProvinceName()
                                    +demand.getJcbdDemandAddressNums().get(i).getCityName()+",";
                        }else {
                            addr = addr +demand.getJcbdDemandAddressNums().get(i).getProvinceName()
                                    +demand.getJcbdDemandAddressNums().get(i).getCityName();
                        }
                    }

                    bdqd.setDemandAddress(addr);
                }
                list.add(bdqd);
            }

            Page<BiddingDemandQueryDto> BiddingDemandQueryDtoPage = new PageImpl<>(list, biddingDemandListQuery.getPageable(), list.size());

            log.info("获取竞价需求列表 DATA : {}", JSON.toJSONString(BiddingDemandQueryDtoPage));

            return new BaseResponse<>(true, "操作成功", BiddingDemandQueryDtoPage);

        } catch (Exception e) {
            log.error("经办人编辑审核-竞价需求查询异常, 竞价编号 {} ", biddingDemandListQuery.getBiddingCode(), e);
            return new BaseResponse<>(false, "竞价需求列表查询异常", null);
        }
    }


    /**
     * 竞价需求查询(详细)
     * @param demandCode
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public BaseResponse<JcbdBiddingDemandDto> getBiddingDemandInfo(String demandCode) {

        if(Lang.isEmpty(demandCode)) {
            return new BaseResponse(false, "查询参数为空");
        }
        JcbdBiddingDemand jcbdBiddingDemand = jcbdBiddingDemandRepo.findByDemandCodeAndIsDelete(demandCode, false);
        JcbdBiddingDemandDto jcbdBiddingDemandDto = null;
        if(!Lang.isEmpty(jcbdBiddingDemand)) {
            jcbdBiddingDemandDto = mapperFacade.map(jcbdBiddingDemand, JcbdBiddingDemandDto.class);
        }
        return new BaseResponse(true, "查询成功",jcbdBiddingDemandDto);
    }

    /**
     * 竞价概要信息查询
     * @param biddingCode
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public BaseResponse<JcbdBiddingDto> getBiddingByBiddingCode(String biddingCode) {

        if(Lang.isEmpty(biddingCode)) {
            return new BaseResponse(false, "查询参数为空");
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingCode, false);
        JcbdBiddingDto jcbdBiddingDto = null;
        if(!Lang.isEmpty(jcbdBidding)) {
            jcbdBiddingDto = mapperFacade.map(jcbdBidding, JcbdBiddingDto.class);
        }
        return new BaseResponse(true, "查询成功",jcbdBiddingDto);

    }

    private Specification getSpecificationForOperator(JcbdBiddingQueryDto jcbdBiddingQueryDto) {
        return (Specification<JcbdBiddingGroupMember>) (root, criteriaQuery, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            //默认查询未删除
            predicates.add(criteriaBuilder.equal(root.get("isDelete"), false));

            if (jcbdBiddingQueryDto.getDescribe() != null) {
                predicates.add(criteriaBuilder.like(root.get("jcbdBidding").get("biddingDescription"), "%" + jcbdBiddingQueryDto.getDescribe() + "%"));
            }
            if (jcbdBiddingQueryDto.getStartTime() != null) {

                predicates.add(criteriaBuilder.greaterThanOrEqualTo(root.get("jcbdBidding").get("timeStartResponse"), jcbdBiddingQueryDto.getStartTime()));
            }

            if (jcbdBiddingQueryDto.getEndTime() != null) {
                predicates.add(criteriaBuilder.lessThanOrEqualTo(root.get("jcbdBidding").get("timeEndResponse"), jcbdBiddingQueryDto.getEndTime()));
            }

            if (jcbdBiddingQueryDto.getJcbdBiddingNo() != null) {
                predicates.add(criteriaBuilder.equal(root.get("jcbdBidding").get("biddingCode"), jcbdBiddingQueryDto.getJcbdBiddingNo()));
            }

            if (!Lang.isEmpty(jcbdBiddingQueryDto.getStates())) {
                predicates.add(root.get("status").in(jcbdBiddingQueryDto.getStates()));
            }

            if (jcbdBiddingQueryDto.getUserId() != null) {
                predicates.add(criteriaBuilder.equal(root.get("groupMemberId"), jcbdBiddingQueryDto.getUserId()));
            }
            if (jcbdBiddingQueryDto.getEmail() != null) {
                predicates.add(criteriaBuilder.equal(root.get("email"), jcbdBiddingQueryDto.getEmail()));
            }

            //更多条件，依次累加
            return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
        };
    }

    private Specification getSpecificationForSupplier(JcbdBiddingQueryDto jcbdBiddingQueryDto) {
        return (Specification<JcbdBdProvider>) (root, criteriaQuery, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();
            //默认查询未删除
            predicates.add(criteriaBuilder.equal(root.get("isDelete"), false));
            predicates.add(criteriaBuilder.isNotNull(root.get("status")));
            //默认查询可以查看的竞价
            predicates.add(criteriaBuilder.equal(root.get("publish"), true));

            if (jcbdBiddingQueryDto.getDescribe() != null) {
                predicates.add(criteriaBuilder.like(root.get("jcbdBidding").get("biddingDescription"), "%" + jcbdBiddingQueryDto.getDescribe() + "%"));
            }

            if (jcbdBiddingQueryDto.getStartTime() != null) {
                predicates.add(criteriaBuilder.greaterThanOrEqualTo(root.get("jcbdBidding").get("timeStartResponse"), jcbdBiddingQueryDto.getStartTime()));
            }
            if (jcbdBiddingQueryDto.getStartTime2() != null) {
                predicates.add(criteriaBuilder.lessThanOrEqualTo(root.get("jcbdBidding").get("timeStartResponse"), jcbdBiddingQueryDto.getStartTime2()));
            }

            if (jcbdBiddingQueryDto.getEndTime() != null) {
                predicates.add(criteriaBuilder.greaterThanOrEqualTo(root.get("jcbdBidding").get("timeEndResponse"), jcbdBiddingQueryDto.getEndTime()));
            }

            if (jcbdBiddingQueryDto.getEndTime2() != null) {
                predicates.add(criteriaBuilder.lessThanOrEqualTo(root.get("jcbdBidding").get("timeEndResponse"), jcbdBiddingQueryDto.getStartTime2()));

            }

            if (jcbdBiddingQueryDto.getJcbdBiddingNo() != null) {
                predicates.add(criteriaBuilder.equal(root.get("jcbdBidding").get("biddingCode"), jcbdBiddingQueryDto.getJcbdBiddingNo()));
            }

            if (!Lang.isEmpty(jcbdBiddingQueryDto.getStates())) {
                predicates.add(root.get("status").in(jcbdBiddingQueryDto.getStates()));
            }

            if (jcbdBiddingQueryDto.getProviderId() != null) {
                predicates.add(criteriaBuilder.equal(root.get("providerId"), jcbdBiddingQueryDto.getProviderId()));
            }

            //更多条件，依次累加
            return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
        };
    }

    @Override
    @Transactional
    public JcbdBiddingDemandDto getBiddingDemandFindOne(JcbdBiddingDemandDto jcbdBiddingDemandDto) {
        // 注释原因...........没有必要拷贝，不是多条件，只需要根据id查即可
        //JcbdBiddingDemand demandQuery = mapperFacade.map(jcbdBiddingDemandDto, JcbdBiddingDemand.class);
        //JcbdBiddingDemand jcbdBiddingDemand = jcbdBiddingDemandRepo.findOne(new BiddingDemandSpecification<>(demandQuery));
        JcbdBiddingDemand jcbdBiddingDemand = jcbdBiddingDemandRepo.findByIdAndIsDelete(jcbdBiddingDemandDto.getId(),false);
        return mapperFacade.map(jcbdBiddingDemand, JcbdBiddingDemandDto.class);
    }

    @Override
    @Transactional
    public List<ProcureInformationDto> getBiddingDemandSku(String biddingCode) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingCode, false);
        List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
        String dealProviderId="";
        if(!Lang.isEmpty(jcbdBdProviders)){
            for(JcbdBdProvider jcbdBdProvider:jcbdBdProviders){
                if(jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_BIDDING_YES_AGAIN.getValue())){
                    dealProviderId=jcbdBdProvider.getId();
                }
            }
        }
//        jcbdBdProviders.stream().anyMatch(n-> {
//            n.getStatus().equals(BiddingStatus.SUPPLIER_BIDDING_YES.getValue());
//            return true;
//        })

        List<ProcureInformationDto> procureInformationDtos=new ArrayList<>();
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        for(JcbdBiddingDemand jcbdBiddingDemand:jcbdBiddingDemands){

            List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses = jcbdBiddingDemand.getJcbdBiddingDemandResponses();
            for(JcbdBiddingDemandResponse jcbdBiddingDemandResponse:jcbdBiddingDemandResponses){
                if(jcbdBiddingDemandResponse.getResponseProviderId().equals(dealProviderId)){
                    ProcureInformationDto procureInformationDto=new ProcureInformationDto();
                    procureInformationDto.setDemandId(jcbdBiddingDemand.getId());
                    procureInformationDto.setDemandCode(jcbdBiddingDemand.getDemandCode());
                    procureInformationDto.setSku(jcbdBiddingDemandResponse.getSku());
                    procureInformationDto.setPrice(jcbdBiddingDemandResponse.getPrice());
                    //设置最高采购数量，以便前端需要进行控制
                    procureInformationDto.setMaxPurchaseAmount(jcbdBiddingDemand.getMaxPurchaseAmount());
                    procureInformationDtos.add(procureInformationDto);

                }
            }
        }
        return procureInformationDtos;
    }

    @Override
    @Transactional
    public JcbdBiddingDto queryBidding(String biddingNo) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo,false);
        return BeanMapper.map(jcbdBidding,JcbdBiddingDto.class);
    }

    @Override
    @Transactional(readOnly = true)
    public List<TempBatchDto> getTempBatchList(String biddingNo) {
        List<JcbdTempOrder> jcbdTempOrders = jcbdTempOrderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(biddingNo, false);
        List<TempBatchDto> tempBatchDtos=new ArrayList<>();
        DecimalFormat df = new DecimalFormat("#0.00");
        for(JcbdTempOrder jcbdTempOrder:jcbdTempOrders){
            TempBatchDto tempBatchDto=new TempBatchDto();
            tempBatchDto.setTempOrderId(jcbdTempOrder.getId());
            List<JcbdTempGoods> jcbdTempGoodses = jcbdTempOrder.getJcbdTempGoodses();
            List<TempBatchListDto> tempBatchListDtos=new ArrayList<>();
            for(JcbdTempGoods jcbdTempGoods:jcbdTempGoodses){
                TempBatchListDto tempBatchListDto=new TempBatchListDto();
                tempBatchListDto.setSku(jcbdTempGoods.getSku());
                tempBatchListDto.setNum(jcbdTempGoods.getNum());
                tempBatchListDto.setPrice(jcbdTempGoods.getPrice());
                if (!Lang.isEmpty(jcbdTempGoods.getNum()) && !Lang.isEmpty(jcbdTempGoods.getPrice())){
                    String totalPrice = df.format(new BigDecimal(jcbdTempGoods.getNum()).multiply(jcbdTempGoods.getPrice()).setScale(2,BigDecimal.ROUND_UP));
                    tempBatchListDto.setTotalPrice(new BigDecimal(totalPrice));
                }
                tempBatchListDto.setAddressId(jcbdTempOrder.getMemberAddressId());
                tempBatchListDtos.add(tempBatchListDto);
            }
            tempBatchDto.setTempBatchLists(tempBatchListDtos);
            tempBatchDtos.add(tempBatchDto);
        }
        return tempBatchDtos;
    }

    @Override
    @Transactional(readOnly = true)
    public JcbdBdProviderDto getTransactionSupplier(String biddingNo) {
        if (Lang.isEmpty(biddingNo)){
            log.error("查询竞价已成交的供应商传入参数为空");
            return null;
        }
        try {
            JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo,false);
            if (Lang.isEmpty(jcbdBidding)) {
                log.error("竞价{}不存在", biddingNo);
                return null;
            }
            JcbdBdProvider provider = new JcbdBdProvider();
            provider.setJcbdBidding(jcbdBidding);
            provider.setStatus(BiddingStatus.SUPPLIER_BIDDING_YES.getValue());
            List<JcbdBdProvider> providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_BIDDING_YES_AGAIN.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_SKU_AUDIT_WAIT.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_SKU_MAKEUP_WAIT.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_SKU_AUDIT_NOT.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty((providers))) {
                return null;
            }
            return BeanMapper.map(providers.get(0),JcbdBdProviderDto.class);
        } catch (Exception e) {
            log.error("竞价编号为{}查询最终成交供应商失败", biddingNo ,e);
            return null;
        }
    }

    @Override
    @Transactional(readOnly = true)
    public JcbdBdProviderDto getProvider(String biddingNo, String supplierId) {
        if (Lang.isEmpty(biddingNo)){
            log.error("查询竞价已成交的供应商传入参数为空");
            return null;
        }
        try {
            JcbdBdProvider provider = jcbdBdProviderRepo.findByJcbdBiddingBiddingCodeAndProviderIdAndIsDelete(biddingNo, supplierId, false);
            if (Lang.isEmpty(provider)) {
                log.error("供应商{}下的竞价{}不存在", supplierId, biddingNo);
                return null;
            }
            return BeanMapper.map(provider,JcbdBdProviderDto.class);
        } catch (Exception e) {
            log.error("供应商{}下的竞价{}查询失败",supplierId, biddingNo ,e);
            return null;
        }
    }

    @Override
    @Transactional
    public List<JcbdTempOrderDto> findAllTempOrderByBiddingNo(String biddingNo) {
        List<JcbdTempOrder> jcbdTempOrderList = jcbdTempOrderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(biddingNo, false);
        if(!Lang.isEmpty(jcbdTempOrderList)){
            return BeanMapper.mapList(jcbdTempOrderList,JcbdTempOrder.class,JcbdTempOrderDto.class);
        }
        return null;
    }

    @Override
    @Transactional(readOnly = true)
    public BaseResponse<JcbdBiddingDto> findOne(String biddingId) {
        if (Lang.isEmpty(biddingId)){
            return new BaseResponse("参数错误");
        }
        JcbdBidding bidding = jcbdBiddingRepo.findByIdAndIsDelete(biddingId, false);
        if (Lang.isEmpty(bidding)){
            return new BaseResponse("无此竞价");
        }
        return new BaseResponse(true,"",mapperFacade.map(bidding,JcbdBiddingDto.class));
    }

    @Override
    @Transactional
    public BigDecimal getBiddingGeneralBudget(String biddingNo) {
        if(StringUtils.isEmpty(biddingNo)){
            return null;
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
        BigDecimal generalBudget=new BigDecimal("0");
        for(JcbdBdProvider jcbdBdProvider:jcbdBdProviders){
            if(jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue())){
                for(JcbdBiddingDemand jcbdBiddingDemand:jcbdBiddingDemands){
                    Long maxPurchaseAmount = jcbdBiddingDemand.getMaxPurchaseAmount();
                    BigDecimal price=new BigDecimal("0");
                    List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses = jcbdBiddingDemand.getJcbdBiddingDemandResponses();
                    for(JcbdBiddingDemandResponse response:jcbdBiddingDemandResponses){
                        if(response.getResponseProviderId().equals(jcbdBdProvider.getId())){
                             price = response.getPrice();
                        }
                    }
                    BigDecimal budget = price.multiply(new BigDecimal(maxPurchaseAmount));
                    generalBudget=generalBudget.add(budget);
                }
            }
        }
        return generalBudget;
    }
}
