package bidding.service.service.impl.query;

import bidding.model.po.*;
import bidding.model.repository.*;
import bidding.model.specification.*;
import bidding.service.service.query.BiddingResponseQueryService;
import com.alibaba.fastjson.JSON;
import definition.BiddingDemandResStatus;
import definition.BiddingStatus;
import definition.ProviderConformanceRespStatus;
import dto.*;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import query.BiddingDemandListQuery;
import query.BiddingDemandRespQuery;
import query.SupplierBiddingDamandQuery;
import sinomall.global.common.response.BaseResponse;
import utils.Lang;
import utils.data.BeanMapper;

import javax.persistence.criteria.Predicate;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 竞价响应查询业务逻辑层
 *
 * @author Drury
 * @date 2018/3/19
 */
@Slf4j
@Service
public class BiddingResponseQueryServiceImpl implements BiddingResponseQueryService {

    public static final String ACCORD_DEMAND = "满足需求";
    public static final String NOT_ACCORD_DEMAND = "不满足需求";

    private static final String MAX_PURCHASE_AMOUNT_DESC = "最高采购数量";
    private static final String MIN_PURCHASE_AMOUNT_DESC = "最低采购数量";

    // 比价信息查询接口
    private static final String SUPPLIER_QUOTEDPRICE_OK = "已报价";
    private static final String SUPPLIER_QUOTEDPRICE_NO = "未报价";

    /**
     * 竞价表
     */
    @Autowired
    JcbdBiddingRepo jcbdBiddingRepo;

    /**
     * 竞价需求表
     */
    @Autowired
    JcbdBiddingDemandRepo jcbdBiddingDemandRepo;

    /**
     * 竞价供应商需求响应表
     */
    @Autowired
    JcbdBiddingDemandRespRepo jcbdBiddingDemandRespRepo;

    /**
     * 竞价供应商表
     */
    @Autowired
    JcbdBdProviderRepo jcbdBdProviderRepo;

    /**
     * 竞价需求符合性表
     */
    @Autowired
    JcbdBiddingConformanceRepo jcbdBiddingConformanceRepo;

    /**
     * 需求地址表
     */
    @Autowired
    JcbdDemandAddressNumRepo jcbdDemandAddressNumRepo;

    /**
     * 供应商需求地址商品数量满足性响应表
     */
    @Autowired
    JcbdProviderAddressRespRepo jcbdProviderAddressRespRepo;

    /**
     * 供应商符合性响应表
     */
    @Autowired
    JcbdProviderConformanceRespRepo jcbdProviderConformanceRespRepo;

    @Autowired
    MapperFacade mapperFacade;

    /**
     * 查询竞价需求最终成交供应商信息
     *
     * @param demandCode
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public BaseResponse<BiddingResultSupplierInfoQueryDto> getBiddingResultSupplierInfo(String demandCode) {

        if (Lang.isEmpty(demandCode)) {
            return new BaseResponse(false, "需求编号为空", null);
        }

        JcbdBiddingDemand jbd = jcbdBiddingDemandRepo.findByDemandCodeAndIsDelete(demandCode, false);

        JcbdBidding jcbdBidding = jbd.getJcbdBidding();
        List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
        List<JcbdBdProvider> noJcBdProviders=new ArrayList<>();
        for(JcbdBdProvider jcbdBdProvider:jcbdBdProviders){
            if(jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_BIDDING_NOT.getValue())||jcbdBdProvider.getStatus().equals(BiddingStatus.SUPPLIER_AUDIT_IN_NOT.getValue())){
                noJcBdProviders.add(jcbdBdProvider);
            }
        }
        jcbdBdProviders.removeAll(noJcBdProviders);
        JcbdBdProvider provider = null;
        if (!Lang.isEmpty(jcbdBdProviders)) {
            provider = jcbdBdProviders.get(0);
        }

        BiddingResultSupplierInfoQueryDto brsi = new BiddingResultSupplierInfoQueryDto();
        brsi.setDemandCode(demandCode);
        brsi.setSupplierId(provider.getProviderId());
        return new BaseResponse(true, "查询成功", brsi);
    }

    /**
     * 供应商竞价响应概要信息查询
     *
     * @param biddingNo  竞价编号
     * @param demandCode 需求编号
     * @return
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse<List<BiddingResponseOutlineDto>> queryBiddingResponseOutline(String biddingNo, String demandCode) {
        log.info("BiddingResponseQueryServiceImpl.queryBiddingResponseOutline请求参数biddingCode:{},demandCode:{}", biddingNo, demandCode);
        if (Lang.isEmpty(biddingNo) || Lang.isEmpty(demandCode)) {
            return new BaseResponse<>("请求参数为空！");
        }
        // 获取竞价数据
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (Lang.isEmpty(bidding)) {
            return new BaseResponse<>("竞价数据为空!");
        }
        // 获取竞价需求
        JcbdBiddingDemand jcbdBiddingDemand = jcbdBiddingDemandRepo.findByDemandCodeAndIsDelete(demandCode, false);
        // 竞价供应商需求响应
        JcbdBiddingDemandResponse biddingDemandResponse = new JcbdBiddingDemandResponse();
        biddingDemandResponse.setJcbdBiddingDemand(jcbdBiddingDemand);
        List<JcbdBiddingDemandResponse> demandResponseList = jcbdBiddingDemandRespRepo.findAll(new BiddingDemandResponseSpecification<>(biddingDemandResponse));
        List<BiddingResponseOutlineDto> biddingOutlineDtoList = null;
        if (!Lang.isEmpty(demandResponseList)) {
            biddingOutlineDtoList = new ArrayList<>(demandResponseList.size());
            // 查询竞价供应商
            String responseProviderId;
            JcbdBdProviderDto jcbdBdProviderDto;
            BiddingResponseOutlineDto biddingResponseOutlineDto;
            String status;
            for (JcbdBiddingDemandResponse demandResponse : demandResponseList) {
                biddingResponseOutlineDto = new BiddingResponseOutlineDto();
                // 供应商表主键
                responseProviderId = demandResponse.getResponseProviderId();
                JcbdBdProvider provider = jcbdBdProviderRepo.findByIdAndIsDelete(responseProviderId, false);
                jcbdBdProviderDto = mapperFacade.map(provider, JcbdBdProviderDto.class);
                // 供应商id
                biddingResponseOutlineDto.setSupplierId(jcbdBdProviderDto.getProviderId());
                // 供应商响应状态
                status = jcbdBdProviderDto.getStatus();
                if (Lang.isEmpty(status) || BiddingStatus.SUPPLIER_RESPONSE_NOT.getValue().equals(status)) {
                    biddingResponseOutlineDto.setResponseStatus(BiddingStatus.SUPPLIER_RESPONSE_NOT.getDescription());
                } else {
                    if (demandResponseIsAllAccord(biddingNo, responseProviderId).isSuccess()) {
                        biddingResponseOutlineDto.setResponseStatus(ACCORD_DEMAND);
                    } else {
                        biddingResponseOutlineDto.setResponseStatus(NOT_ACCORD_DEMAND);
                    }
                }
                // 供应商名称
                biddingResponseOutlineDto.setSupplierName(jcbdBdProviderDto.getProviderName());
                biddingOutlineDtoList.add(biddingResponseOutlineDto);
            }
        }
        log.info("BiddingResponseQueryServiceImpl.queryBiddingResponseOutline 查询结果:{}", JSON.toJSONString(biddingOutlineDtoList));
        return new BaseResponse<>(true, "查询成功!", biddingOutlineDtoList);
    }

    /**
     * 供应商竞价响应 需求概要信息查询
     *
     * @param supplierBiddingDamandQuery 查询参数
     * @return
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse<Page<BiddingResponseDemandOutlineDto>> supplierBiddingResponseDemandOutline(SupplierBiddingDamandQuery supplierBiddingDamandQuery) {
        log.info("BiddingResponseServiceImpl.findSupplierBiddingResponse请求参数 {}", supplierBiddingDamandQuery);
        BaseResponse<Page<BiddingResponseDemandOutlineDto>> baseResponse;
        if (Lang.isEmpty(supplierBiddingDamandQuery)) {
            baseResponse = new BaseResponse<>("获取供应商竞价响应列表失败:请求参数为空!");
            log.info("BiddingResponseServiceImpl.findSupplierBiddingResponse:" + baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(supplierBiddingDamandQuery.getBiddingCode())) {
            baseResponse = new BaseResponse<>("获取供应商竞价响应列表失败:竞价编号为空!");
            log.info("BiddingResponseServiceImpl.findSupplierBiddingResponse:" + baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(supplierBiddingDamandQuery.getPageable())) {
            baseResponse = new BaseResponse<>("获取供应商竞价响应列表失败:分页参数为空!");
            log.info("BiddingResponseServiceImpl.findSupplierBiddingResponse:" + baseResponse.getResultMessage());
            return baseResponse;
        }
        try {
            // 获取竞价数据
            JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(supplierBiddingDamandQuery.getBiddingCode(), false);
            if (Lang.isEmpty(bidding)) {
                return new BaseResponse<>("竞价数据为空!");
            }
            // 获取竞价需求
            JcbdBiddingDemand biddingDemand = new JcbdBiddingDemand();
            biddingDemand.setJcbdBidding(bidding);
            Page<JcbdBiddingDemand> biddingDemandPage = jcbdBiddingDemandRepo.findAll(new BiddingDemandSpecification<>(biddingDemand),
                    supplierBiddingDamandQuery.getPageable());
            List<JcbdBiddingDemand> jcbdBiddingDemandList = biddingDemandPage.getContent();
            if (Lang.isEmpty(jcbdBiddingDemandList)) {
                log.info("BiddingResponseServiceImpl.findSupplierBiddingResponse:获取竞价需求列表为空!");
                return new BaseResponse<>("获取竞价需求列表为空!");
            }
            // 查询供应商
            JcbdBdProvider jcbdBdProvider = null;
            if (!Lang.isEmpty(supplierBiddingDamandQuery.getSupplierId())) {
                jcbdBdProvider = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(bidding.getId(), supplierBiddingDamandQuery.getSupplierId(), false);
            }
            List<BiddingResponseDemandOutlineDto> biddingResponseDemandOutlineDtos = new ArrayList<>();
            BiddingResponseDemandOutlineDto biddingResponseDemandOutlineDto;
            JcbdBiddingDemandResponse jcbdBiddingDemandResponse;
            List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses;
            JcbdBiddingConformance jcbdBiddingConformance;
            JcbdBiddingDemandResponse biddingDemandResponse;
            JcbdDemandAddressNum jcbdDemandAddressNum;
            StringBuffer stringBuffer;
            Long accordCheckCount;
            Long supplierResCount;
            for (JcbdBiddingDemand jcbdBiddingDemand : jcbdBiddingDemandList) {
                biddingResponseDemandOutlineDto = new BiddingResponseDemandOutlineDto();
                biddingResponseDemandOutlineDto.setDemandId(jcbdBiddingDemand.getId());
                biddingResponseDemandOutlineDto.setDemandNo(jcbdBiddingDemand.getDemandCode());
                biddingResponseDemandOutlineDto.setGoodsModel(jcbdBiddingDemand.getGoodsModel());
                biddingResponseDemandOutlineDto.setGoodsBrand(jcbdBiddingDemand.getGoodsBrand());
                biddingResponseDemandOutlineDto.setSpecification(jcbdBiddingDemand.getSpecification());
                biddingResponseDemandOutlineDto.setGoodsDesription(jcbdBiddingDemand.getGoodsDesription());
                biddingResponseDemandOutlineDto.setHighestPurchaseNumber(jcbdBiddingDemand.getMaxPurchaseAmount());
                biddingResponseDemandOutlineDto.setLowestPurchaseNumber(jcbdBiddingDemand.getMinPurchaseAmount());
                biddingResponseDemandOutlineDto.setRequiredDeliveryDate(jcbdBiddingDemand.getDateRequiredDelivery());
                // 查询符合性审查数量
                jcbdBiddingConformance = new JcbdBiddingConformance();
                jcbdBiddingConformance.setJcbdBiddingDemand(jcbdBiddingDemand);
                accordCheckCount = jcbdBiddingConformanceRepo.count(new BiddingConformanceSpecification<>(jcbdBiddingConformance));
                biddingResponseDemandOutlineDto.setConformanceNumber(accordCheckCount.intValue());
                // 查询需求地区
                jcbdDemandAddressNum = new JcbdDemandAddressNum();
                jcbdDemandAddressNum.setJcbdBiddingDemand(jcbdBiddingDemand);
                List<JcbdDemandAddressNum> jcbdDemandAddressNums = jcbdDemandAddressNumRepo.findAll(new BiddingDemandAddressSpecification<>(jcbdDemandAddressNum));
                stringBuffer = new StringBuffer();
                if (!Lang.isEmpty(jcbdDemandAddressNums)) {
                    for (JcbdDemandAddressNum demandAddressNum : jcbdDemandAddressNums) {
                        stringBuffer.append(demandAddressNum.getProvinceName())
                                .append(demandAddressNum.getCityName())
                                .append(",");
                    }
                    biddingResponseDemandOutlineDto.setDemandAddress(stringBuffer.substring(0, stringBuffer.length() - 1));
                }
                // 如果有传供应商id，是供应商查看的页面，则返回响应状态，反之是审查小组查看的页面，则返回响应供应商数量
                if (!Lang.isEmpty(jcbdBdProvider)) {
                    // 查询响应表
                    jcbdBiddingDemandResponse = new JcbdBiddingDemandResponse();
                    jcbdBiddingDemandResponse.setJcbdBiddingDemand(jcbdBiddingDemand);
                    jcbdBiddingDemandResponse.setResponseProviderId(jcbdBdProvider.getId());
                    jcbdBiddingDemandResponses = jcbdBiddingDemandRespRepo.findAll(new BiddingDemandResponseSpecification<>(jcbdBiddingDemandResponse));
                    // 如果有响应数据，设置需求响应状态为已响应，否则状态为未响应
                    if (jcbdBiddingDemandResponses.size() > 0) {
                        biddingResponseDemandOutlineDto.setDemandResponseStatus(BiddingDemandResStatus.DEMAND_RESPONSE_STATUS.getDescription());
                        biddingResponseDemandOutlineDto.setPrice(jcbdBiddingDemandResponses.get(0).getPrice());
                    } else {
                        biddingResponseDemandOutlineDto.setDemandResponseStatus(BiddingDemandResStatus.NOT_DEMAND_RESPONSE_STATUS.getDescription());
                    }
                } else {
                    // 查询供应商响应数量
                    biddingResponseDemandOutlineDto.setResponseSupplierNumber(bidding.getProviderResponseNum());
                }
                biddingResponseDemandOutlineDtos.add(biddingResponseDemandOutlineDto);
            }
            Page<BiddingResponseDemandOutlineDto> suppBiddingResponseDtoPage = new PageImpl<>(biddingResponseDemandOutlineDtos, supplierBiddingDamandQuery.getPageable(),
                    biddingResponseDemandOutlineDtos.size());
            log.info("BiddingResponseServiceImpl.findSupplierBiddingResponse 获取结果:{}", JSON.toJSONString(suppBiddingResponseDtoPage));
            return new BaseResponse<>(true, "获取成功", suppBiddingResponseDtoPage);
        } catch (Exception e) {
            log.info("获取供应商竞价响应列表异常，竞价编号 {}", supplierBiddingDamandQuery.getBiddingCode(), e);
            return new BaseResponse<>("获取供应商竞价响应列表异常!");
        }
    }

    /**
     * 供应商需求响应查询接口
     *
     * @param biddingDemandRespQuery 查询参数
     * @return
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse<JcbdBiddingDemandResponseDto> queryBiddingDemandResponse(BiddingDemandRespQuery biddingDemandRespQuery) {
        log.info("BiddingResponseQueryServiceImpl.queryBiddingDemandResponse请求参数 {}", JSON.toJSONString(biddingDemandRespQuery));
        if (Lang.isEmpty(biddingDemandRespQuery) || Lang.isEmpty(biddingDemandRespQuery.getSupplierId()) ||
                Lang.isEmpty(biddingDemandRespQuery.getDemandCode()) || Lang.isEmpty(biddingDemandRespQuery.getBiddingNo())) {
            return new BaseResponse<>("查询竞价供应商需求响应请求参数为空!");
        }
        JcbdBiddingDemandResponseDto jcbdBiddingDemandResponseDto = new JcbdBiddingDemandResponseDto();
        // 获取竞价数据
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingDemandRespQuery.getBiddingNo(), false);
        if (Lang.isEmpty(bidding)) {
            return new BaseResponse<>("竞价数据为空!");
        }
        // 获取竞价需求
        JcbdBiddingDemand jcbdBiddingDemand = jcbdBiddingDemandRepo.findByDemandCodeAndIsDelete(biddingDemandRespQuery.getDemandCode(), false);
        // 查询竞价供应商需求响应数据
        JcbdBiddingDemandResponse jcbdBiddingDemandResponse = new JcbdBiddingDemandResponse();
        jcbdBiddingDemandResponse.setJcbdBiddingDemand(jcbdBiddingDemand);

        JcbdBdProvider jcbdBdProvider = new JcbdBdProvider();
        jcbdBdProvider.setProviderId(biddingDemandRespQuery.getSupplierId());
        jcbdBdProvider.setJcbdBidding(bidding);
        JcbdBdProvider provider = jcbdBdProviderRepo.findOne(new BiddingProviderSpecification<>(jcbdBdProvider));
        if (!Lang.isEmpty(provider)) {
            // 设置供应商表主键id
            jcbdBiddingDemandResponse.setResponseProviderId(provider.getId());
            // 封装报价有效期数据
            jcbdBiddingDemandResponseDto.setOfferValidity(provider.getOfferValidity());
        }
        JcbdBiddingDemandResponse biddingDemandResponse = jcbdBiddingDemandRespRepo.findOne(new BiddingDemandResponseSpecification<>(jcbdBiddingDemandResponse));
        JcbdBiddingDemandDto jcbdBiddingDemandDto = new JcbdBiddingDemandDto();
        if (!Lang.isEmpty(jcbdBiddingDemand)) {
            jcbdBiddingDemandDto = mapperFacade.map(jcbdBiddingDemand, JcbdBiddingDemandDto.class);
            // 返回竞价id和竞价编号
            jcbdBiddingDemandDto.setBiddingId(bidding.getId());
            jcbdBiddingDemandDto.setBiddingCode(bidding.getBiddingCode());
        }
        // 如果需求响应数据为空，则查询地址表和符合性审查表
        if (Lang.isEmpty(biddingDemandResponse)) {
            // 封装地址列表
            JcbdDemandAddressNum jcbdDemandAddressNum = new JcbdDemandAddressNum();
            jcbdDemandAddressNum.setJcbdBiddingDemand(jcbdBiddingDemand);
            List<JcbdDemandAddressNum> addressNums = jcbdDemandAddressNumRepo.findAll(new BiddingDemandAddressSpecification<>(jcbdDemandAddressNum));
            List<JcbdProviderAddressResponseDto> jcbdProviderAddressResponseDtos = new ArrayList<>(addressNums.size());
            if (!Lang.isEmpty(addressNums)) {
                JcbdProviderAddressResponseDto jcbdProviderAddressResponseDto;
                for (JcbdDemandAddressNum addressNum : addressNums) {
                    jcbdProviderAddressResponseDto = new JcbdProviderAddressResponseDto();
                    jcbdProviderAddressResponseDto.setDemandAddressId(addressNum.getId());
                    jcbdProviderAddressResponseDto.setContent(addressNum.getProvinceName() + addressNum.getCityName());
                    // 需求数量
                    jcbdProviderAddressResponseDto.setNumberNeed(addressNum.getNumberNeed());
                    jcbdProviderAddressResponseDtos.add(jcbdProviderAddressResponseDto);
                }
            }
            // 最低采购数量
            JcbdProviderAddressResponseDto minPurchaseAmount = new JcbdProviderAddressResponseDto();
            minPurchaseAmount.setContent(MIN_PURCHASE_AMOUNT_DESC);
            minPurchaseAmount.setNumberNeed(jcbdBiddingDemand.getMinPurchaseAmount());
            // 最高采购数量
            JcbdProviderAddressResponseDto maxPurchaseAmount = new JcbdProviderAddressResponseDto();
            maxPurchaseAmount.setNumberNeed(jcbdBiddingDemand.getMaxPurchaseAmount());
            maxPurchaseAmount.setContent(MAX_PURCHASE_AMOUNT_DESC);
            jcbdProviderAddressResponseDtos.add(minPurchaseAmount);
            jcbdProviderAddressResponseDtos.add(maxPurchaseAmount);
            jcbdBiddingDemandResponseDto.setJcbdProviderAddressResponses(jcbdProviderAddressResponseDtos);
            // 封装符合性审查列表
            JcbdBiddingConformance jcbdBiddingConformance = new JcbdBiddingConformance();
            jcbdBiddingConformance.setJcbdBiddingDemand(jcbdBiddingDemand);
            List<JcbdBiddingConformance> jcbdBiddingConformances = jcbdBiddingConformanceRepo.findAll(new BiddingConformanceSpecification<>(jcbdBiddingConformance));
            if (!Lang.isEmpty(jcbdBiddingConformances)) {
                List<JcbdProviderConformanceRespDto> jcbdProviderConformanceResDtos = new ArrayList<>(jcbdBiddingConformances.size());
                JcbdProviderConformanceRespDto jcbdProviderConformanceRespDto;
                for (JcbdBiddingConformance biddingConformance : jcbdBiddingConformances) {
                    jcbdProviderConformanceRespDto = new JcbdProviderConformanceRespDto();
                    jcbdProviderConformanceRespDto.setBiddingConformanceId(biddingConformance.getId());
                    // 符合性内容
                    jcbdProviderConformanceRespDto.setConformance(biddingConformance.getConformance());
                    jcbdProviderConformanceResDtos.add(jcbdProviderConformanceRespDto);
                }
                jcbdBiddingDemandResponseDto.setJcbdProviderConformanceResps(jcbdProviderConformanceResDtos);
            }
        } else {
            if (!Lang.isEmpty(biddingDemandResponse)) {
                jcbdBiddingDemandResponseDto = mapperFacade.map(biddingDemandResponse, JcbdBiddingDemandResponseDto.class);
                if (!Lang.isEmpty(biddingDemandResponse.getQouteIndate())) {
                    jcbdBiddingDemandResponseDto.setOfferValidity(biddingDemandResponse.getQouteIndate());
                }
                // 返回竞价id和竞价编号
                jcbdBiddingDemandDto.setBiddingId(bidding.getId());
                jcbdBiddingDemandDto.setBiddingCode(bidding.getBiddingCode());
            }
            // 封装需求地区及数量列表
            JcbdProviderAddressResponse providerAddressResponse = new JcbdProviderAddressResponse();
            providerAddressResponse.setJcbdBiddingDemandResponse(biddingDemandResponse);
            List<JcbdProviderAddressResponse> providerAddressList = jcbdProviderAddressRespRepo.
                    findAll(new BiddingProviderAddRespSpecification<>(providerAddressResponse));
            if (!Lang.isEmpty(providerAddressList)) {
                JcbdProviderAddressResponseDto jcbdProviderAddressResponseDto;
                List<JcbdProviderAddressResponseDto> jcbdProviderAddressResponseDtos = new ArrayList<>(providerAddressList.size());
                for (JcbdProviderAddressResponse addressResponse : providerAddressList) {
                    jcbdProviderAddressResponseDto = new JcbdProviderAddressResponseDto();
                    jcbdProviderAddressResponseDto.setDemandAddressId(addressResponse.getDemandAddressId());
                    // 需求数量
                    jcbdProviderAddressResponseDto.setNumberNeed(addressResponse.getNumberNeed());
                    jcbdProviderAddressResponseDto.setContent(addressResponse.getContent());
                    // 备注
                    jcbdProviderAddressResponseDto.setRemark(addressResponse.getRemark());
                    // 状态
                    jcbdProviderAddressResponseDto.setStatus(addressResponse.getStatus());
                    jcbdProviderAddressResponseDtos.add(jcbdProviderAddressResponseDto);
                }
                jcbdBiddingDemandResponseDto.setJcbdProviderAddressResponses(jcbdProviderAddressResponseDtos);
            }
            // 封装供应商符合性审查响应
            JcbdProviderConformanceResp providerConformanceResp = new JcbdProviderConformanceResp();
            providerConformanceResp.setJcbdBiddingDemandResponse(biddingDemandResponse);
            List<JcbdProviderConformanceResp> providerConformanceRespList = jcbdProviderConformanceRespRepo.
                    findAll(new BiddingProviderConformanceRespSpec<>(providerConformanceResp));
            if (!Lang.isEmpty(providerConformanceRespList)) {
                // 供应商符合性响应列表
                List<JcbdProviderConformanceRespDto> jcbdProviderConformanceResDtos = new ArrayList<>(providerConformanceRespList.size());
                JcbdProviderConformanceRespDto jcbdProviderConformanceRespDto;
                JcbdBiddingConformance jcbdBiddingConformance;
                for (JcbdProviderConformanceResp conformanceResp : providerConformanceRespList) {
                    jcbdProviderConformanceRespDto = new JcbdProviderConformanceRespDto();
                    jcbdBiddingConformance = jcbdBiddingConformanceRepo.findOne(conformanceResp.getBiddingConformanceId());
                    // 符合性内容
                    jcbdProviderConformanceRespDto.setConformance(jcbdBiddingConformance.getConformance());
                    jcbdProviderConformanceRespDto.setBiddingConformanceId(jcbdBiddingConformance.getId());
//                    if (JcbdProviderConformanceRespDto.CONFORMANCE_STATUS_YES_NUM.equals(conformanceResp.getStatus())){
//                        jcbdProviderConformanceRespDto.setStatus(JcbdProviderConformanceRespDto.CONFORMANCE_STATUS_YES_DESC);
//                    }else{
//                        jcbdProviderConformanceRespDto.setStatus(JcbdProviderConformanceRespDto.CONFORMANCE_STATUS_NO_DESC);
//                    }
                    jcbdProviderConformanceRespDto.setStatus(conformanceResp.getStatus());
                    jcbdProviderConformanceRespDto.setRemark(conformanceResp.getRemark());
                    jcbdProviderConformanceResDtos.add(jcbdProviderConformanceRespDto);
                }
                jcbdBiddingDemandResponseDto.setJcbdProviderConformanceResps(jcbdProviderConformanceResDtos);
            }
        }
        jcbdBiddingDemandResponseDto.setJcbdBiddingDemand(jcbdBiddingDemandDto);
        log.info("BiddingResponseQueryServiceImpl.queryBiddingDemandResponse查询结果:{}", JSON.toJSONString(jcbdBiddingDemandResponseDto));
        return new BaseResponse<>(true, "查询成功", jcbdBiddingDemandResponseDto);
    }

    /**
     * 比价信息查询
     *
     * @param biddingNo 竞价编号
     * @return
     */
    @Override
    @Transactional
    public BaseResponse<BiddingPriceParityDto> queryBiddingPriceParity(String biddingNo) {
        log.info("BiddingResponseQueryServiceImpl.queryBiddingPriceParity-请求参数biddingNo {}", biddingNo);
        if (Lang.isEmpty(biddingNo)) {
            return new BaseResponse<>("竞价编号为空!");
        }
        // 获取竞价数据
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (Lang.isEmpty(jcbdBidding)) {
            return new BaseResponse<>("竞价数据为空!");
        }
        BiddingPriceParityDto biddingPriceParityDto = new BiddingPriceParityDto();
        BiddingResponseResultDto biddingResponseResult;
        List<JcbdBdProvider> jcbdBdProviders = jcbdBidding.getJcbdBdProviders();
        List<BiddingResponseResultDto> biddingResponseResultList = null;
        if (!Lang.isEmpty(jcbdBdProviders)) {
            biddingResponseResultList = new ArrayList<>(jcbdBdProviders.size());
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviders) {
                biddingResponseResult = new BiddingResponseResultDto();
                // 封装比价页竞价响应信息
                biddingResponseResult.setSupllierName(jcbdBdProvider.getProviderName());
                // 供应商id
                biddingResponseResult.setProviderId(jcbdBdProvider.getProviderId());
                // 报价状态
                biddingResponseResult.setQuotePriceStatus(SUPPLIER_QUOTEDPRICE_NO);
                if (ProviderConformanceRespStatus.ACCORD.getValue().equals(jcbdBdProvider.getResponseStatus())) {
                    biddingResponseResult.setQuotePriceStatus(SUPPLIER_QUOTEDPRICE_OK);
                }
                // 符合性审查状态
                biddingResponseResult.setConformanceStatus(ProviderConformanceRespStatus.NOT_ACCORD.getDescription());
                if (ProviderConformanceRespStatus.ACCORD.getValue().equals(jcbdBdProvider.getConformanceCheckStatus())) {
                    biddingResponseResult.setConformanceStatus(ProviderConformanceRespStatus.ACCORD.getDescription());
                }
                // 竞价结果
                biddingResponseResult.setBiddingResult(jcbdBdProvider.getDealStatus());
                biddingResponseResultList.add(biddingResponseResult);
            }
        }
        // 获取竞价需求
        JcbdBiddingDemand jcbdBiddingDemand = new JcbdBiddingDemand();
        jcbdBiddingDemand.setJcbdBidding(jcbdBidding);
        List<JcbdBiddingDemand> jcbdBiddingDemandList = jcbdBiddingDemandRepo.findAll(new BiddingDemandSpecification<>(jcbdBiddingDemand));
        log.info("BiddingResponseQueryServiceImpl.queryBiddingPriceParity-竞价需求列表：{}", jcbdBiddingDemandList);

        List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponseList;
        JcbdBdProvider jcbdBdProvider;
        PriceParityInfoDto priceParityInfo;
        Long minPurchaseAmount;
        BigDecimal price;
        DecimalFormat df = new DecimalFormat("#.00");
        BigDecimal totalPrice = BigDecimal.ZERO;
        List<PriceParityInfoDto> priceParityInfoList = new ArrayList<>(jcbdBiddingDemandList.size());
        if (!Lang.isEmpty(jcbdBiddingDemandList)) {
            for (JcbdBiddingDemand biddingDemand : jcbdBiddingDemandList) {
                // 查询竞价需求响应
                jcbdBiddingDemandResponseList = jcbdBiddingDemandRespRepo.findByJcbdBiddingDemandIdAndIsDelete(biddingDemand.getId(), false);
                if (!Lang.isEmpty(jcbdBiddingDemandResponseList)) {
                    for (JcbdBiddingDemandResponse biddingDemandResponse : jcbdBiddingDemandResponseList) {
                        priceParityInfo = new PriceParityInfoDto();
                        jcbdBdProvider = jcbdBdProviderRepo.findOne(biddingDemandResponse.getResponseProviderId());
                        if (!Lang.isEmpty(jcbdBdProvider)) {
                            // 比价页比价信息
                            priceParityInfo.setSupllierName(jcbdBdProvider.getProviderName());
                            priceParityInfo.setSupplierStatus(jcbdBdProvider.getStatus());
                            // 供应商id
                            priceParityInfo.setProviderId(jcbdBdProvider.getProviderId());
                        }
                        // 最低采购数量
                        minPurchaseAmount = biddingDemand.getMinPurchaseAmount();
                        if(minPurchaseAmount==null){
                            minPurchaseAmount=Long.parseLong("1");
                        }
                        // 单价
                        price = biddingDemandResponse.getPrice();
                        if (!Lang.isEmpty(minPurchaseAmount) && !Lang.isEmpty(price)) {
                            // 计算总价
                            totalPrice = price.multiply(new BigDecimal(df.format(minPurchaseAmount.doubleValue())));
                        }
                        // 比价页比价信息
                        priceParityInfo.setDemandNo(biddingDemand.getDemandCode());
                        priceParityInfo.setLowestPurchaseNumber(minPurchaseAmount);
                        priceParityInfo.setPrice(price);
                        priceParityInfo.setTotalPrice(totalPrice);
                        priceParityInfoList.add(priceParityInfo);
                    }
                }
            }
            // 处理信息
            List<PriceParityInfoDto> priceParityInfosDtos = dealPriceParityInfos(priceParityInfoList, biddingResponseResultList);
            biddingPriceParityDto.setBiddingResponseResultDtos(biddingResponseResultList);
            biddingPriceParityDto.setPriceParityInfosDtos(priceParityInfosDtos);
            log.info("BiddingResponseQueryServiceImpl.queryBiddingPriceParity 查询结果：{}", JSON.toJSONString(biddingPriceParityDto));
        }
        return new BaseResponse<>(true, "查询成功!", biddingPriceParityDto);
    }

    @Autowired
    JcbdStoreContactsRepo jcbdStoreContactsRepo;

    /**
     * 新建竞价时获取当前所有可用供应商
     *
     * @param biddingNo 竞价编号
     * @return
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse<List<BiddingSupplierDTO>> getAllSupplier(String biddingNo) {
        log.info("BiddingResponseQueryServiceImpl.getAllSupplier-请求参数biddingNo: {}", biddingNo);
        if (Lang.isEmpty(biddingNo)) {
            return new BaseResponse<>("请求参数竞价编号为空!");
        }
        JcbdBidding bidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        // 查询所有供应商
        JcbdBdProvider jcbdBdProvider = new JcbdBdProvider();
        jcbdBdProvider.setJcbdBidding(bidding);
        List<JcbdBdProvider> jcbdBdProviders = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(jcbdBdProvider));
        if (Lang.isEmpty(jcbdBdProviders)) {
            return new BaseResponse<>("BiddingResponseQueryServiceImpl.getAllSupplier-查询供应商列表为空!");
        }
        List<BiddingSupplierDTO> biddingSupplierDTOS = new ArrayList<>();
        BiddingSupplierDTO biddingSupplierDTO;
        for (JcbdBdProvider provider : jcbdBdProviders) {
            biddingSupplierDTO = new BiddingSupplierDTO();
            biddingSupplierDTO.setId(provider.getId());
            biddingSupplierDTO.setName(provider.getProviderName());
            List<JcbdStoreContacts> jcbdStoreContactses = jcbdStoreContactsRepo.findByJcbdBdProviderIdAndIsDelete(provider.getId(), false);
            if (!Lang.isEmpty(jcbdStoreContactses)) {
                biddingSupplierDTO.setContacts(BeanMapper.mapList(jcbdStoreContactses, JcbdStoreContacts.class, BiddingSupplierContactDTO.class));
            }
            biddingSupplierDTOS.add(biddingSupplierDTO);
        }
        log.info("BiddingResponseQueryServiceImpl.getAllSupplier 查询结果：{}", biddingSupplierDTOS);
        return new BaseResponse<>(true, "查询成功", biddingSupplierDTOS);
    }

    @Override
    @Transactional
    public Page<DemandResponseQueryDto> findDemandAndResponse(DemanResponseParmDto parms) {
        JcbdBidding jcbidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(parms.getBiddingNo(), false);
        String jcbiddingId = jcbidding.getId();
        String providerId = parms.getProviderId();
        JcbdBdProvider provider = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(jcbiddingId, providerId, false);
        List<DemandResponseQueryDto> demandResponseQueryDtos = new ArrayList<>();
        Page<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBiddingDemandRepo.findByJcbdBiddingIdAndIsDelete(jcbiddingId, false, parms.getPageable());
        for (JcbdBiddingDemand jcbdBiddingDemand : jcbdBiddingDemands) {
            DemandResponseQueryDto demandResponseQueryDto = new DemandResponseQueryDto();
            demandResponseQueryDto.setDemandNo(jcbdBiddingDemand.getDemandCode());
            List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses = jcbdBiddingDemand.getJcbdBiddingDemandResponses();
            for (JcbdBiddingDemandResponse demandResponse : jcbdBiddingDemandResponses) {
                if (provider.getId().equals(demandResponse.getResponseProviderId())) {
                    demandResponseQueryDto.setSku(demandResponse.getSku());
                    demandResponseQueryDto.setGoodsName(demandResponse.getGoodsTitle());
                    demandResponseQueryDto.setGoodsBrand(demandResponse.getGoodsBrand());
                    demandResponseQueryDto.setGoodsModel(demandResponse.getGoodsModel());
                    demandResponseQueryDto.setGoodsParams(demandResponse.getGoodsParams());
                    demandResponseQueryDto.setPrice(demandResponse.getPrice());
                    demandResponseQueryDto.setUrl(demandResponse.getGoodsLink());
                    demandResponseQueryDtos.add(demandResponseQueryDto);
                }
            }
        }
        return new PageImpl(demandResponseQueryDtos, new PageRequest(jcbdBiddingDemands.getNumber(), jcbdBiddingDemands.getSize(), jcbdBiddingDemands.getSort()), jcbdBiddingDemands.getTotalElements());
    }

    @Override
    @Transactional
    public List<String> findBiddingResponseSkus(String jcbdBiddingNo, String supplierId) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(jcbdBiddingNo, false);
        JcbdBdProvider provider = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(jcbdBidding.getId(), supplierId, false);
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        List<String> skus = new ArrayList<>();
        jcbdBiddingDemands.forEach(jcbdBiddingDemand -> {
            jcbdBiddingDemand.getJcbdBiddingDemandResponses().forEach(jcbdBiddingDemandResponse -> {
                if (jcbdBiddingDemandResponse.getResponseProviderId().equals(provider.getId()) && !Lang.isEmpty(jcbdBiddingDemandResponse.getSku())) {
                    skus.add(jcbdBiddingDemandResponse.getSku());
                }
            });
        });

        return skus;
    }


    /**
     * 通过竞价编号查询已成交供应商需求响应列表
     *
     * @param biddingDemandListQuery
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public BaseResponse<Page<BiddingResultDemandQueryDto>> getBiddingResultDemandList(BiddingDemandListQuery biddingDemandListQuery) {

        BaseResponse<Page<BiddingResultDemandQueryDto>> baseResponse;
        log.info("采购经办人-竞价结果页面已成交供应商响应需求列表-请求参数 {}", JSON.toJSONString(biddingDemandListQuery));
        // 请求参数校验
        if (Lang.isEmpty(biddingDemandListQuery)) {
            baseResponse = new BaseResponse<>(false, "采购经办人-竞价结果页面已成交供应商响应需求列表-请求参数不能为空", null);
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(biddingDemandListQuery.getBiddingCode())) {
            baseResponse = new BaseResponse<>(false, "采购经办人-竞价结果页面已成交供应商响应需求列表-竞价编号不能为空", null);
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        } else if (Lang.isEmpty(biddingDemandListQuery.getPageable())) {
            baseResponse = new BaseResponse<>(false, "采购经办人-竞价结果页面已成交供应商响应需求列表-分页信息不能为空", null);
            log.error(baseResponse.getResultMessage());
            return baseResponse;
        }
        try {
            JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingDemandListQuery.getBiddingCode(), false);
            if (Lang.isEmpty(jcbdBidding)) {
                baseResponse = new BaseResponse<>(false, "竞价不存在");
                log.error("竞价{}不存在", biddingDemandListQuery.getBiddingCode());
                return baseResponse;
            }
            //查询已成交供应商
            JcbdBdProvider provider = new JcbdBdProvider();
            provider.setJcbdBidding(jcbdBidding);
            provider.setStatus(BiddingStatus.SUPPLIER_BIDDING_YES.getValue());
            List<JcbdBdProvider> providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_BIDDING_YES_AGAIN.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_SKU_AUDIT_WAIT.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_AUDIT_IN_YES.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_SKU_MAKEUP_WAIT.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty(providers)) {
                provider.setStatus(BiddingStatus.SUPPLIER_SKU_AUDIT_NOT.getValue());
                providers = jcbdBdProviderRepo.findAll(new BiddingProviderSpecification<>(provider));
            }
            if (Lang.isEmpty((providers))) {
                baseResponse = new BaseResponse<>(true, "查询成功", null);
                return baseResponse;
            }
            provider = providers.get(0);

            //查询竞价需求
            JcbdBiddingDemand biddingDemand = new JcbdBiddingDemand();
            biddingDemand.setJcbdBidding(jcbdBidding);
            Page<JcbdBiddingDemand> biddingDemandPage = jcbdBiddingDemandRepo.findAll(new BiddingDemandSpecification<>(biddingDemand), biddingDemandListQuery.getPageable());
            List<JcbdBiddingDemand> biddingDemands = biddingDemandPage.getContent();
            if (Lang.isEmpty(biddingDemands)) {
                baseResponse = new BaseResponse<>(true, "查询成功", null);
                return baseResponse;
            }

            List<BiddingResultDemandQueryDto> list = new ArrayList<>();
            for (JcbdBiddingDemand demand : biddingDemands) {
                JcbdBiddingDemandResponse jbdr = new JcbdBiddingDemandResponse();
                jbdr.setJcbdBiddingDemand(demand);
                jbdr.setResponseProviderId(provider.getId());
                List<JcbdBiddingDemandResponse> jbdrs = jcbdBiddingDemandRespRepo.findAll(new BiddingDemandResponseSpecification(jbdr));
                if (!Lang.isEmpty(jbdrs)) {
                    BiddingResultDemandQueryDto brdqo = mapperFacade.map(jbdrs.get(0), BiddingResultDemandQueryDto.class);
                    brdqo.setDemandCode(demand.getDemandCode());
                    brdqo.setResponseProviderName(provider.getProviderName());
                    brdqo.setGoodsTitle(demand.getGoodsDesription());
                    brdqo.setGoodsBrand(demand.getGoodsBrand());
                    brdqo.setGoodsModel(demand.getGoodsModel());
                    brdqo.setGoodsParams(demand.getSpecification());
                    list.add(brdqo);
                }
            }

            Page<BiddingResultDemandQueryDto> BiddingResultDemandQueryDtoPage = new PageImpl<>(list, biddingDemandListQuery.getPageable(), list.size());

            log.info("获取竞价需求列表 DATA : {}", JSON.toJSONString(BiddingResultDemandQueryDtoPage));

            return new BaseResponse<>(true, "查询成功", BiddingResultDemandQueryDtoPage);

        } catch (Exception e) {
            log.error("采购经办人-竞价结果页面已成交供应商响应需求列表, 竞价编号 {} ", biddingDemandListQuery.getBiddingCode(), e);
            return new BaseResponse<>(false, "查询异常", null);
        }
    }

    @Override
    @Transactional
    public JcbdBiddingDemandResponseDto findProviderResponsePrice(String biddingNo, String sku) {
        List<JcbdBdProvider> jcbdBdProviders = jcbdBdProviderRepo.findByJcbdBiddingBiddingCodeAndIsDelete(biddingNo, false);
        String providerId = "";
        if (!Lang.isEmpty(jcbdBdProviders)) {
            for (JcbdBdProvider jcbdBdProvider : jcbdBdProviders) {
                if (BiddingStatus.SUPPLIER_BIDDING_YES_AGAIN.getValue().equals(jcbdBdProvider.getStatus())) {
                    providerId = jcbdBdProvider.getId();
                    break;
                }
            }
        }
        JcbdBiddingDemandResponse biddingDemandResponse = jcbdBiddingDemandRespRepo.findByResponseProviderIdAndSku(providerId, sku);
        if (!Lang.isEmpty(biddingDemandResponse)) {
            return mapperFacade.map(biddingDemandResponse, JcbdBiddingDemandResponseDto.class);
        }
        return null;
    }

    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse isAllDemandAreResponse(String jcbdBiddingNo, String responseProviderId) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(jcbdBiddingNo, false);
        JcbdBdProvider jcbdBdProvider = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(jcbdBidding.getId(), responseProviderId, false);
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        for (JcbdBiddingDemand jcbdBiddingDemand : jcbdBiddingDemands) {
//            List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses = jcbdBiddingDemand.getJcbdBiddingDemandResponses();
//            for (JcbdBiddingDemandResponse jcbdBiddingDemandResponse : jcbdBiddingDemandResponses) {
//                if (!jcbdBiddingDemandResponse.getResponseProviderId().equals(jcbdBdProvider.getId())) {
//                    return new BaseResponse("该需求未被全部响应！");
//                }
//            }
            JcbdBiddingDemandResponse demandResponse = jcbdBiddingDemandRespRepo.findByJcbdBiddingDemandIdAndResponseProviderIdAndIsDelete(jcbdBiddingDemand.getId(), jcbdBdProvider.getId(), false);
            if (demandResponse == null) {
                return new BaseResponse("该需求未被全部响应！");
            }

        }
        return new BaseResponse(true, "该需求已经全部响应！");
    }

    @Override
    @Transactional(readOnly = true, rollbackFor = Exception.class)
    public BaseResponse  demandResponseIsAllAccord(String jcbdBiddingNo, String responseProviderId) {
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(jcbdBiddingNo, false);
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        for (JcbdBiddingDemand jcbdBiddingDemand : jcbdBiddingDemands) {
            List<JcbdBiddingDemandResponse> jcbdBiddingDemandResponses = jcbdBiddingDemand.getJcbdBiddingDemandResponses();
            for (JcbdBiddingDemandResponse jcbdBiddingDemandResponse : jcbdBiddingDemandResponses) {
                if (jcbdBiddingDemandResponse.getResponseProviderId().equals(responseProviderId)) {
                    String isAllAccord = jcbdBiddingDemandResponse.getIsAllAccord();
                    if ("0".equals(isAllAccord)) {
                        return new BaseResponse("存在不满足的需求响应！");
                    }
                }
            }
        }
        return new BaseResponse(true, "所有需求响应都满足！");
    }


    @Override
    public List<JcbdBiddingDemandResponseDto> findAll(List<String> demandRspIds) {
        List<JcbdBiddingDemandResponse> demandResponses = jcbdBiddingDemandRespRepo.findAll((root, query, cb) -> cb.and(cb.equal(root.get("isDelete"), false),
                cb.or(demandRspIds.stream().map(id -> cb.equal(root.get("id"), id)).
                        collect(Collectors.toList()).toArray(new Predicate[0]))));
        return mapperFacade.mapAsList(demandResponses, JcbdBiddingDemandResponseDto.class);
    }


    /**
     * 分类处理比价信息
     *
     * @param priceParityInfoList
     * @param biddingResponseResultList
     * @return
     */
    private List<PriceParityInfoDto> dealPriceParityInfos(List<PriceParityInfoDto> priceParityInfoList,
                                                          List<BiddingResponseResultDto> biddingResponseResultList) {
        List<PriceParityInfoDto> priceParityInfoDtoList = new ArrayList<>();
        if (Lang.isEmpty(priceParityInfoList)) {
            return priceParityInfoDtoList;
        }
        Map<String, List<PriceParityInfoDto>> priceParityInfoMap = priceParityInfoList.stream().collect(Collectors.groupingBy(PriceParityInfoDto::getSupllierName));
        // 处理map
        Set<Map.Entry<String, List<PriceParityInfoDto>>> entries = priceParityInfoMap.entrySet();
        List<PriceParityInfoDto> parityInfoDtoList;
        for (Map.Entry<String, List<PriceParityInfoDto>> entry : entries) {
            parityInfoDtoList = entry.getValue();
            BigDecimal summation = BigDecimal.ZERO;
            // 计算合计
            for (PriceParityInfoDto parityInfoDto : parityInfoDtoList) {
                summation = summation.add(parityInfoDto.getTotalPrice());
            }
            for (BiddingResponseResultDto responseResultDto : biddingResponseResultList) {
                for (PriceParityInfoDto priceParityInfoDto : parityInfoDtoList) {
                    priceParityInfoDto.setSummation(summation);
                    if (responseResultDto.getSupllierName().equals(priceParityInfoDto.getSupllierName())) {
                        responseResultDto.setQuotePrice(summation);
                    }

                }
            }
            priceParityInfoDtoList.addAll(parityInfoDtoList);
        }
        return priceParityInfoDtoList;
    }

    @Override
    @Transactional
    public BaseResponse findModifySkuFlag(String biddingNo, String supplierId) {
        if (Lang.isEmpty(biddingNo) || Lang.isEmpty(supplierId)) {
            log.error("查询是否修改sku 参数传入错误");
            return new BaseResponse("参数传入错误");
        }
        JcbdBidding jcbdBidding = jcbdBiddingRepo.findByBiddingCodeAndIsDelete(biddingNo, false);
        if (Lang.isEmpty(jcbdBidding)) {
            log.error("查询是否修改sku  无编号为{}的竞价", biddingNo);
            return new BaseResponse("无此竞价");
        }
        JcbdBdProvider provider = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(jcbdBidding.getId(), supplierId, false);
        if (Lang.isEmpty(provider)) {
            log.error("查询是否修改sku  无供应商id为{}的供应商", supplierId);
            return new BaseResponse("无此供应商");
        }
        List<JcbdBiddingDemand> jcbdBiddingDemands = jcbdBidding.getJcbdBiddingDemands();
        if (Lang.isEmpty(jcbdBiddingDemands)) {
            log.error("查询是否修改sku  编号为{}的竞价下无需求", biddingNo);
            return new BaseResponse("该竞价下无需求");
        }
        for (JcbdBiddingDemand demand : jcbdBiddingDemands) {
            if (Lang.isEmpty(demand.getJcbdBiddingDemandResponses())) {
                log.error("查询是否修改sku  编号为{}的竞价下无需求响应", biddingNo);
                return new BaseResponse("该竞价下无需求响应");
            }
            for (JcbdBiddingDemandResponse response : demand.getJcbdBiddingDemandResponses()) {
                if (response.getResponseProviderId().equals(provider.getId())) {
                    if (Lang.isEmpty(response.getSkuModify())||response.getSkuModify()) {
                        return new BaseResponse<Boolean>(true, "", true);
                    }
                }
            }
        }
        /*jcbdBiddingDemands.forEach(demand ->{
            demand.getJcbdBiddingDemandResponses().forEach(response ->{
                if (response.getResponseProviderId().equals(provider.getId())){
                    if (response.getSkuModify()){
                        return new BaseResponse<Boolean>(true,"",true);
                    }
                }
            });
        });*/
        return new BaseResponse<Boolean>(true, "", false);
    }

    @Override
    @Transactional(readOnly = true)
    public JcbdBdProviderDto getProviderByBiddingNoAndSupplierId(String biddingId, String supplierId) {
        JcbdBdProvider supplierInfo = jcbdBdProviderRepo.findByJcbdBiddingIdAndProviderIdAndIsDelete(biddingId, supplierId, false);
        JcbdBdProviderDto jcbdBdProviderDto = BeanMapper.map(supplierInfo, JcbdBdProviderDto.class);
        return jcbdBdProviderDto;
    }
}
