/*
 * Copyright 2018, OpenCensus Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.nacos.shaded.io.opencensus.trace.internal;

import com.alibaba.nacos.shaded.io.opencensus.common.Internal;
import com.alibaba.nacos.shaded.io.opencensus.internal.Utils;

/**
 * Helper class to convert/cast between for {@link com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent} and {@link
 * com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent}.
 */
@Internal
@SuppressWarnings("deprecation")
public final class BaseMessageEventUtils {
  /**
   * Cast or convert a {@link com.alibaba.nacos.shaded.io.opencensus.trace.BaseMessageEvent} to {@link
   * com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent}.
   *
   * <p>Warning: if the input is a {@code com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent} and contains {@code
   * kernelTimestamp} information, this information will be dropped.
   *
   * @param event the {@code BaseMessageEvent} that is being cast or converted.
   * @return a {@code MessageEvent} representation of the input.
   */
  public static com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent asMessageEvent(
      com.alibaba.nacos.shaded.io.opencensus.trace.BaseMessageEvent event) {
    Utils.checkNotNull(event, "event");
    if (event instanceof com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent) {
      return (com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent) event;
    }
    com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent networkEvent = (com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent) event;
    com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent.Type type =
        (networkEvent.getType() == com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent.Type.RECV)
            ? com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent.Type.RECEIVED
            : com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent.Type.SENT;
    return com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent.builder(type, networkEvent.getMessageId())
        .setUncompressedMessageSize(networkEvent.getUncompressedMessageSize())
        .setCompressedMessageSize(networkEvent.getCompressedMessageSize())
        .build();
  }

  /**
   * Cast or convert a {@link com.alibaba.nacos.shaded.io.opencensus.trace.BaseMessageEvent} to {@link
   * com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent}.
   *
   * @param event the {@code BaseMessageEvent} that is being cast or converted.
   * @return a {@code com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent} representation of the input.
   */
  public static com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent asNetworkEvent(
      com.alibaba.nacos.shaded.io.opencensus.trace.BaseMessageEvent event) {
    Utils.checkNotNull(event, "event");
    if (event instanceof com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent) {
      return (com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent) event;
    }
    com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent messageEvent = (com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent) event;
    com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent.Type type =
        (messageEvent.getType() == com.alibaba.nacos.shaded.io.opencensus.trace.MessageEvent.Type.RECEIVED)
            ? com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent.Type.RECV
            : com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent.Type.SENT;
    return com.alibaba.nacos.shaded.io.opencensus.trace.NetworkEvent.builder(type, messageEvent.getMessageId())
        .setUncompressedMessageSize(messageEvent.getUncompressedMessageSize())
        .setCompressedMessageSize(messageEvent.getCompressedMessageSize())
        .build();
  }

  private BaseMessageEventUtils() {}
}
