/*
 * Decompiled with CFR 0.152.
 */
package com.amadeus.session;

import com.amadeus.session.EncryptingSerializerDeserializer;
import com.amadeus.session.ExecutorFacade;
import com.amadeus.session.JdkSerializerDeserializer;
import com.amadeus.session.RepositoryBackedSession;
import com.amadeus.session.RequestWithSession;
import com.amadeus.session.ResponseWithSessionId;
import com.amadeus.session.SerializerDeserializer;
import com.amadeus.session.SessionConfiguration;
import com.amadeus.session.SessionData;
import com.amadeus.session.SessionFactory;
import com.amadeus.session.SessionNotifier;
import com.amadeus.session.SessionRepository;
import com.amadeus.session.SessionTracking;
import com.codahale.metrics.JmxReporter;
import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import java.io.Closeable;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class SessionManager
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(SessionManager.class);
    static final String SESSIONS_METRIC_PREFIX = "com.amadeus.session";
    private static final String COMMIT_TIMER_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"timer", "commit"});
    private static final String FETCH_TIMER_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"timer", "fetch"});
    private static final String CREATED_SESSIONS_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"created"});
    private static final String DELETED_SESSIONS_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"deleted"});
    private static final String MISSING_SESSIONS_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"missing"});
    private static final String RETRIEVED_SESSIONS_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"retrieved"});
    static final String INVALIDATION_ON_EXPIRY_ERRORS_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"invalidation", "errors", "expiry"});
    static final String INVALIDATION_ERRORS_METRIC = MetricRegistry.name((String)"com.amadeus.session", (String[])new String[]{"invalidation", "errors"});
    static final String SESSION_PROPAGATED = "com.amadeus.session.sessionPropagated";
    protected final SessionRepository repository;
    protected final SessionNotifier notifier;
    protected final SessionTracking tracking;
    protected final SessionFactory factory;
    protected final ExecutorFacade executors;
    protected final SessionConfiguration configuration;
    protected final SerializerDeserializer serializerDeserializer;
    private final Meter createdSessions;
    private final Meter deletedSessions;
    private final Meter retrievedSessions;
    private final Meter invalidationErrors;
    private final Meter invalidationExpiryErrors;
    private final Meter missingSessions;
    private final Timer commitTimer;
    private final Timer fetchTimer;
    private final MetricRegistry monitoring;
    private final ClassLoader classLoader;
    private JmxReporter reporter;

    public SessionManager(ExecutorFacade executors, SessionFactory factory, SessionRepository repository, SessionTracking tracking, SessionNotifier notifier, SessionConfiguration configuration, ClassLoader classLoader) {
        this.repository = repository;
        this.tracking = tracking;
        this.notifier = notifier;
        this.configuration = configuration;
        this.factory = factory;
        this.executors = executors;
        this.classLoader = classLoader;
        this.monitoring = new MetricRegistry();
        this.createdSessions = this.monitoring.meter(CREATED_SESSIONS_METRIC);
        this.deletedSessions = this.monitoring.meter(DELETED_SESSIONS_METRIC);
        this.retrievedSessions = this.monitoring.meter(RETRIEVED_SESSIONS_METRIC);
        this.missingSessions = this.monitoring.meter(MISSING_SESSIONS_METRIC);
        this.invalidationErrors = this.monitoring.meter(INVALIDATION_ERRORS_METRIC);
        this.invalidationExpiryErrors = this.monitoring.meter(INVALIDATION_ON_EXPIRY_ERRORS_METRIC);
        this.commitTimer = this.monitoring.timer(COMMIT_TIMER_METRIC);
        this.fetchTimer = this.monitoring.timer(FETCH_TIMER_METRIC);
        this.serializerDeserializer = configuration.isUsingEncryption() ? new EncryptingSerializerDeserializer() : new JdkSerializerDeserializer();
        this.serializerDeserializer.setSessionManager(this);
        factory.setSessionManager(this);
        repository.setSessionManager(this);
        this.startMonitoring();
    }

    private void startMonitoring() {
        this.executors.startMetrics(this.monitoring);
        this.reporter = JmxReporter.forRegistry((MetricRegistry)this.monitoring).inDomain(this.getJmxDomain()).build();
        this.reporter.start();
    }

    private String getJmxDomain() {
        return "metrics.session." + this.configuration.getNamespace();
    }

    private RepositoryBackedSession fetchSession(String sessionId, boolean updateTimestamp) {
        SessionData sessionData;
        logger.debug("Fetching session from cache, sessionId: '{}'", (Object)sessionId);
        try (Timer.Context ctx = this.fetchTimer.time();){
            sessionData = this.repository.getSessionData(sessionId);
        }
        if (sessionData == null) {
            this.missingSessions.mark();
            logger.debug("Session was not found in cache, considered expired or invalid, sessionId: {}", (Object)sessionId);
            return null;
        }
        sessionData.setRepositoryKeys(this.configuration.getNonCacheable());
        RepositoryBackedSession session = this.factory.build(sessionData);
        this.retrievedSessions.mark();
        if (session.isExpired()) {
            logger.debug("Session was in cache, but it was expired, sessionId: {}", (Object)sessionId);
            if (session.isValid()) {
                this.markSessionDeletion(sessionId);
                session.doInvalidate(true);
            }
            return null;
        }
        if (updateTimestamp) {
            sessionData.setLastAccessedTime(System.currentTimeMillis());
            this.repository.storeSessionData(sessionData);
        }
        return session;
    }

    private RepositoryBackedSession newSession(String sessionId) {
        RepositoryBackedSession session = this.factory.build(new SessionData(sessionId, System.currentTimeMillis(), this.configuration.getMaxInactiveInterval()));
        this.createdSessions.mark();
        this.notifier.sessionCreated(session);
        return session;
    }

    public RepositoryBackedSession getSession(RequestWithSession request, boolean create, String forceId) {
        String id = this.retrieveId(request, forceId);
        this.putIdInLoggingMdc(id);
        request.setRequestedSessionId(id);
        RepositoryBackedSession session = null;
        if (id != null && (session = this.fetchSession(id, true)) == null && !request.isRepositoryChecked()) {
            logger.info("Session with sessionId: '{}' but it was not in repository!", (Object)id);
            request.repositoryChecked();
        }
        if (session == null && create) {
            id = forceId != null ? forceId : this.tracking.newId();
            this.putIdInLoggingMdc(id);
            logger.info("Creating new session with sessionId: '{}'", (Object)id);
            session = this.newSession(id);
        }
        if (session != null) {
            session.checkUsedAndLock();
        }
        return session;
    }

    private void putIdInLoggingMdc(String id) {
        if (this.configuration.isLoggingMdcActive()) {
            if (id == null) {
                MDC.remove((String)this.configuration.getLoggingMdcKey());
            } else {
                MDC.put((String)this.configuration.getLoggingMdcKey(), (String)id);
            }
        }
    }

    private String retrieveId(RequestWithSession request, String forceId) {
        if (forceId != null) {
            return forceId;
        }
        if (request.isIdRetrieved()) {
            return request.getRequestedSessionId();
        }
        return this.tracking.retrieveId(request);
    }

    public void propagateSession(RequestWithSession request, ResponseWithSessionId response) {
        if (request.getAttribute(SESSION_PROPAGATED) == null) {
            request.setAttribute(SESSION_PROPAGATED, Boolean.TRUE);
            this.tracking.propagateSession(request, response);
        }
    }

    public void delete(String sessionId, boolean expired) {
        RepositoryBackedSession session = this.fetchSession(sessionId, false);
        if (session != null) {
            this.markSessionDeletion(sessionId);
            session.doInvalidate(expired);
        } else if (!expired) {
            logger.debug("Session not found in repository for sessionId: '{}'", (Object)sessionId);
        }
    }

    private void markSessionDeletion(String sessionId) {
        logger.info("deleting session with sessionId: '{}'", (Object)sessionId);
        this.deletedSessions.mark();
    }

    public void requestFinished() {
        this.repository.requestFinished();
    }

    public Future<?> submit(String timer, Runnable task) {
        if (timer != null) {
            return this.executors.submit(new RunnableWithTimer(timer, task));
        }
        return this.executors.submit(task);
    }

    public ScheduledFuture<?> schedule(String timer, Runnable task, long period) {
        if (timer != null) {
            return this.executors.scheduleAtFixedRate(new RunnableWithTimer(timer, task), period, period, TimeUnit.SECONDS);
        }
        return this.executors.scheduleAtFixedRate(task, period, period, TimeUnit.SECONDS);
    }

    public void deleteAsync(final String sessionId, final boolean expired) {
        Runnable task = new Runnable(){

            @Override
            public void run() {
                try {
                    SessionManager.this.delete(sessionId, expired);
                }
                catch (Exception e) {
                    logger.error("Exception occured while deleting sessionId '{}'", (Object)sessionId, (Object)e);
                }
            }
        };
        this.submit("delete-async", task);
    }

    public SessionRepository getRepository() {
        return this.repository;
    }

    public SessionNotifier getNotifier() {
        return this.notifier;
    }

    public void invokeCommit(RepositoryBackedSession session) {
        try (Timer.Context ctx = this.commitTimer.time();){
            session.getCommitter().run();
        }
        catch (Exception e) {
            logger.error("Exception occured while commiting sessionId: '" + session.getId() + "'", (Throwable)e);
        }
    }

    public SessionConfiguration getConfiguration() {
        return this.configuration;
    }

    public ClassLoader getSessionClassLoader() {
        return this.classLoader;
    }

    public SerializerDeserializer getSerializerDeserializer() {
        return this.serializerDeserializer;
    }

    public void invalidationConflict(RepositoryBackedSession session, boolean onExpiry) {
        if (onExpiry) {
            this.invalidationExpiryErrors.mark();
            logger.warn("Conflict on removing session: {}", (Object)session.getId());
        } else {
            this.invalidationErrors.mark();
            logger.info("Conflict on removing session during exipre management: {}", (Object)session.getId());
        }
    }

    public MetricRegistry getMetrics() {
        return this.monitoring;
    }

    public String encodeUrl(RequestWithSession request, String url) {
        return this.tracking.encodeUrl(request, url);
    }

    @Override
    public void close() {
        if (this.reporter != null) {
            this.reporter.close();
        }
        if (this.repository.cleanSessionsOnShutdown()) {
            for (String sessionId : this.repository.getOwnedSessionIds()) {
                this.delete(sessionId, false);
            }
        } else {
            logger.warn("Cleanup of sessions on shutdown is not supported by the session repository {} used by session manager {}", (Object)this.repository, (Object)this);
        }
        this.repository.close();
        this.executors.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void switchSessionId(RepositoryBackedSession session) {
        SessionData sessionData = session.getSessionData();
        boolean switched = false;
        SessionData sessionData2 = sessionData;
        synchronized (sessionData2) {
            if (!sessionData.isIdChanged()) {
                String newId = this.tracking.newId();
                logger.info("Switching session id {} to {}", (Object)sessionData.getId(), (Object)newId);
                sessionData.setNewSessionId(newId);
                this.putIdInLoggingMdc(newId);
                switched = true;
            } else {
                logger.warn("Session id was already switched for session: {}", (Object)sessionData);
            }
        }
        if (switched) {
            this.repository.sessionIdChange(sessionData);
            this.factory.sessionIdChange(sessionData);
            this.notifier.sessionIdChanged(session, sessionData.getOldSessionId());
        }
    }

    public String toString() {
        return "SessionManager [namespace=" + this.configuration.getNamespace() + "]";
    }

    public void remove(SessionData sessionData) {
        this.markSessionDeletion(sessionData.getId());
        this.getRepository().remove(sessionData);
    }

    final class RunnableWithTimer
    implements Runnable {
        final Runnable task;
        final Timer timer;

        RunnableWithTimer(String timerName, Runnable task) {
            this.task = task;
            this.timer = SessionManager.this.monitoring.timer(MetricRegistry.name((String)SessionManager.SESSIONS_METRIC_PREFIX, (String[])new String[]{"timers", timerName}));
        }

        @Override
        public void run() {
            try (Timer.Context ctx = this.timer.time();){
                this.task.run();
            }
            catch (Exception e) {
                logger.warn("Unexpected exception occured in time measured session related task in category " + this.timer, (Throwable)e);
            }
        }
    }
}

