/*
 * Copyright (C) 2020 Baidu, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidubce.services.iotdmp;

import com.baidubce.AbstractBceClient;
import com.baidubce.BceClientConfiguration;
import com.baidubce.BceClientException;
import com.baidubce.auth.DefaultBceCredentials;
import com.baidubce.http.Headers;
import com.baidubce.http.HttpMethodName;
import com.baidubce.http.handler.BceErrorResponseHandler;
import com.baidubce.http.handler.BceJsonResponseHandler;
import com.baidubce.http.handler.BceMetadataResponseHandler;
import com.baidubce.http.handler.HttpResponseHandler;
import com.baidubce.internal.InternalRequest;
import com.baidubce.internal.RestartableInputStream;
import com.baidubce.model.AbstractBceRequest;
import com.baidubce.model.AbstractBceResponse;
import com.baidubce.model.GenericAccountRequest;
import com.baidubce.services.iotdmp.model.CommonListRequest;
import com.baidubce.services.iotdmp.model.CommonListResponse;
import com.baidubce.services.iotdmp.model.CommonResult;
import com.baidubce.services.iotdmp.model.alarm.AlarmRecordInfo;
import com.baidubce.services.iotdmp.model.alarm.AlarmRuleInfo;
import com.baidubce.services.iotdmp.model.alarm.BatchDeleteAlarmRuleRequest;
import com.baidubce.services.iotdmp.model.alarm.BatchProcessAlarmRecordRequest;
import com.baidubce.services.iotdmp.model.alarm.CreateAlarmRuleRequest;
import com.baidubce.services.iotdmp.model.alarm.ListAlarmRecordRequest;
import com.baidubce.services.iotdmp.model.alarm.ListAlarmRecordResponse;
import com.baidubce.services.iotdmp.model.alarm.ListAlarmRuleResponse;
import com.baidubce.services.iotdmp.model.alarm.TriggerAlarmRequest;
import com.baidubce.services.iotdmp.model.alarm.UpdateAlarmRuleRequest;
import com.baidubce.services.iotdmp.model.c2c.ComputationSourceResponse;
import com.baidubce.services.iotdmp.model.component.BindComponentRequest;
import com.baidubce.services.iotdmp.model.component.ListBindComponentResponse;
import com.baidubce.services.iotdmp.model.device.AuthRequest;
import com.baidubce.services.iotdmp.model.device.BatchCreateDeviceRequest;
import com.baidubce.services.iotdmp.model.device.BatchDeleteDeviceRequest;
import com.baidubce.services.iotdmp.model.device.CreateDeviceRequest;
import com.baidubce.services.iotdmp.model.device.DeviceInfo;
import com.baidubce.services.iotdmp.model.device.DeviceKey;
import com.baidubce.services.iotdmp.model.device.DeviceResourcesConnectionInfo;
import com.baidubce.services.iotdmp.model.device.ListDeviceKeyRequest;
import com.baidubce.services.iotdmp.model.device.ListDeviceRequest;
import com.baidubce.services.iotdmp.model.device.ListDeviceResponse;
import com.baidubce.services.iotdmp.model.device.ListDeviceStatesResponse;
import com.baidubce.services.iotdmp.model.device.ResourcesRequest;
import com.baidubce.services.iotdmp.model.device.UpdateDeviceRequest;
import com.baidubce.services.iotdmp.model.device.UpdateDeviceStateRequest;
import com.baidubce.services.iotdmp.model.device.evs.AddEvsDeviceRequest;
import com.baidubce.services.iotdmp.model.device.evs.EvsDeviceInfo;
import com.baidubce.services.iotdmp.model.device.evs.EvsDurationRequest;
import com.baidubce.services.iotdmp.model.device.evs.EvsPtzRequest;
import com.baidubce.services.iotdmp.model.device.evs.EvsUrlProtocol;
import com.baidubce.services.iotdmp.model.device.evs.UpdateEvsDeviceRequest;
import com.baidubce.services.iotdmp.model.device.tag.ListTagResponse;
import com.baidubce.services.iotdmp.model.device.topic.ListTopicResponse;
import com.baidubce.services.iotdmp.model.device.topo.DeviceSubsetsFileResponse;
import com.baidubce.services.iotdmp.model.fm.AddConfigRequest;
import com.baidubce.services.iotdmp.model.fm.AddTaskRequest;
import com.baidubce.services.iotdmp.model.fm.ConfigManagementResponse;
import com.baidubce.services.iotdmp.model.fm.ConfigTaskDetailResponse;
import com.baidubce.services.iotdmp.model.fm.ConfigTaskResponse;
import com.baidubce.services.iotdmp.model.fm.ConfigVersionListResponse;
import com.baidubce.services.iotdmp.model.fm.GetConfigVersionResponse;
import com.baidubce.services.iotdmp.model.fm.ProductConfigCommonListResponse;
import com.baidubce.services.iotdmp.model.group.CreateGroupRequest;
import com.baidubce.services.iotdmp.model.group.GroupInfo;
import com.baidubce.services.iotdmp.model.group.ListGroupRequest;
import com.baidubce.services.iotdmp.model.group.ListGroupResponse;
import com.baidubce.services.iotdmp.model.group.UpdateGroupInfoRequest;
import com.baidubce.services.iotdmp.model.instance.ExtensionResourceResponse;
import com.baidubce.services.iotdmp.model.instance.InstanceInfo;
import com.baidubce.services.iotdmp.model.instance.ListInstanceResourceResponse;
import com.baidubce.services.iotdmp.model.instance.ListInstancesResponse;
import com.baidubce.services.iotdmp.model.instance.ResourceSupplier;
import com.baidubce.services.iotdmp.model.instance.ResourceType;
import com.baidubce.services.iotdmp.model.instance.UpdateInstanceResourcePropertiesRequest;
import com.baidubce.services.iotdmp.model.linkage.BatchDeleteLinkageRuleRequest;
import com.baidubce.services.iotdmp.model.linkage.CreateLinkageRuleRequest;
import com.baidubce.services.iotdmp.model.linkage.CreateLinkageRuleResponse;
import com.baidubce.services.iotdmp.model.linkage.LinkageRuleInfo;
import com.baidubce.services.iotdmp.model.linkage.ListLinkageRuleResponse;
import com.baidubce.services.iotdmp.model.linkage.UpdateLinkageRuleRequest;
import com.baidubce.services.iotdmp.model.platform.BatchDeleteRuleChainExternalDestinationRequest;
import com.baidubce.services.iotdmp.model.platform.BatchDeleteRuleChainRequest;
import com.baidubce.services.iotdmp.model.platform.CreateRuleChainExternalDestinationRequest;
import com.baidubce.services.iotdmp.model.platform.CreateRuleChainRequest;
import com.baidubce.services.iotdmp.model.platform.CreateRuleChainResponse;
import com.baidubce.services.iotdmp.model.platform.ListRuleChainDestinationRequest;
import com.baidubce.services.iotdmp.model.platform.ListRuleChainDestinationResponse;
import com.baidubce.services.iotdmp.model.platform.ListRuleChainRequest;
import com.baidubce.services.iotdmp.model.platform.ListRuleChainResponse;
import com.baidubce.services.iotdmp.model.platform.PlatformRuleChainInfo;
import com.baidubce.services.iotdmp.model.platform.UpdatePlatformRuleChainRequest;
import com.baidubce.services.iotdmp.model.platform.UpdateRuleChainStateRequest;
import com.baidubce.services.iotdmp.model.platform.ValidateRuleChainRequest;
import com.baidubce.services.iotdmp.model.platform.ValidateRuleChainResponse;
import com.baidubce.services.iotdmp.model.product.CreateProductInfoRequest;
import com.baidubce.services.iotdmp.model.product.CreateTagRequest;
import com.baidubce.services.iotdmp.model.product.ListProductCategoryResponse;
import com.baidubce.services.iotdmp.model.product.ListProductModelRequest;
import com.baidubce.services.iotdmp.model.product.ListProductModelResponse;
import com.baidubce.services.iotdmp.model.product.ListProductRequest;
import com.baidubce.services.iotdmp.model.product.ListProductResponse;
import com.baidubce.services.iotdmp.model.product.PermanentConnectRequest;
import com.baidubce.services.iotdmp.model.product.ProductInfo;
import com.baidubce.services.iotdmp.model.product.ProductModelInfo;
import com.baidubce.services.iotdmp.model.product.UpdateProductInfoRequest;
import com.baidubce.services.iotdmp.model.product.evs.CreateEvsSpaceRequest;
import com.baidubce.services.iotdmp.model.product.evs.EvsSpaceInfo;
import com.baidubce.services.iotdmp.model.product.evs.UpdateEvsSpaceRequest;
import com.baidubce.services.iotdmp.model.product.feature.DtmlDetailResponse;
import com.baidubce.services.iotdmp.model.product.feature.command.CreateFeatureCommandRequest;
import com.baidubce.services.iotdmp.model.product.feature.command.ListFeatureCommandResponse;
import com.baidubce.services.iotdmp.model.product.feature.command.ProductFeatureCommandInfo;
import com.baidubce.services.iotdmp.model.product.feature.command.UpdateProductCommandRequest;
import com.baidubce.services.iotdmp.model.product.feature.event.CreateFeatureEventRequest;
import com.baidubce.services.iotdmp.model.product.feature.event.ListFeatureEventResponse;
import com.baidubce.services.iotdmp.model.product.feature.event.ProductFeatureEventInfo;
import com.baidubce.services.iotdmp.model.product.feature.event.UpdateProductEventRequest;
import com.baidubce.services.iotdmp.model.product.feature.property.CreateFeaturePropertyRequest;
import com.baidubce.services.iotdmp.model.product.feature.property.ListFeaturePropertyResponse;
import com.baidubce.services.iotdmp.model.product.feature.property.ProductFeaturePropertyInfo;
import com.baidubce.services.iotdmp.model.product.feature.property.UpdateProductPropertyRequest;
import com.baidubce.services.iotdmp.model.product.feature.thing.Thing;
import com.baidubce.services.iotdmp.model.service.ConsumerGroupQueueInfoResponse;
import com.baidubce.services.iotdmp.model.service.ConsumerGroupUserInfoResponse;
import com.baidubce.services.iotdmp.model.service.CreateConsumerGroupRequest;
import com.baidubce.services.iotdmp.model.service.CreateConsumerGroupResponse;
import com.baidubce.services.iotdmp.model.service.ListConsumerGroupResponse;
import com.baidubce.services.iotdmp.model.service.ListSubResponse;
import com.baidubce.services.iotdmp.model.service.ProductSubscriptionResponse;
import com.baidubce.services.iotdmp.model.service.ResetConsumerGroupUserPwdResponse;
import com.baidubce.services.iotdmp.model.service.SendMessageRequest;
import com.baidubce.services.iotdmp.model.service.UpdateProductSubscriptionRequest;
import com.baidubce.services.iotdmp.model.service.rulechain.AvailableMessageTypeResponse;
import com.baidubce.services.iotdmp.model.service.rulechain.TopicDecodeRequest;
import com.baidubce.services.iotdmp.model.service.rulechain.TopicDecodeResponse;
import com.baidubce.services.iotdmp.model.service.rulechain.TopicEncodeRequest;
import com.baidubce.services.iotdmp.model.service.rulechain.TopicEncodeResponse;
import com.baidubce.services.iotdmp.model.shadow.DeviceShadowResponse;
import com.baidubce.services.iotdmp.model.shadow.ListDeviceShadowRequest;
import com.baidubce.services.iotdmp.model.shadow.ListDeviceShadowSnapshotResponse;
import com.baidubce.services.iotdmp.model.shadow.ShadowStatesRequest;
import com.baidubce.services.iotdmp.model.shadow.UpdateDesiredRequest;
import com.baidubce.services.iotdmp.model.userlog.ListUserLogRequest;
import com.baidubce.services.iotdmp.model.userlog.ListUserLogResponse;
import com.baidubce.util.HttpUtils;
import com.baidubce.util.JsonUtils;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

public class PlatformClient extends AbstractBceClient {

    private static final String ENDPOINT = "TBD";
    private static final String VERSION = "v1";
    private static final String CONTENT_TYPE = "application/json;charset=UTF-8";
    private static final String INSTANCE = "instances";
    private static final String EXTENSION = "extensions";
    private static final String AVAILABLE = "available";
    private static final String CONFIG = "config";
    private static final String ENABLED = "enabled";
    private static final String ENABLE = "enable";
    private static final String PLATFORM_RULECHAINS = "platform/rulechains";
    private static final String PLATFORM_LINKAGES = "platform/linkages";
    private static final String BATCH_DELETE = "batchDelete";
    private static final String UPDATE_STATE = "updateState";
    private static final String COMPUTE = "compute";
    private static final String DESTINATIONS = "destinations";
    private static final String SERVICE = "services";
    private static final String C2C = "c2c";
    private static final String DEVICE = "devices";
    private static final String SECRET = "secret";
    private static final String TOPICS = "topics";
    private static final String TAGS = "tags";
    private static final String KEY = "key";
    private static final String EVS = "evs";
    private static final String CHANNEL = "channel";
    private static final String PTZ = "ptz";
    private static final String THUMBNAIL = "thumbnail";
    private static final String RECORDING = "recording";
    private static final String RESOURCES = "resources";
    private static final String STATES = "states";
    private static final String STATE = "state";
    private static final String AUTH = "auth";
    private static final String SHADOWS = "shadows";
    private static final String SHADOW = "shadow";
    private static final String DESIRED = "desired";
    private static final String SIGNED_URL = "signedUrl";
    private static final String PRODUCT = "products";
    private static final String PERMANENT_CONNECT = "permanentConnect";
    private static final String FEATURE = "features";
    private static final String DETAIL = "detail";
    private static final String COMMAND = "commands";
    private static final String EVENT = "events";
    private static final String PROPERTIES = "properties";
    private static final String MESSAGES = "messages";
    private static final String BLINK = "blink";
    private static final String ENCODE = "encode";
    private static final String DECODE = "decode";
    private static final String MESSAGE = "message";
    private static final String TYPES = "types";
    private static final String TYPE = "type";
    private static final String SOURCE = "sources";
    private static final String SINK = "sinks";
    private static final String CONSUMER = "consumers";
    private static final String USER = "user";
    private static final String RESET = "reset";
    private static final String SUBSCRIPTIONS = "subscriptions";
    private static final String BATCH = "batch";
    private static final String SUBSET = "subsets";
    private static final String DELETE = "delete";
    private static final String GATEWAY = "gateway";
    private static final String DOWNWARD = "downward";
    private static final String LOG = "logs";
    private static final String GROUP = "groups";
    private static final String EXPORT = "export";
    private static final String COMPONENTS = "components";
    private static final String VERIFY = "verify";
    private static final String CATEGORIES = "categories";
    private static final String REPOSITORIES = "repositories";
    private static final String MODELS = "models";
    private static final String INFO = "info";
    private static final String ALARMS = "alarms";
    private static final String RULES = "rules";
    private static final String TRIGGER = "trigger";
    private static final String RECORDS = "records";
    private static final String PROCESS = "process";
    private static final String FM = "fm";
    private static final String CONFIG_VERSION = "version";
    private static final String URL = "url";
    private static final String VERSIONS = "versions";
    private static final String TASK = "task";
    private static final String CSV = "csv";
    private static final String IMPORT = "import";

    private static final HttpResponseHandler[] HANDLERS = new HttpResponseHandler[] {
            new BceMetadataResponseHandler(), new BceErrorResponseHandler(), new BceJsonResponseHandler() };

    public PlatformClient(String accessKey, String secretKey) {
        this(new BceClientConfiguration()
                .withCredentials(new DefaultBceCredentials(accessKey, secretKey))
                .withEndpoint(ENDPOINT));
    }

    public PlatformClient(BceClientConfiguration config) {
        super(config.getEndpoint() == null ? config.withEndpoint(ENDPOINT) : config, HANDLERS);
    }

    // Instance API
    public InstanceInfo getInstance(String instanceId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, INSTANCE, instanceId);
        return invokeHttpClient(internalRequest, InstanceInfo.class);
    }

    public ListInstancesResponse listInstances(int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, INSTANCE);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListInstancesResponse.class);
    }

    // extensions
    public ExtensionResourceResponse getConfigExtensionResources(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, EXTENSION, CONFIG);
        return invokeHttpClient(internalRequest, ExtensionResourceResponse.class);
    }

    public ExtensionResourceResponse getConfigExtensionResources(String instanceId, String productKey,
                                                                 String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                DEVICE, instanceId, productKey, deviceName, EXTENSION, CONFIG);
        return invokeHttpClient(internalRequest, ExtensionResourceResponse.class);
    }

    public ExtensionResourceResponse getEnabledExtensionResources(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT,
                instanceId, productKey, EXTENSION, ENABLED);
        return invokeHttpClient(internalRequest, ExtensionResourceResponse.class);
    }


    public ExtensionResourceResponse getEnabledExtensionResources(String instanceId, String productKey,
                                                                  String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                DEVICE, instanceId, productKey, deviceName, EXTENSION, ENABLED);
        return invokeHttpClient(internalRequest, ExtensionResourceResponse.class);
    }

    public ListInstanceResourceResponse listInstanceResources(String instanceId, ResourceSupplier supplier) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                INSTANCE, instanceId, RESOURCES);
        internalRequest.addParameter("supplier", supplier.name());
        return invokeHttpClient(internalRequest, ListInstanceResourceResponse.class);
    }

    public void updateResourceProperties(String instanceId, UpdateInstanceResourcePropertiesRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                INSTANCE, instanceId, RESOURCES);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void enableResource(String instanceId, ResourceSupplier supplier, ResourceType resourceType) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                INSTANCE, instanceId, RESOURCES, ENABLE);
        internalRequest.addParameter("supplier", supplier.name());
        internalRequest.addParameter("resourceType", resourceType.name());

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void verifyResourceProperties(String instanceId, UpdateInstanceResourcePropertiesRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                INSTANCE, instanceId, RESOURCES, VERIFY);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // Platform API
    public CreateRuleChainResponse createRuleChain (String instanceId, CreateRuleChainRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_RULECHAINS, instanceId);
        return invokeHttpClient(internalRequest, CreateRuleChainResponse.class);
    }

    public void deleteRuleChain (String instanceId, BatchDeleteRuleChainRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_RULECHAINS, instanceId, BATCH_DELETE);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateRuleChain (String instanceId, String rulechainId,
                                         UpdatePlatformRuleChainRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, PLATFORM_RULECHAINS, instanceId, rulechainId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateRuleChainState (String instanceId, String rulechainId,
                                              UpdateRuleChainStateRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, PLATFORM_RULECHAINS,
                instanceId, rulechainId, UPDATE_STATE);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public PlatformRuleChainInfo getRuleChain (String instanceId, String rulechainId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PLATFORM_RULECHAINS, instanceId, rulechainId);
        return invokeHttpClient(internalRequest, PlatformRuleChainInfo.class);
    }

    public ListRuleChainResponse listRuleChain (String instanceId, ListRuleChainRequest request) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PLATFORM_RULECHAINS, instanceId);
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        if (StringUtils.isNotEmpty(request.getOrder())) {
            internalRequest.addParameter("order", request.getOrder());
        }
        if (StringUtils.isNotEmpty(request.getOrderBy())) {
            internalRequest.addParameter("orderBy", request.getOrderBy());
        }
        if (StringUtils.isNotEmpty(request.getName())) {
            internalRequest.addParameter("name", request.getName());
        }
        if (StringUtils.isNotEmpty(request.getState())) {
            internalRequest.addParameter("state", request.getState());
        }
        if (StringUtils.isNotEmpty(request.getStatus())) {
            internalRequest.addParameter("status", request.getStatus());
        }
        return invokeHttpClient(internalRequest, ListRuleChainResponse.class);
    }

    public ValidateRuleChainResponse validateRuleChain (ValidateRuleChainRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_RULECHAINS, COMPUTE);
        return invokeHttpClient(internalRequest, ValidateRuleChainResponse.class);
    }

    public CommonResult createRuleChainExternalDestination (String instanceId,
                                                              CreateRuleChainExternalDestinationRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_RULECHAINS, instanceId, DESTINATIONS);
        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    public ListRuleChainDestinationResponse listRuleChainDestinations (String instanceId,
                                              ListRuleChainDestinationRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, PLATFORM_RULECHAINS, instanceId, DESTINATIONS);
        internalRequest.addParameter("sourceType", request.getSourceType());
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        if (StringUtils.isNotEmpty(request.getOrder())) {
            internalRequest.addParameter("order", request.getOrder());
        }
        if (StringUtils.isNotEmpty(request.getOrderBy())) {
            internalRequest.addParameter("orderBy", request.getOrderBy());
        }
        if (StringUtils.isNotEmpty(request.getRegion())) {
            internalRequest.addParameter("region", request.getRegion());
        }
        if (StringUtils.isNotEmpty(request.getType())) {
            internalRequest.addParameter("type", request.getType());
        }
        return invokeHttpClient(internalRequest, ListRuleChainDestinationResponse.class);
    }

    public void batchDeleteRuleChainExternalDestinations (String instanceId,
                                                          BatchDeleteRuleChainExternalDestinationRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_RULECHAINS, instanceId, DESTINATIONS, BATCH_DELETE);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public CommonResult validateRuleChainDestinationConnect (String instanceId, String destId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST, PLATFORM_RULECHAINS,
                instanceId, DESTINATIONS, destId);
        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    // C2C API
    public ComputationSourceResponse getC2CSource(String instanceId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE, instanceId, C2C);
        return invokeHttpClient(internalRequest, ComputationSourceResponse.class);
    }

    public void updateC2CDownwardState(String instanceId, boolean state) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, SERVICE, instanceId, C2C, DOWNWARD);
        internalRequest.addParameter("state", Boolean.toString(state));
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // Device API
    public DeviceInfo createDevice(String instanceId, String productKey, CreateDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE, instanceId, productKey);
        return invokeHttpClient(internalRequest, DeviceInfo.class);
    }

    public void importCsvCreateDevice(String instanceId, String productKey, File file) {
        InternalRequest internalRequest = createUploadRequest(
                "importFile", file, HttpMethodName.POST, DEVICE, instanceId, productKey, CSV, IMPORT);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void deleteDevice(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE, DEVICE, instanceId, productKey, deviceName);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void batchDeleteDevice(String instanceId, BatchDeleteDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, DEVICE, instanceId, BATCH, DELETE);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public DeviceInfo getDevice(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE, instanceId, productKey, deviceName);
        return invokeHttpClient(internalRequest, DeviceInfo.class);
    }

    public ListDeviceResponse getDeviceList(String instanceId, ListDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, DEVICE, instanceId);
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        internalRequest.addParameter("cursor", request.getCursor());
        if (StringUtils.isNotEmpty(request.getProductKey())) {
            internalRequest.addParameter("productKey", request.getProductKey());
        }
        if (StringUtils.isNotEmpty(request.getAlias())) {
            internalRequest.addParameter("alias", request.getAlias());
        }
        return invokeHttpClient(internalRequest, ListDeviceResponse.class);
    }

    public void resetDeviceSecret(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, DEVICE,
                instanceId, productKey, deviceName, SECRET);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void batchCreateDevice(String instanceId, String productKey, BatchCreateDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE,
                instanceId, productKey, BATCH);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public DeviceInfo updateDevice(String instanceId, String productKey,
                                   String deviceName, UpdateDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, DEVICE, instanceId, productKey, deviceName);
        return invokeHttpClient(internalRequest, DeviceInfo.class);
    }

    public ListDeviceStatesResponse getDeviceStates(String instanceId, ListDeviceKeyRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE, instanceId, STATES);
        return invokeHttpClient(internalRequest, ListDeviceStatesResponse.class);
    }

    // topic
    public ListTopicResponse getDeviceTopic(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, TOPICS);
        return invokeHttpClient(internalRequest, ListTopicResponse.class);
    }

    // tag
    public ListTagResponse createDeviceTag(String instanceId, String productKey,
                                     String deviceName, CreateTagRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE,
                instanceId, productKey, deviceName, TAGS);
        return invokeHttpClient(internalRequest, ListTagResponse.class);
    }

    public void deleteDeviceTag(String instanceId, String productKey,
                                     String deviceName, String key) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE, DEVICE,
                instanceId, productKey, deviceName, TAGS, key);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ListTagResponse getDeviceTagList(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, TAGS);
        return invokeHttpClient(internalRequest, ListTagResponse.class);
    }

    // video
    public EvsSpaceInfo getEvs(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, EVS);
        return invokeHttpClient(internalRequest, EvsSpaceInfo.class);
    }

    public EvsDeviceInfo getEvs(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, EVS);
        return invokeHttpClient(internalRequest, EvsDeviceInfo.class);
    }

    public void createEvs(String instanceId, String productKey,
                                    CreateEvsSpaceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, EVS);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void createEvs(String instanceId, String productKey,
                                   String deviceName, AddEvsDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE,
                instanceId, productKey, deviceName, EVS);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public CommonResult getEvsChannelUrl(String instanceId, String productKey,
                                    String deviceName, String channelId, EvsUrlProtocol protocol) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, EVS, CHANNEL, channelId);

        internalRequest.addParameter("protocol", protocol.name());

        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    public void getEvsChannelPtz(String instanceId, String productKey,
                                           String deviceName, String channelId,
                                           EvsPtzRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, DEVICE,
                instanceId, productKey, deviceName, EVS, CHANNEL, channelId, PTZ);
        internalRequest.addParameter("ptz", request.getPtzCommand());
        if (request.getSpeed() != null) {
            internalRequest.addParameter("speed", Integer.toString(request.getSpeed()));
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public CommonResult getEvsChannel(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, EVS, CHANNEL);
        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    public CommonResult getEvsThumbnail(String instanceId, String productKey,
                                        String deviceName, EvsDurationRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, EVS, THUMBNAIL);
        internalRequest.addParameter("begin", Integer.toString(request.getBegin()));
        internalRequest.addParameter("end", Integer.toString(request.getEnd()));
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    public CommonResult getEvsRecording(String instanceId, String productKey,
                                        String deviceName, EvsDurationRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, EVS, RECORDING);
        internalRequest.addParameter("begin", Integer.toString(request.getBegin()));
        internalRequest.addParameter("end", Integer.toString(request.getEnd()));
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    public void auth(String instanceId, String productKey,
                                              String deviceName, AuthRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE,
                instanceId, productKey, deviceName, AUTH);
        internalRequest.addHeader("signature", request.getSignature());
        internalRequest.addHeader("expiryTime", Long.toString(request.getExpiryTime()));
        if (request.getAlgorithmType() != null) {
            internalRequest.addHeader("algorithmType", request.getAlgorithmType().name());
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public DeviceResourcesConnectionInfo getResourcesInfo(String instanceId, String productKey,
                                                          String deviceName, ResourcesRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, DEVICE,
                instanceId, productKey, deviceName, RESOURCES);
        return invokeHttpClient(internalRequest, DeviceResourcesConnectionInfo.class);
    }

    public void updateDeviceStates(String instanceId, String productKey,
                                       String deviceName, UpdateDeviceStateRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, DEVICE,
                instanceId, productKey, deviceName, STATES);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // DeviceShadow API
    public DeviceShadowResponse getDeviceShadow(String instanceId, String productKey,
                                                String deviceName, ListDeviceShadowRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET,
                SHADOWS, instanceId, productKey, deviceName);
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        if (StringUtils.isNotEmpty(request.getPropertyName())) {
            internalRequest.addParameter("propertyName", request.getPropertyName());
        }
        return invokeHttpClient(internalRequest, DeviceShadowResponse.class);
    }

    public void updateDeviceShadowState(String instanceId, String productKey,
                                                        boolean shadowState) {
        ShadowStatesRequest request = new ShadowStatesRequest();
        request.setShadowState(shadowState);
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                SHADOWS, instanceId, productKey, STATES);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateDesired(String instanceId, String productKey,
                              String deviceName,  UpdateDesiredRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                SHADOWS, instanceId, productKey, deviceName, DESIRED);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ListDeviceShadowSnapshotResponse listShadow(String instanceId, ListDeviceKeyRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                SHADOWS, instanceId, SHADOW, BATCH);
        return invokeHttpClient(internalRequest, ListDeviceShadowSnapshotResponse.class);
    }

    // EVS
    public CommonResult getEvsStream(String domain, String app,
                                     String stream, EvsUrlProtocol protocol) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest() , HttpMethodName.GET,
                EVS, domain, app, stream, SIGNED_URL);
        internalRequest.addParameter("protocol", protocol.name());
        return invokeHttpClient(internalRequest, CommonResult.class);
    }

    // Product API
    public ProductInfo createProduct(String instanceId, CreateProductInfoRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId);
        return invokeHttpClient(internalRequest, ProductInfo.class);
    }

    public void resetSecret(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT,
                PRODUCT, instanceId, productKey, SECRET);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ProductInfo updateProduct(String instanceId, String productKey,
                                        UpdateProductInfoRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                PRODUCT, instanceId, productKey);
        return invokeHttpClient(internalRequest, ProductInfo.class);
    }

    public ListProductResponse getProductList(String instanceId, ListProductRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, PRODUCT, instanceId);
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        if (StringUtils.isNotEmpty(request.getProductName())) {
            internalRequest.addParameter("productName", request.getProductName());
        }
        if (StringUtils.isNotEmpty(request.getTagKey())) {
            internalRequest.addParameter("tagKey", request.getTagKey());
        }
        if (StringUtils.isNotEmpty(request.getTagValue())) {
            internalRequest.addParameter("tagValue", request.getTagValue());
        }
        if (request.getDeviceType() != null) {
            internalRequest.addParameter("deviceType", request.getDeviceType().toString());
        }
        return invokeHttpClient(internalRequest, ListProductResponse.class);
    }

    public ProductInfo getProduct(String instanceId,  String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey);
        return invokeHttpClient(internalRequest, ProductInfo.class);
    }

    public void deleteProduct(String instanceId,  String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE,
                PRODUCT, instanceId, productKey);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updatePermanentConnect(String instanceId, String productKey,
                                                 boolean permanentConnect) {
        PermanentConnectRequest request = new PermanentConnectRequest(permanentConnect);
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, PERMANENT_CONNECT);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // tag
    public ListTagResponse createProductTag(String instanceId, String productKey,
                                            CreateTagRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, TAGS);
        return invokeHttpClient(internalRequest, ListTagResponse.class);
    }

    public void deleteProductTag(String instanceId, String productKey,
                                            String key) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE,
                PRODUCT, instanceId, productKey, TAGS, key);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ListTagResponse getProductTagList(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, TAGS);
        return invokeHttpClient(internalRequest, ListTagResponse.class);
    }

    // topic
    public ListTopicResponse getTopics(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, TOPICS);
        return invokeHttpClient(internalRequest, ListTopicResponse.class);
    }

    // function
    public DtmlDetailResponse getDTMLDetail(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                PRODUCT, instanceId, productKey, FEATURE, DETAIL);
        return invokeHttpClient(internalRequest, DtmlDetailResponse.class);
    }

    public void importDTMLDetail(String instanceId, String productKey, Thing thing) {
        InternalRequest internalRequest = createRequest(
                thing, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, FEATURE, BATCH);
        invokeHttpClient(internalRequest, CommonResult.class);
    }

    // feature command
    public ProductFeatureCommandInfo createFeatureCommand(String instanceId, String productKey,
                                                          CreateFeatureCommandRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, FEATURE, COMMAND);
        return invokeHttpClient(internalRequest, ProductFeatureCommandInfo.class);
    }

    public ProductFeatureCommandInfo updateFeatureCommand(String instanceId, String productKey,
                                                          String name, UpdateProductCommandRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                PRODUCT, instanceId, productKey, FEATURE, COMMAND, name);
        return invokeHttpClient(internalRequest, ProductFeatureCommandInfo.class);
    }

    public void deleteFeatureCommand(String instanceId, String productKey,
                                                          String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE,
                PRODUCT, instanceId, productKey, FEATURE, COMMAND, name);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ProductFeatureCommandInfo getFeatureCommand(String instanceId, String productKey,
                                               String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, FEATURE, COMMAND, name);
        return invokeHttpClient(internalRequest, ProductFeatureCommandInfo.class);
    }

    public ListFeatureCommandResponse getFeatureCommandList(String instanceId, String productKey,
                                                            int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, FEATURE, COMMAND);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListFeatureCommandResponse.class);
    }

    // feature event
    public ProductFeatureEventInfo createFeatureEvent(String instanceId, String productKey,
                                                      CreateFeatureEventRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, FEATURE, EVENT);
        return invokeHttpClient(internalRequest, ProductFeatureEventInfo.class);
    }

    public ProductFeatureEventInfo updateFeatureEvent(String instanceId, String productKey,
                                                      String name, UpdateProductEventRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                PRODUCT, instanceId, productKey, FEATURE, EVENT, name);
        return invokeHttpClient(internalRequest, ProductFeatureEventInfo.class);
    }

    public void deleteFeatureEvent(String instanceId, String productKey,
                                             String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE,
                PRODUCT, instanceId, productKey, FEATURE, EVENT, name);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ProductFeatureEventInfo getFeatureEvent(String instanceId, String productKey, String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, FEATURE, EVENT, name);
        return invokeHttpClient(internalRequest, ProductFeatureEventInfo.class);
    }

    public ListFeatureEventResponse getFeatureEventList(String instanceId, String productKey,
                                                        int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, FEATURE, EVENT);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListFeatureEventResponse.class);
    }

    // feature property
    public ProductFeaturePropertyInfo createFeatureProperty(String instanceId, String productKey,
                                                            CreateFeaturePropertyRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST,
                PRODUCT, instanceId, productKey, FEATURE, PROPERTIES);
        return invokeHttpClient(internalRequest, ProductFeaturePropertyInfo.class);
    }

    public ProductFeaturePropertyInfo updateFeatureProperty(String instanceId, String productKey,
                                                      String name, UpdateProductPropertyRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                PRODUCT, instanceId, productKey, FEATURE, PROPERTIES, name);
        return invokeHttpClient(internalRequest, ProductFeaturePropertyInfo.class);
    }

    public void deleteFeatureProperty(String instanceId, String productKey,
                                             String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE,
                PRODUCT, instanceId, productKey, FEATURE, PROPERTIES, name);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ProductFeaturePropertyInfo getFeatureProperty(String instanceId, String productKey,
                                                   String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, FEATURE, PROPERTIES, name);
        return invokeHttpClient(internalRequest, ProductFeaturePropertyInfo.class);
    }

    public ListFeaturePropertyResponse getFeaturePropertyList(String instanceId, String productKey,
                                                                       int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, productKey, FEATURE, PROPERTIES);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListFeaturePropertyResponse.class);
    }

    // video
    public void updateEvs(String instanceId, String productKey,
                                    UpdateEvsSpaceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                PRODUCT, instanceId, productKey, EVS);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateEvs(String instanceId, String productKey, String deviceName,
                          UpdateEvsDeviceRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT,
                DEVICE, instanceId, productKey, deviceName, EVS);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // topo

    public ListProductResponse getSubsets(String instanceId, String productKey,
                                                       int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, SUBSET, productKey);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListProductResponse.class);
    }

    public ListDeviceResponse getSubsets(String instanceId, String productKey, String deviceName,
                           int pageNo, int pageSize, String name) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                DEVICE, instanceId, productKey, deviceName, SUBSET);
        if (StringUtils.isNotBlank(name)) {
            internalRequest.addParameter("name", name);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListDeviceResponse.class);
    }

    public void deleteSubsets(String instanceId, String productKey,
                                                       List<String> subProductKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                PRODUCT, instanceId, SUBSET, productKey, DELETE);
        String jsonStr = JsonUtils.toJsonString(subProductKey);
        try {
            byte[] content = jsonStr.getBytes(DEFAULT_ENCODING);
            internalRequest.setContent(RestartableInputStream.wrap(content));
            internalRequest.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void deleteSubsets(String instanceId, String productKey, String deviceName,
                              List<DeviceKey> subDeviceKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                DEVICE, instanceId, productKey, deviceName, SUBSET, DELETE);
        String jsonStr = JsonUtils.toJsonString(subDeviceKey);
        try {
            byte[] content = jsonStr.getBytes(DEFAULT_ENCODING);
            internalRequest.setContent(RestartableInputStream.wrap(content));
            internalRequest.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ListProductResponse getAllSubsets(String instanceId, String productKey, String subProductName,
                                             int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                PRODUCT, instanceId, SUBSET);

        internalRequest.addParameter("productKey", productKey);
        if (StringUtils.isNotBlank(subProductName)) {
            internalRequest.addParameter("subProductName", subProductName);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListProductResponse.class);
    }

    public ListDeviceResponse getAllSubsets(String instanceId, String productKey,
                                             String subProductKey, String deviceName, String name,
                                             int pageNo, int pageSize) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET,
                DEVICE, instanceId, productKey, SUBSET);

        internalRequest.addParameter("subProductKey", subProductKey);
        internalRequest.addParameter("deviceName", deviceName);
        if (StringUtils.isNotBlank(name)) {
            internalRequest.addParameter("name", name);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListDeviceResponse.class);
    }

    public void addSubsets(String instanceId, String productKey,
                                     List<String> subProductKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                PRODUCT, instanceId, SUBSET, productKey);
        String jsonStr = JsonUtils.toJsonString(subProductKey);
        try {
            byte[] content = jsonStr.getBytes(DEFAULT_ENCODING);
            internalRequest.setContent(RestartableInputStream.wrap(content));
            internalRequest.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void addSubsets(String instanceId, String productKey, String deviceName,
                           List<DeviceKey> subDeviceKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                DEVICE, instanceId, productKey, deviceName, SUBSET);
        String jsonStr = JsonUtils.toJsonString(subDeviceKey);
        try {
            byte[] content = jsonStr.getBytes(DEFAULT_ENCODING);
            internalRequest.setContent(RestartableInputStream.wrap(content));
            internalRequest.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void importSubsets(String instanceId, String productKey, String deviceName,
                                                   File file) {
        InternalRequest internalRequest = createUploadRequest(
                "importFile", file, HttpMethodName.POST,
                DEVICE, instanceId, productKey, deviceName, SUBSET, IMPORT);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public DeviceSubsetsFileResponse exportSubsets(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST,
                DEVICE, instanceId, productKey, deviceName, SUBSET, EXPORT);
        return invokeHttpClient(internalRequest, DeviceSubsetsFileResponse.class);
    }

    // ruleChain API
    public TopicEncodeResponse topicEncode(TopicEncodeRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, SERVICE,
                BLINK, TOPICS, ENCODE);
        return invokeHttpClient(internalRequest, TopicEncodeResponse.class);
    }

    public TopicDecodeResponse topicDecode(TopicDecodeRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, SERVICE,
                BLINK, TOPICS, DECODE);
        return invokeHttpClient(internalRequest, TopicDecodeResponse.class);
    }

    public AvailableMessageTypeResponse getSourceTypes(String instanceId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, MESSAGES, TYPES, SOURCE);
        return invokeHttpClient(internalRequest, AvailableMessageTypeResponse.class);
    }

    public AvailableMessageTypeResponse getSinkTypes(String instanceId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, MESSAGES, TYPES, SINK);
        return invokeHttpClient(internalRequest, AvailableMessageTypeResponse.class);
    }

    // service API
    public ConsumerGroupUserInfoResponse getUserInfo(String instanceId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, CONSUMER, USER);
        return invokeHttpClient(internalRequest, ConsumerGroupUserInfoResponse.class);
    }

    public void sendMessage(String instanceId, String productKey,
                                      String deviceName, SendMessageRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, SERVICE,
                instanceId, productKey, deviceName, MESSAGE);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public CreateConsumerGroupResponse createConsumerGroup(String instanceId, String name) {
        CreateConsumerGroupRequest request = new CreateConsumerGroupRequest(name);
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, SERVICE,
                instanceId, CONSUMER);
        return invokeHttpClient(internalRequest, CreateConsumerGroupResponse.class);
    }

    public void deleteConsumerGroup(String instanceId, String consumerGroupId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE, SERVICE,
                instanceId, CONSUMER, consumerGroupId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ListConsumerGroupResponse getConsumerGroupList(String instanceId,
                                                          CommonListRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET,
                SERVICE, instanceId, CONSUMER);
        if (request.getPageNo() != null) {
            internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        }
        if (request.getPageSize() != null) {
            internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        }
        if (StringUtils.isNotEmpty(request.getOrderBy())) {
            internalRequest.addParameter("orderBy", request.getOrderBy());
        }
        if (StringUtils.isNotEmpty(request.getOrder())) {
            internalRequest.addParameter("order", request.getOrder());
        }
        return invokeHttpClient(internalRequest, ListConsumerGroupResponse.class);
    }

    public ResetConsumerGroupUserPwdResponse resetUserPwd(String instanceId, String username) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, SERVICE,
                instanceId, CONSUMER, username, RESET);
        return invokeHttpClient(internalRequest, ResetConsumerGroupUserPwdResponse.class);
    }

    public ConsumerGroupQueueInfoResponse getQueueInfo(String instanceId, String consumerGroupId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, CONSUMER, consumerGroupId);
        return invokeHttpClient(internalRequest, ConsumerGroupQueueInfoResponse.class);
    }

    public ListSubResponse getSubList(String instanceId, CommonListRequest request) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, SUBSCRIPTIONS);
        if (request.getPageNo() != null) {
            internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        }
        if (request.getPageSize() != null) {
            internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        }
        if (StringUtils.isNotEmpty(request.getOrderBy())) {
            internalRequest.addParameter("orderBy", request.getOrderBy());
        }
        if (StringUtils.isNotEmpty(request.getOrder())) {
            internalRequest.addParameter("order", request.getOrder());
        }
        return invokeHttpClient(internalRequest, ListSubResponse.class);
    }

    @Deprecated
    public ProductSubscriptionResponse getSub(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, productKey, SUBSCRIPTIONS);
        return invokeHttpClient(internalRequest, ProductSubscriptionResponse.class);
    }

    public ProductSubscriptionResponse getSubTopics(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, productKey, SUBSCRIPTIONS, TOPICS);
        return invokeHttpClient(internalRequest, ProductSubscriptionResponse.class);
    }

    @Deprecated
    public void updateSub(String instanceId, String productKey,
                                    UpdateProductSubscriptionRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, SERVICE,
                instanceId, productKey, SUBSCRIPTIONS);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateSubTopics(String instanceId, String productKey,
                          UpdateProductSubscriptionRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, SERVICE,
                instanceId, productKey, SUBSCRIPTIONS, TOPICS);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateSubState(String instanceId, String productKey,
                                         boolean state) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST, SERVICE,
                instanceId, productKey, SUBSCRIPTIONS);
        internalRequest.addParameter("state", Boolean.toString(state));
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ProductSubscriptionResponse getMessageType(String instanceId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, MESSAGE, TYPE);
        return invokeHttpClient(internalRequest, ProductSubscriptionResponse.class);
    }

    // gateway
    public void updateGatewayState(String instanceId, String productKey, String deviceName, boolean state) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, SERVICE,
                instanceId, productKey, deviceName, GATEWAY);
        internalRequest.addParameter("state", Boolean.toString(state));
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ComputationSourceResponse getGatewayInfo(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, SERVICE,
                instanceId, productKey, deviceName, GATEWAY);
        return invokeHttpClient(internalRequest, ComputationSourceResponse.class);
    }

    public void resetGatewaySecret(String instanceId, String productKey, String deviceName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, SERVICE,
                instanceId, productKey, deviceName, GATEWAY, RESET);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // userlog
    public ListUserLogResponse getLogList(String instanceId, ListUserLogRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, LOG, instanceId);

        internalRequest.addParameter("logType", request.getLogType());
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        if (StringUtils.isNotEmpty(request.getLogSubType())) {
            internalRequest.addParameter("logSubType", request.getLogSubType());
        }
        if (StringUtils.isNotEmpty(request.getFlag())) {
            internalRequest.addParameter("flag", request.getFlag());
        }
        if (StringUtils.isNotEmpty(request.getProductKey())) {
            internalRequest.addParameter("productKey", request.getProductKey());
        }
        if (StringUtils.isNotEmpty(request.getDeviceName())) {
            internalRequest.addParameter("deviceName", request.getDeviceName());
        }
        if (StringUtils.isNotEmpty(request.getKeyword())) {
            internalRequest.addParameter("keyword", request.getKeyword());
        }
        if (request.getBeginTime() != null) {
            internalRequest.addParameter("beginTime", Long.toString(request.getBeginTime()));
        }
        if (request.getEndTime() != null) {
            internalRequest.addParameter("endTime", Long.toString(request.getEndTime()));
        }
        return invokeHttpClient(internalRequest, ListUserLogResponse.class);
    }

    // group
    public GroupInfo createGroup(String instanceId, CreateGroupRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, GROUP, instanceId);
        return invokeHttpClient(internalRequest, GroupInfo.class);
    }

    public ListGroupResponse getGroupList(String instanceId, ListGroupRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, GROUP, instanceId);
        if (StringUtils.isNotEmpty(request.getSuperGroupId())) {
            internalRequest.addParameter("superGroupId", request.getSuperGroupId());
        }
        if (StringUtils.isNotEmpty(request.getRootGroupId())) {
            internalRequest.addParameter("rootGroupId", request.getRootGroupId());
        }
        if (StringUtils.isNotEmpty(request.getKeyword())) {
            internalRequest.addParameter("keyword", request.getKeyword());
        }
        if (request.getPageNo() != null) {
            internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        }
        if (request.getPageSize() != null) {
            internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        }
        return invokeHttpClient(internalRequest, ListGroupResponse.class);
    }

    public void deleteGroup(String instanceId, String groupId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE, GROUP, instanceId, groupId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public GroupInfo updateGroup(String instanceId, String groupId, UpdateGroupInfoRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, GROUP, instanceId, groupId);
        return invokeHttpClient(internalRequest, GroupInfo.class);
    }

    public GroupInfo getGroup(String instanceId, String groupId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, GROUP, instanceId, groupId);
        return invokeHttpClient(internalRequest, GroupInfo.class);
    }

    public void addDeviceToGroup(String instanceId, String groupId, List<DeviceKey> listDeviceKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST, GROUP, instanceId, groupId, DEVICE);
        String jsonStr = JsonUtils.toJsonString(listDeviceKey);
        try {
            byte[] content = jsonStr.getBytes(DEFAULT_ENCODING);
            internalRequest.setContent(RestartableInputStream.wrap(content));
            internalRequest.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void deleteDeviceFromGroup(String instanceId, String groupId, List<DeviceKey> listDeviceKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, GROUP, instanceId, groupId, DEVICE, DELETE);
        String jsonStr = JsonUtils.toJsonString(listDeviceKey);
        try {
            byte[] content = jsonStr.getBytes(DEFAULT_ENCODING);
            internalRequest.setContent(RestartableInputStream.wrap(content));
            internalRequest.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // component
    public ListBindComponentResponse bindProductComponents(String instanceId, String productKey,
                                                           BindComponentRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PRODUCT, instanceId, productKey, COMPONENTS);
        return invokeHttpClient(internalRequest, ListBindComponentResponse.class);
    }

    public void unbindProductComponent(String instanceId, String productKey,
                                                     String bindName, BindComponentRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.DELETE, PRODUCT, instanceId, productKey, COMPONENTS, bindName);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ListBindComponentResponse listProductComponents(String instanceId, String productKey) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, instanceId, productKey, COMPONENTS);
        return invokeHttpClient(internalRequest, ListBindComponentResponse.class);
    }

    public ListBindComponentResponse listDeviceComponents(String instanceId, String productKey,
                                                         String deviceName, String bindName) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, DEVICE,
                instanceId, productKey, deviceName, COMPONENTS);
        if (StringUtils.isNotEmpty(bindName)) {
            internalRequest.addParameter("bindName", bindName);
        }
        return invokeHttpClient(internalRequest, ListBindComponentResponse.class);
    }

    // linkage
    public CreateLinkageRuleResponse createLinkageRule(String instanceId, CreateLinkageRuleRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_LINKAGES, instanceId);
        return invokeHttpClient(internalRequest, CreateLinkageRuleResponse.class);
    }

    public void deleteLinkageRule(String instanceId, BatchDeleteLinkageRuleRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, PLATFORM_LINKAGES, instanceId, BATCH_DELETE);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void updateLinkageRule(String instanceId, String ruleId, UpdateLinkageRuleRequest request) {
        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, PLATFORM_LINKAGES, instanceId, ruleId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public LinkageRuleInfo getLinkageRule(String instanceId, String ruleId) {
        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PLATFORM_LINKAGES, instanceId, ruleId);
        return invokeHttpClient(internalRequest, LinkageRuleInfo.class);
    }

    public ListLinkageRuleResponse listLinkageRule(String instanceId, int pageNo, int pageSize, String name) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PLATFORM_LINKAGES, instanceId);
        if (StringUtils.isNotEmpty(name)) {
            internalRequest.addParameter("name", name);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListLinkageRuleResponse.class);
    }

    public void updateLinkageRuleState(String instanceId, String ruleId, boolean state) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, PLATFORM_LINKAGES, instanceId, ruleId, STATE);
        internalRequest.addParameter("state", Boolean.toString(state));

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // products categories
    public ListProductCategoryResponse listProductCategory() {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, CATEGORIES);

        return invokeHttpClient(internalRequest, ListProductCategoryResponse.class);
    }

    // products repositories
    public ListProductModelResponse listProductModel(ListProductModelRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS);
        if (StringUtils.isNotEmpty(request.getProductName())) {
            internalRequest.addParameter("productName", request.getProductName());
        }
        internalRequest.addParameter("pageNo", Integer.toString(request.getPageNo()));
        internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        return invokeHttpClient(internalRequest, ListProductModelResponse.class);
    }

    public ProductModelInfo getMainProductInfo(String modelId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS, modelId, INFO);

        return invokeHttpClient(internalRequest, ProductModelInfo.class);
    }

    public ListBindComponentResponse getMainProductComponents(String modelId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS, modelId, COMPONENTS);

        return invokeHttpClient(internalRequest, ListBindComponentResponse.class);
    }

    public CommonListResponse<ProductFeaturePropertyInfo> getMainProductProperties(String modelId,
                                                                                   int pageNo, int pageSize) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS,
                modelId, FEATURE, PROPERTIES);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, CommonListResponse.class);
    }

    public CommonListResponse<ProductFeatureEventInfo> getMainProductEvents(String modelId, int pageNo, int pageSize) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS,
                modelId, FEATURE, EVENT);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, CommonListResponse.class);
    }

    public CommonListResponse<ProductFeatureCommandInfo> getMainProductCommands(String modelId,
                                                                                int pageNo, int pageSize) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS,
                modelId, FEATURE, COMMAND);
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, CommonListResponse.class);
    }

    public DtmlDetailResponse getMainProductDTMLDetail(String modelId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, PRODUCT, REPOSITORIES, MODELS,
                modelId, FEATURE, DETAIL);

        return invokeHttpClient(internalRequest, DtmlDetailResponse.class);
    }

    public void importProductModel(String instanceId, String modelId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.POST, PRODUCT, MODELS, instanceId, modelId);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // Alarm rule
    public AlarmRuleInfo createAlarmRule(String instanceId, CreateAlarmRuleRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, ALARMS, RULES, instanceId);

        return invokeHttpClient(internalRequest, AlarmRuleInfo.class);
    }

    public AlarmRuleInfo updateAlarmRule(String instanceId, String ruleId, UpdateAlarmRuleRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, ALARMS, RULES, instanceId, ruleId);

        return invokeHttpClient(internalRequest, AlarmRuleInfo.class);
    }

    public AlarmRuleInfo getAlarmRule(String instanceId, String ruleId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, ALARMS, RULES, instanceId, ruleId);

        return invokeHttpClient(internalRequest, AlarmRuleInfo.class);
    }

    public ListAlarmRuleResponse listAlarmRule(String instanceId, String name, int pageNo, int pageSize) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, ALARMS, RULES, instanceId);
        if (StringUtils.isNotEmpty(name)) {
            internalRequest.addParameter("name", name);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ListAlarmRuleResponse.class);
    }

    public void updateAlarmRuleActiveState(String instanceId, String ruleId, boolean active) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.PUT, ALARMS, RULES, instanceId, ruleId, STATES);
        internalRequest.addParameter("active", Boolean.toString(active));

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void triggerAlarmRule(String instanceId, String ruleId, TriggerAlarmRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.POST, ALARMS, RULES, instanceId, ruleId, TRIGGER);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void batchDeleteAlarmRule(String instanceId, BatchDeleteAlarmRuleRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, ALARMS, RULES, instanceId, BATCH, DELETE);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    // Alarm records
    public void deleteAlarmRecord(String instanceId, String recordId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.DELETE, ALARMS, RECORDS, instanceId, recordId);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void batchProcessAlarmRecord(String instanceId, BatchProcessAlarmRecordRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.PUT, ALARMS, RECORDS, instanceId, BATCH, PROCESS);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public AlarmRecordInfo getAlarmRecord(String instanceId, String recordId) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, ALARMS, RECORDS, instanceId, recordId);

        return invokeHttpClient(internalRequest, AlarmRecordInfo.class);
    }

    public ListAlarmRecordResponse listAlarmRecord(String instanceId, ListAlarmRecordRequest request) {

        InternalRequest internalRequest = createRequest(
                request, HttpMethodName.GET, ALARMS, RECORDS, instanceId);
        if (request.getPageSize() != null) {
            internalRequest.addParameter("pageSize", Integer.toString(request.getPageSize()));
        }
        if (StringUtils.isNotEmpty(request.getCursor())) {
            internalRequest.addParameter("cursor", request.getCursor());
        }
        if (StringUtils.isNotEmpty(request.getName())) {
            internalRequest.addParameter("name", request.getName());
        }
        if (request.getAlarmLevel() != null) {
            internalRequest.addParameter("alarmLevel", Integer.toString(request.getAlarmLevel()));
        }
        return invokeHttpClient(internalRequest, ListAlarmRecordResponse.class);
    }

    // 配置管理 fm
    public CommonListResponse<ConfigManagementResponse> getConfigList(String instanceId, String productKey,
                                                                      String configName, Integer pageNo,
                                                                      Integer pageSize) {

        InternalRequest internalRequest = createRequest(
                new GenericAccountRequest(), HttpMethodName.GET, FM, CONFIG, instanceId);
        if (StringUtils.isNotEmpty(productKey)) {
            internalRequest.addParameter("productKey", productKey);
        }
        if (StringUtils.isNotEmpty(configName)) {
            internalRequest.addParameter("configName", configName);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));

        return invokeHttpClient(internalRequest, CommonListResponse.class);
    }

    public void addConfig(String instanceId, AddConfigRequest request) {
        InternalRequest internalRequest = createRequest(request, HttpMethodName.POST, FM, CONFIG, instanceId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void deleteConfig(String instanceId, String productKey, String configId) {

        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.DELETE, FM, CONFIG, instanceId, productKey, configId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void modifyConfig(String instanceId, String productKey, String configId, AddConfigRequest request) {

        InternalRequest internalRequest = createRequest(request,
                HttpMethodName.PUT, FM, CONFIG, instanceId, productKey, configId);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void addConfigVersion(String instanceId, String productKey, String configId, String version, File file) {
        InternalRequest internalRequest = createUploadRequest("importFile", file,
                HttpMethodName.POST, FM, CONFIG, instanceId, productKey, configId);
        internalRequest.addParameter("version", version);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public void deleteConfigVersion(String instanceId, String productKey,
                                    String configId, String configVersion) {

        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.DELETE, FM, CONFIG, instanceId, productKey, configId, CONFIG_VERSION);
        internalRequest.addParameter("configVersion", configVersion);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public GetConfigVersionResponse downloadConfigVersion(String instanceId, String productKey,
                                                          String configId, String configVersion) {
        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.GET, FM, CONFIG, instanceId, productKey, configId, URL);
        internalRequest.addParameter("configVersion", configVersion);
        return invokeHttpClient(internalRequest, GetConfigVersionResponse.class);
    }

    public ConfigVersionListResponse getConfigVersionList(String instanceId, String productKey, String configId) {
        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.GET, FM, CONFIG, instanceId, productKey, configId, VERSIONS);
        return invokeHttpClient(internalRequest, ConfigVersionListResponse.class);
    }

    public ConfigManagementResponse getConfigInfo(String instanceId, String productKey, String configId) {
        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.GET, FM, CONFIG, instanceId, productKey, configId);
        return invokeHttpClient(internalRequest, ConfigManagementResponse.class);
    }

    public ProductConfigCommonListResponse<ConfigTaskResponse> getTaskList(String instanceId, String productKey,
                                                                           String configId, int pageNo, int pageSize,
                                                                           String taskId) {
        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.GET, FM, CONFIG, instanceId, productKey, configId, TASK);

        if (StringUtils.isNotEmpty(taskId)) {
            internalRequest.addParameter("taskId", taskId);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));

        return invokeHttpClient(internalRequest, ProductConfigCommonListResponse.class);
    }

    public void addTask(String instanceId, String productKey,
                        String configId, AddTaskRequest request) {
        InternalRequest internalRequest = createRequest(request,
                HttpMethodName.POST, FM, CONFIG, instanceId, productKey, configId, TASK);

        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    public ProductConfigCommonListResponse<ConfigTaskDetailResponse> getTaskDetail(String instanceId, String productKey,
                                                                                   String configId, int pageNo,
                                                                                   int pageSize, String keyword) {
        InternalRequest internalRequest = createRequest(new GenericAccountRequest(),
                HttpMethodName.GET, FM, CONFIG, instanceId, productKey, configId, TASK, DETAIL);
        if (StringUtils.isNotEmpty(keyword)) {
            internalRequest.addParameter("keyword", keyword);
        }
        internalRequest.addParameter("pageNo", Integer.toString(pageNo));
        internalRequest.addParameter("pageSize", Integer.toString(pageSize));
        return invokeHttpClient(internalRequest, ProductConfigCommonListResponse.class);
    }

    public void addTaskCsv(String instanceId, String productKey,
                           String configId, String configVersion, File file) {
        InternalRequest internalRequest = createUploadRequest("importFile", file,
                HttpMethodName.POST, FM, CONFIG, instanceId, productKey, configId, TASK, CSV);
        internalRequest.addParameter("configVersion", configVersion);
        invokeHttpClient(internalRequest, AbstractBceResponse.class);
    }

    private InternalRequest createRequest(AbstractBceRequest bceRequest, HttpMethodName httpMethod,
                                          String... pathVariables) {
        List<String> path = new ArrayList<String>();
        path.add(VERSION);
        if (pathVariables != null) {
            for (String pathVariable : pathVariables) {
                path.add(pathVariable);
            }
        }
        URI uri = HttpUtils.appendUri(this.getEndpoint(), path.toArray(new String[path.size()]));
        InternalRequest request = new InternalRequest(httpMethod, uri);
        request.setCredentials(bceRequest.getRequestCredentials());
        if (httpMethod == HttpMethodName.POST || httpMethod == HttpMethodName.PUT) {
            fillInHeadAndBody(bceRequest, request);
        }
        return request;
    }
    private InternalRequest createUploadRequest(String name, File file,
                                                HttpMethodName httpMethod, String... pathVariables) {
        List<String> path = new ArrayList<String>();
        path.add(VERSION);
        if (pathVariables != null) {
            for (String pathVariable : pathVariables) {
                path.add(pathVariable);
            }
        }
        URI uri = HttpUtils.appendUri(this.getEndpoint(), path.toArray(new String[path.size()]));
        InternalRequest request = new InternalRequest(httpMethod, uri);
        if (httpMethod == HttpMethodName.POST || httpMethod == HttpMethodName.PUT) {
            FilePart fp;
            try {
                fp = new FilePart(name, file);
            } catch (FileNotFoundException e) {
                throw new BceClientException("文件不存在或路径错误.");
            }
            Part[] parts = {fp};
            MultipartRequestEntity content = new MultipartRequestEntity(parts, new HttpMethodParams());
            request.addHeader(Headers.CONTENT_LENGTH, Long.toString(content.getContentLength()));
            request.addHeader(Headers.CONTENT_TYPE, content.getContentType());
            try {
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                content.writeRequest(os);
                request.setContent(RestartableInputStream.wrap(os.toByteArray()));
            } catch (IOException e) {
                throw new BceClientException("内部错误.", e);
            }
            return request;
        }
        throw new BceClientException("上传文件只能使用post或put请求方法.");
    }
    private void fillInHeadAndBody(AbstractBceRequest bceRequest, InternalRequest request) {
        byte[] content = toJson(bceRequest);
        request.addHeader(Headers.CONTENT_LENGTH, Integer.toString(content.length));
        request.addHeader(Headers.CONTENT_TYPE, CONTENT_TYPE);
        request.setContent(RestartableInputStream.wrap(content));
    }
    private byte[] toJson(AbstractBceRequest bceRequest) {
        String jsonStr = JsonUtils.toJsonString(bceRequest);
        try {
            return jsonStr.getBytes(DEFAULT_ENCODING);
        } catch (UnsupportedEncodingException e) {
            throw new BceClientException("Fail to get UTF-8 bytes", e);
        }
    }
}
