package com.jdaz.sinosoftgz.apis.log.api;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import okhttp3.ConnectionPool;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.Transient;
import java.io.IOException;
import java.io.Serializable;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * api 日志对象 <br>
 * 需要先调 init 全局初始化 <br>
 * create 生成一个新日志， <br>
 * createChild 可以生成子日志 <br>
 * 结束时需要调用end() <br>
 * 可以调用current()取当前 日志对象，对象保存在 ThreadLocal 中 <br>
 */
public class ApisLog implements Serializable {

    private static ThreadLocal<ApisLog> current = new ThreadLocal<>();

    private static ThreadLocal<Object> requestHolder = new ThreadLocal();

    public static void setReqeust(Object request) {
        requestHolder.set(request);
    }

    public static Object getRequest() {
        return requestHolder.get();
    }

    public static void clearRequest() {
        requestHolder.set(null);
    }

    public static void removeRequest() {
        requestHolder.remove();
    }

    public static void clearCurrent() {
        current.remove();
    }

    private static Boolean initFlag = false;

    /**
     * 初始化日志
     *
     * @param logServerUrl
     */
    public static void init(String logServerUrl, String initAppName, String initHostName) {
        apiLogServerUrl = logServerUrl;
        appName = initAppName;
        hostName = initHostName;
        initFlag = true;
        logger.warn("测试连接地址打印log，url为：{}，appName为：{}，hostName为：{}", apiLogServerUrl, appName, hostName);
    }

    private static final Logger logger = LoggerFactory.getLogger(ApisLog.class);

    private static String apiLogServerUrl = "";

    private static String appName = "";

    /**
     * 主机名
     */
    private static String hostName;

    private static ThreadPoolExecutor executor =
            new ThreadPoolExecutor(5, 20, 60, TimeUnit.SECONDS, new LinkedTransferQueue<>(), new ThreadFactoryBuilder().setNameFormat("log-app-common-pool-%d").build());

    private ApisLog() {
    }

    static SnowFlake snowFlake = new SnowFlake(1L, 1L);

    /**
     * 取当前log
     *
     * @return
     */
    public static ApisLog current() {
        return current.get();
    }

    /**
     * 取当前或者创建新的日志
     *
     * @param unionId
     * @return
     */
    public static ApisLog currentOrMakeNew(String unionId) {
        ApisLog apisLog = current.get();
        if (apisLog != null) {
            return apisLog;
        }
        return create(unionId);
    }

    /**
     * 生成顶级日志实例
     *
     * @return
     */
    public static ApisLog create() {
        ApisLog apisLog = new ApisLog();
        apisLog.unionId = "_$_" + apisLog.id;
        apisLog.setAppName(appName);
        ApisLog oldApisLog = current.get();
        if (oldApisLog != null && !oldApisLog.endFlag) {
            oldApisLog.sendAndEnd();
        }
        current.set(apisLog);
        return apisLog;
    }

    /**
     * 生成顶级日志实例
     *
     * @return
     */
    public static ApisLog create(String unionId) {
        ApisLog apisLog = create();
        apisLog.unionId = unionId;
        return apisLog;
    }


    /**
     * 生成子日志
     *
     * @return
     */
    public ApisLog createChild() {
        ApisLog apisLog = new ApisLog();
        apisLog.parentId = parentId;
        apisLog.unionId = unionId;
        apisLog.parent = this;
        return apisLog;
    }

    /**
     * 保存当前日志数据
     *
     * @return
     */
    public ApisLog send() {
        if (this.startTime == null) {
            this.startTime = new Date();
        }
        sendLog();
        return this;
    }

    /**
     * 结束当前日志
     */
    public void sendAndEnd() {
        if (this.endTime == null) {
            this.endTime = new Date();
        }
        //发送日志
        sendLog();
        endFlag = true;
        current.set(null);
    }

    /**
     * 结束当前日志
     *
     * @param endDate
     */
    public void sendAndEnd(Date endDate) {
        this.endTime = endDate;
        this.sendAndEnd();
    }

    /**
     * 数据已发送标志，内容体与tags 可以追加
     */
    Map<String, Object> dataHadSend = new ConcurrentHashMap<>();

    Set<String> tagHadSend = new HashSet<>();

    public static final MediaType JSON = MediaType.get("application/json; charset=utf-8");
    public static final OkHttpClient CLIENT = new OkHttpClient.Builder()
            .connectionPool(new ConnectionPool(2, 60, TimeUnit.SECONDS))
            .connectTimeout(10, TimeUnit.SECONDS)       //设置连接超时
            .callTimeout(10, TimeUnit.SECONDS)
            .readTimeout(10, TimeUnit.SECONDS)          //设置读超时
            .writeTimeout(10, TimeUnit.SECONDS)          //设置写超时
            .retryOnConnectionFailure(true)             //是否自动重连
            .build();                                   //构建OkHttpClient对象

    private static boolean post(String json) {

        RequestBody body = RequestBody.create(JSON, json.getBytes());
        logger.warn("测试连接地址打印log，当次请求url为：{},参数值为：{}", apiLogServerUrl, com.alibaba.fastjson.JSON.toJSONString(body));
        Request request = new Request.Builder()
                .url(apiLogServerUrl)
                .post(body)
                .build();
        try (Response response = CLIENT.newCall(request).execute()) {
            return response.code() >= 200 && response.code() <= 300;
        } catch (IOException e) {
            logger.error("发送日志出错" + e.getLocalizedMessage(), e);
            return false;
        }
    }

    /**
     * 通过http 发送日志
     */
    private ApisLog sendLog() {
        if (!initFlag) {
            logger.error("logApi 未初始化");
            return this;
        }
        final ApisLog log = this;
        executor.execute(() -> {
            //全集数据
            Map<String, Object> sendDataMap = new ConcurrentHashMap<>();
            if (log.getId() != null) {
                sendDataMap.put("id", log.getId());
            }
            if (log.getParentId() != null) {
                sendDataMap.put("parentId", log.getParentId());
            }
            if (log.getBusinessNode() != null) {
                sendDataMap.put("businessNode", log.getBusinessNode());
            }
            if (log.getContent() != null) {
                sendDataMap.put("content", log.getContent());
            }
            if (log.getEndTime() != null) {
                sendDataMap.put("endTime", log.getEndTime());
            }
            if (log.getStartTime() != null) {
                sendDataMap.put("startTime", log.getStartTime());
            }
            if (log.getAppName() != null) {
                sendDataMap.put("appName", log.getAppName());
            }
            if (hostName != null) {
                sendDataMap.put("hostName", hostName);
            }
            if (log.getLevel() != null) {
                sendDataMap.put("level", log.getLevel());
            }
            Set<String> tagsMap = new HashSet();
            if (log.tags != null) {
                tagsMap.addAll(log.getTags());
                sendDataMap.put("tags", tagsMap);
            }
            if (log.getUnionId() == null) {
                log.setUnionId("_$_" + log.getId());
            }
            sendDataMap.put("unionId", log.getUnionId());
            //排除已经发送过的数据
            for (String key : dataHadSend.keySet()) {
                if (sendDataMap.containsKey(key) && !"content".equals(key) && !"tags".equals(key) && !"businessNode".equals(key)) {
                    sendDataMap.remove(key);
                }
            }
            //增加本次发送的数据
            for (String key : sendDataMap.keySet()) {
                if ("id".equals(key)) {
                    continue;
                } else if ("tags".equals(key)) {
                    tagsMap.removeAll(tagHadSend);
                    tagHadSend.addAll(tagsMap);
                } else if ("content".equals(key)) {
                    String oldContent = "";
                    String newContent = "";
                    if (dataHadSend.get("content") instanceof String) {
                        oldContent = (String) dataHadSend.get("content");
                    }
                    if (sendDataMap.get("content") instanceof String) {
                        newContent = (String) sendDataMap.get("content");
                    }

                    if (newContent != null) {
                        if (!newContent.equals(oldContent)) {
                            dataHadSend.put("content", newContent);
                        } else {
                            sendDataMap.remove("content");
                        }
                    }
                } else {
                    dataHadSend.put(key, sendDataMap.get(key));
                }
            }

            try {
                post(com.alibaba.fastjson.JSON.toJSONString(sendDataMap));
            } catch (Exception e) {
                logger.error("发送日志到日志服务器时出错" + e.getMessage(), e);
            }
        });

        return this;
    }

    /**
     * 结束标志,不序列化输出，不加set get
     */
    private Boolean endFlag = false;

    /**
     * 不序列化输出，不加set get
     */
    private ApisLog parent;

    @Transient
    public ApisLog getParent() {
        return parent;
    }

    /**
     * 保证在一定时间内唯一
     */
    private Long id = snowFlake.nextId();

    /**
     * 父日志ID
     */
    private Long parentId;

    /**
     * 统一业务ID
     */
    private String unionId;

    /**
     * 产生日志的应用id
     */
//    private String appName;

    /**
     * 产生时间
     */
    private Date startTime;

    /**
     * 结束时间
     */
    private Date endTime;

    /**
     * 业务结点
     */
    private String businessNode;

    /**
     * 标签
     */
    private Set<String> tags = new HashSet<>();

    /**
     * 日志内容
     */
    private String content;

    /**
     * 日志级别
     */
    private String level;

    public Long getId() {
        return id;
    }

    public Long getParentId() {
        return parentId;
    }


    public String getLevel() {
        return level;
    }

    public ApisLog setLevel(String level) {
        this.level = level;
        return this;
    }

    public String getAppName() {
        return appName;
    }

    public ApisLog setAppName(String appName) {
        this.appName = appName;
        return this;
    }

    public String getUnionId() {
        return unionId;
    }

    public ApisLog setUnionId(String unionId) {
        this.unionId = unionId;
        return this;
    }

    public Date getStartTime() {
        return startTime;
    }

    public ApisLog setStartTime(Date startTime) {
        this.startTime = startTime;
        return this;
    }

    public Date getEndTime() {
        return endTime;
    }

    public ApisLog setEndTime(Date endTime) {
        this.endTime = endTime;
        return this;
    }

    public String getBusinessNode() {
        return businessNode;
    }

    public ApisLog setBusinessNode(String businessNode) {
        this.businessNode = businessNode;
        return this;
    }

    public Set<String> getTags() {
        return tags;
    }

    public ApisLog setTags(Set<String> tags) {
        this.tags = tags;
        return this;
    }

    public ApisLog addTag(String... tag) {
        if (tag != null && tag.length > 0) {
            for (String s : tag) {
                tags.add(s);
            }
        }
        return this;
    }

    public String getContent() {
        return content;
    }

    public ApisLog setContent(String content) {
        this.content = content;
        return this;
    }

    public static void shutdown() {
        executor.shutdown();
    }

}
