/*
 * Decompiled with CFR 0.152.
 */
package com.mzlion.core.io;

import com.mzlion.core.exceptions.FatalFileException;
import com.mzlion.core.io.IOUtils;
import com.mzlion.core.lang.Assert;
import com.mzlion.core.lang.StringUtils;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.util.List;

public class FileUtils {
    public static final long ONE_KB = 1024L;
    public static final long ONE_MB = 0x100000L;
    private static final long FILE_COPY_BUFFER_SIZE = 0x1E00000L;

    public static File getTempDirectory() {
        return new File(FileUtils.getTempDirectoryPath());
    }

    public static String getTempDirectoryPath() {
        return System.getProperty("java.io.tmpdir");
    }

    public static File getUserDirectory() {
        return new File(FileUtils.getUserDirectoryPath());
    }

    public static String getUserDirectoryPath() {
        return System.getProperty("user.home");
    }

    public static List<String> readLines(File file) {
        return FileUtils.readLines(file, Charset.defaultCharset());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<String> readLines(File file, Charset charset) {
        Assert.notNull(file, "The parameter[destFile] is null.");
        Assert.notNull(charset, "The parameter[charset] is null.");
        FileInputStream in = null;
        try {
            in = FileUtils.openFileInputStream(file);
            List<String> list = IOUtils.readLines(in, charset);
            return list;
        }
        finally {
            IOUtils.closeQuietly(in);
        }
    }

    public static void copyFile(File srcFile, OutputStream output) {
        FileInputStream in = FileUtils.openFileInputStream(srcFile);
        try {
            IOUtils.copyLarge(new BufferedInputStream(in), output);
        }
        finally {
            IOUtils.closeQuietly(in);
        }
    }

    public static void copyFile(File srcFile, File destFile) {
        FileUtils.copyFile(srcFile, destFile, true);
    }

    public static void copyFile(File srcFile, File destFile, boolean holdFileDate) {
        Assert.notNull(srcFile, "Source file must not be null.");
        Assert.notNull(destFile, "Destination file must not be null.");
        if (!srcFile.exists()) {
            throw new FatalFileException("Source [" + srcFile + "] does not exist.");
        }
        if (srcFile.isDirectory()) {
            throw new FatalFileException("Source [" + srcFile + "] exists but it is a directory.");
        }
        try {
            if (srcFile.getCanonicalPath().equals(destFile.getCanonicalPath())) {
                throw new FatalFileException(String.format("Source [%s] and destination [%s] are the same.", srcFile, destFile));
            }
            File parentFile = destFile.getParentFile();
            if (parentFile != null && !parentFile.mkdirs() && !parentFile.isDirectory()) {
                throw new FatalFileException("Destination [" + parentFile + "] directory cannot be created.");
            }
            if (destFile.exists() && !destFile.canWrite()) {
                throw new FatalFileException(" ===> Destination [" + parentFile + "] directory cannot be written.");
            }
            FileUtils.doCopyFile(srcFile, destFile, holdFileDate);
        }
        catch (IOException e) {
            throw new FatalFileException(e);
        }
    }

    private static void doCopyFile(File srcFile, File destFile, boolean holdFileDate) throws IOException {
        if (destFile.exists() && destFile.isDirectory()) {
            throw new FatalFileException("Destination [" + destFile + "] exists but it is a directory.");
        }
        try (FileInputStream in = new FileInputStream(srcFile);
             FileOutputStream out = new FileOutputStream(destFile);
             FileChannel inChannel = in.getChannel();
             FileChannel outChannel = out.getChannel();){
            long count;
            long size = inChannel.size();
            for (long pos = 0L; pos < size; pos += outChannel.transferFrom(inChannel, pos, count)) {
                count = size - pos > 0x1E00000L ? 0x1E00000L : size - pos;
            }
        }
        if (srcFile.length() != destFile.length()) {
            throw new IOException(String.format("Failed to copy full contents from [%s] to [%s]", srcFile.getPath(), destFile.getPath()));
        }
        if (holdFileDate) {
            destFile.setLastModified(srcFile.lastModified());
        }
    }

    public static void copyFileToDirectory(File srcFile, File destDir) {
        FileUtils.copyFileToDirectory(srcFile, destDir, true);
    }

    private static void copyFileToDirectory(File srcFile, File destDir, boolean holdFileDate) {
        Assert.notNull(srcFile, "Source file must not be null.");
        Assert.notNull(destDir, "Destination Directory must not be null.");
        if (destDir.exists() && !destDir.isDirectory()) {
            throw new FatalFileException("Destination [" + destDir + "] is not a directory.");
        }
        File destFile = new File(destDir, srcFile.getName());
        FileUtils.copyFile(srcFile, destFile, holdFileDate);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void copyStream(InputStream in, File destFile) {
        Assert.notNull(in, "The parameter[in] is null.");
        Assert.notNull(destFile, "The parameter[destFile] is null.");
        FileOutputStream fos = null;
        try {
            fos = FileUtils.openFileOutputStream(destFile);
            BufferedOutputStream bos = new BufferedOutputStream(fos);
            IOUtils.copy(in, (OutputStream)bos);
            IOUtils.closeQuietly(bos);
        }
        finally {
            IOUtils.closeQuietly(fos);
            IOUtils.closeQuietly(in);
        }
    }

    public static void copyDirectory(File srcDir, File destDir) {
        FileUtils.copyDirectory(srcDir, destDir, true);
    }

    public static void copyDirectory(File srcDir, File destDir, boolean holdFileDate) {
        FileUtils.copyDirectory(srcDir, destDir, holdFileDate, null);
    }

    public static void copyDirectory(File srcDir, File destDir, boolean holdFileDate, FileFilter filter) {
        Assert.notNull(srcDir, "Source Directory must not be null.");
        Assert.notNull(destDir, "Destination Directory must not be null.");
        if (!srcDir.exists()) {
            throw new FatalFileException("Source [" + srcDir + "] does not exist.");
        }
        if (destDir.isFile()) {
            throw new FatalFileException("Destination [" + destDir + "] exists but is not a directory.");
        }
        try {
            if (srcDir.getCanonicalPath().equals(destDir.getCanonicalPath())) {
                throw new FatalFileException(String.format("Source [%s] and destination [%s] are the same.", srcDir, destDir));
            }
            if (destDir.getCanonicalPath().startsWith(srcDir.getCanonicalPath() + File.separator)) {
                throw new FatalFileException(String.format("Destination [%s] is child directory of source [%s].", destDir, srcDir));
            }
            FileUtils.doCopyDirectory(srcDir, destDir, holdFileDate, filter);
        }
        catch (IOException e) {
            throw new FatalFileException(e);
        }
    }

    private static void doCopyDirectory(File srcDir, File destDir, boolean holdFileDate, FileFilter filter) throws IOException {
        File[] srcFiles;
        File[] fileArray = srcFiles = filter == null ? srcDir.listFiles() : srcDir.listFiles(filter);
        if (srcFiles == null) {
            throw new IOException("Failed to list contents of [" + srcDir + "]");
        }
        if (destDir.exists() && !destDir.isDirectory()) {
            throw new IOException("Destination [" + destDir + "] exists but is not a directory.");
        }
        if (!destDir.mkdirs() && !destDir.isDirectory()) {
            throw new IOException("Destination [" + destDir + "] directory cannot be created.");
        }
        if (!destDir.canWrite()) {
            throw new IOException("Destination [" + destDir + "] cannot be written to.");
        }
        for (File srcFile : srcFiles) {
            File destFile = new File(destDir, srcFile.getName());
            if (srcFile.isDirectory()) {
                FileUtils.doCopyDirectory(srcFile, destFile, holdFileDate, filter);
                continue;
            }
            FileUtils.doCopyFile(srcFile, destFile, holdFileDate);
        }
        if (holdFileDate) {
            destDir.setLastModified(srcDir.lastModified());
        }
    }

    public static void moveFile(File srcFile, File destFile) {
        Assert.notNull(srcFile, "Source must not be null.");
        Assert.notNull(destFile, "Destination must not be null.");
        if (!srcFile.exists()) {
            throw new FatalFileException("Source [" + srcFile + "] does not exist.");
        }
        if (srcFile.isDirectory()) {
            throw new FatalFileException("Source [" + srcFile + "] is a directory.");
        }
        if (destFile.isFile() && destFile.exists()) {
            throw new FatalFileException("Destination [" + destFile + "] already exists.");
        }
        if (destFile.isDirectory() && !destFile.canWrite()) {
            throw new FatalFileException("Destination [" + destFile + "] cannot be written to.");
        }
        File targetFile = destFile.isDirectory() ? new File(destFile, srcFile.getName()) : destFile;
        boolean renameTo = srcFile.renameTo(targetFile);
        if (!renameTo) {
            FileUtils.copyFile(srcFile, targetFile);
            if (!srcFile.delete()) {
                targetFile.delete();
                throw new FatalFileException(String.format("Failed to delete original file [%s], after copy to [%s]", srcFile, destFile));
            }
        }
    }

    public static void moveDirectory(File source, File destDir) {
        FileUtils.moveDirectory(source, destDir, false);
    }

    public static void moveDirectory(File srcDir, File destDir, boolean toDir) {
        File targetDir;
        Assert.notNull(srcDir, "Source must not be null.");
        Assert.notNull(destDir, "Destination must not be null.");
        if (!srcDir.exists()) {
            throw new FatalFileException("Source [" + srcDir + "] does not exist.");
        }
        if (!srcDir.isDirectory()) {
            throw new FatalFileException("Destination [" + srcDir + "] is not a directory.");
        }
        if (destDir.exists() && !destDir.isDirectory()) {
            throw new FatalFileException("Destination [" + destDir + "] is not a directory.");
        }
        File file = targetDir = toDir ? new File(destDir, srcDir.getName()) : destDir;
        if (!targetDir.mkdirs()) {
            throw new FatalFileException("Directory [" + targetDir + "] could not be created.");
        }
        boolean renameTo = srcDir.renameTo(targetDir);
        if (!renameTo) {
            FileUtils.copyDirectory(srcDir, targetDir);
            FileUtils.delete(srcDir);
            if (srcDir.exists()) {
                throw new FatalFileException(String.format("Failed to delete original directory '%s' after copy to '%s'", srcDir, destDir));
            }
        }
    }

    public static void delete(File file) {
        Assert.notNull(file, "File must not be null.");
        if (!file.exists()) {
            return;
        }
        if (file.isDirectory()) {
            FileUtils.cleanDirectory(file);
        }
        if (!file.delete()) {
            throw new FatalFileException("Unable to delete file: " + file);
        }
    }

    public static void cleanDirectory(File directory) {
        Assert.notNull(directory, "Directory must not be null.");
        if (!directory.exists()) {
            throw new FatalFileException("Directory [" + directory + "] does not exist.");
        }
        if (!directory.isDirectory()) {
            throw new FatalFileException("The [" + directory + "] is not a directory.");
        }
        File[] listFiles = directory.listFiles();
        if (listFiles == null) {
            throw new FatalFileException("Failed to list contents of " + directory);
        }
        for (File listFile : listFiles) {
            if (listFile.isDirectory()) {
                FileUtils.cleanDirectory(listFile);
            }
            if (listFile.delete()) continue;
            throw new FatalFileException("Unable to delete file: " + listFile);
        }
    }

    public static FileInputStream openFileInputStream(File file) {
        Assert.notNull(file, "File must not be null.");
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new FatalFileException("File '" + file + "' exists but is a directory");
            }
            if (!file.canRead()) {
                throw new FatalFileException("File '" + file + "' cannot be read");
            }
            try {
                return new FileInputStream(file);
            }
            catch (IOException e) {
                throw new FatalFileException(e);
            }
        }
        throw new FatalFileException("File '" + file + "' does not exist");
    }

    public static FileOutputStream openFileOutputStream(File file) {
        return FileUtils.openFileOutputStream(file, false);
    }

    private static FileOutputStream openFileOutputStream(File file, boolean append) {
        Assert.notNull(file, "File must not be null.");
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new FatalFileException("Destination [" + file + "] exists but is a directory.");
            }
            if (!file.canWrite()) {
                throw new FatalFileException(String.format("Destination [%s] exists but cannot write.", file));
            }
        } else {
            File parent = file.getParentFile();
            if (parent != null && !parent.mkdirs() && !parent.isDirectory()) {
                throw new FatalFileException("Directory [" + parent + "] could not be created.");
            }
        }
        try {
            return new FileOutputStream(file, append);
        }
        catch (IOException e) {
            throw new FatalFileException(e);
        }
    }

    public static String md5Hex(File file) {
        FileInputStream in = null;
        try {
            in = FileUtils.openFileInputStream(file);
            String string = IOUtils.md5Hex(in);
            return string;
        }
        finally {
            IOUtils.closeQuietly(in);
        }
    }

    public static String sha1Hex(File file) {
        FileInputStream in = null;
        try {
            in = FileUtils.openFileInputStream(file);
            String string = IOUtils.sha1Hex(in);
            return string;
        }
        finally {
            IOUtils.closeQuietly(in);
        }
    }

    public static String formatSize(long fileSize) {
        if (fileSize < 0L) {
            return "";
        }
        return FileUtils.formatSize((double)fileSize);
    }

    public static String formatSize(double fileSize) {
        if (fileSize < 0.0) {
            return "";
        }
        double size = fileSize;
        if (size < 1024.0) {
            return size + "Byte";
        }
        if ((size /= 1024.0) < 1024.0) {
            return (double)Math.round(size * 100.0) / 100.0 + "KB";
        }
        if ((size /= 1024.0) < 1024.0) {
            return (double)Math.round(size * 100.0) / 100.0 + "MB";
        }
        if ((size /= 1024.0) < 1024.0) {
            return (double)Math.round(size * 100.0) / 100.0 + "GB";
        }
        return (double)Math.round(size * 100.0) / 100.0 + "TB";
    }

    public static String formatSizeAsString(String fileSize) {
        if (StringUtils.isEmpty(fileSize)) {
            return "";
        }
        double size = Double.parseDouble(fileSize);
        return FileUtils.formatSize(size);
    }

    public static void forceMakeDir(String directoryPath) {
        FileUtils.forceMakeDir(new File(directoryPath));
    }

    public static void forceMakeDir(File directory) {
        if (directory.exists()) {
            if (!directory.isDirectory()) {
                throw new FatalFileException("The file[" + directory + "] exists and is not a directory.Unable to create directory.");
            }
        } else if (!directory.mkdirs() && !directory.isDirectory()) {
            throw new FatalFileException("Unable to create directory[" + directory + "]");
        }
    }
}

