package com.picc.gz.sfzn.api.security;

import com.picc.gz.sfzn.api.ApiConstants;
import com.picc.gz.sfzn.api.vo.basic.BaseRequest;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.security.MessageDigest;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author Liang Wenxu
 * @since 2018/10/29
 */
public class SignUtil {
    private static ThreadLocal<DateFormat> dateFormat = new ThreadLocal<>();

    private static DateFormat getDateFormater() {
        DateFormat f = dateFormat.get();
        if(f == null) {
            f = new SimpleDateFormat("yyyyMMddHHmmssSSS");
            dateFormat.set(f);
        }
        return f;
    }

    private static final char[] HEX_DIGITS = {'0', '1', '2', '3', '4', '5',
            '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

    /**
     * Takes the raw bytes from the digest and formats them correct.
     *
     * @param bytes the raw bytes from the digest.
     * @return the formatted bytes.
     */
    private static String getFormattedText(byte[] bytes) {
        int len = bytes.length;
        StringBuilder buf = new StringBuilder(len * 2);
        // 把密文转换成十六进制的字符串形式
        for (int j = 0; j < len; j++) {
            buf.append(HEX_DIGITS[(bytes[j] >> 4) & 0x0f]);
            buf.append(HEX_DIGITS[bytes[j] & 0x0f]);
        }
        return buf.toString();
    }


    /**
     * 根据请求签名
     * 注意：appId, appKey，appSecret必须
     *
     * @param request
     * @return
     */
    public static String signRequest(BaseRequest request, String secret) throws IntrospectionException, InvocationTargetException, IllegalAccessException {
        String signStr = String.format("%s&secret=%s", getParamStr(request), secret);
        System.out.println("生成的签名字符串：" + signStr);
        String sign = encodeSHA1(signStr).toUpperCase();
        System.out.println("生成的签名：" + sign);
        return sign;
    }

    /**
     * 获取请求参数字符串
     * @param request
     * @return 返回的字符串格式为header.appId=aaaaaaaa&header.appKey=22222222&header.timeStamp=201801010000000&body.p1=11111
     * @throws IntrospectionException
     */
    public static String getParamStr(BaseRequest request) throws IntrospectionException, InvocationTargetException, IllegalAccessException {
        SortedMap<String, String> signMap = new TreeMap<>(new Comparator<String>() {
            @Override
            public int compare(String o1, String o2) {
                return o1.compareTo(o2);
            }
        });
        signMap.put("header.appId", request.getHeader().getAppId());
        signMap.put("header.appKey", request.getHeader().getAppKey());
        if(request.getHeader().getTimeStamp() != null) {
            signMap.put("header.timeStamp", getDateFormater().format(request.getHeader().getTimeStamp()));
        }
        if (request.getBody() != null) {
            BeanInfo srcBeanInfo = Introspector.getBeanInfo(request.getBody().getClass());
            PropertyDescriptor[] descriptorSrc = srcBeanInfo.getPropertyDescriptors();
            for (PropertyDescriptor property : descriptorSrc) {
                SortedMap<String, String> childMap = getSignMap("body." + property.getName(), property.getReadMethod().invoke(request.getBody()));
                signMap.putAll(childMap);
            }
        }

        StringBuilder signBuilder = new StringBuilder();
        for (Map.Entry<String, String> m : signMap.entrySet()) {
            signBuilder.append("&").append(m.getKey()).append("=").append(m.getValue());
        }
        return signBuilder.substring(1);
    }

    private static SortedMap<String, String> getSignMap(String basicName, Object object) throws IntrospectionException, InvocationTargetException, IllegalAccessException {
        SortedMap<String, String> signMap = new TreeMap<>();
        if(object != null) {
            if(object instanceof Date) {
                signMap.put(basicName, getDateFormater().format(object));
            } else if (isBaseType(object.getClass())) {
                if(!basicName.contains(".class")) {
                    signMap.put(basicName, object.toString());
                }
            } else if (object.getClass().isArray()) {
                int i = 0;
                for (Object o : (Object[]) object) {
                    SortedMap<String, String> childMap = getSignMap(basicName + "[" + i + "].", o);
                    signMap.putAll(childMap);
                    i++;
                }
            } else if (object instanceof Collection) {
                Collection collection = (Collection) object;
                Iterator it = collection.iterator();
                int i = 0;
                while (it.hasNext()) {
                    SortedMap<String, String> childMap = getSignMap(basicName + "[" + i + "].", it.next());
                    signMap.putAll(childMap);
                    i++;
                }
            } else {
                BeanInfo srcBeanInfo = Introspector.getBeanInfo(object.getClass());

                PropertyDescriptor[] descriptorSrc = srcBeanInfo.getPropertyDescriptors();
                for (PropertyDescriptor pd : descriptorSrc) {
                    SortedMap<String, String> childMap = getSignMap(basicName + pd.getName(), pd.getReadMethod().invoke(object));
                    signMap.putAll(childMap);
                }
            }
        }

        return signMap;
    }

    public static String encodeSHA1(String source) {
        if (source == null) {
            return null;
        }
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA1");
            messageDigest.update(source.getBytes(ApiConstants.HTTP_REQUEST_DEFAULT_CHARSET));
            return getFormattedText(messageDigest.digest());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    private static boolean isBaseType(Class<?> type) {
        if (type.isPrimitive()) {
            return true;
        }
        if (CharSequence.class.isAssignableFrom(type)) {
            return true;
        }
        if (Number.class.isAssignableFrom(type)) {
            return true;
        }
        if (Date.class.isAssignableFrom(type)) {
            return true;
        }
        if (Boolean.class.equals(type)) {
            return true;
        }
        if (Character.class.equals(type)) {
            return true;
        }
        if (Class.class.equals(type)) {
            return true;
        }
        if (StringBuilder.class.equals(type)) {
            return true;
        }
        if (StringBuffer.class.equals(type)) {
            return true;
        }
        if (Object.class.equals(type)) {
            return true;
        }
        if (Void.class.equals(type)) {
            return true;
        }
        return false;
    }
}
