package com.sinosoftgz.starter.redis.utils;

import com.google.common.base.Preconditions;
import com.sinosoftgz.global.common.constants.CommonConstants;
import com.sinosoftgz.starter.redis.properties.RedisLockProperties;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import javax.xml.ws.Action;
import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * redis锁 调用lock锁住资源，方法执行完后要调用unlock释放锁
 * Created by Roney on 2019/3/21 17:19
 *
 * @author Roney
 */
@Component
@Slf4j
public class RedisLockUtils {

    @Resource
    StringRedisTemplate stringRedisTemplate;

    @Autowired
    RedisLockProperties redisLockProperties;


    public StringRedisTemplate getStringRedisTemplate() {
        return stringRedisTemplate;
    }

    public void setStringRedisTemplate(StringRedisTemplate stringRedisTemplate) {
        this.stringRedisTemplate = stringRedisTemplate;
    }

    public RedisLockUtils(StringRedisTemplate stringRedisTemplate) {
        this.stringRedisTemplate = stringRedisTemplate;
    }

    /**
     * 加锁
     *
     * @param key      redis key 不能为空
     * @param lockTime 加锁时长 (加锁时长必须大于当前时间)
     * @return true 加锁成功，false 加锁失败
     */
    public boolean lock(String key, long lockTime) {
        Preconditions.checkArgument(lockTime > System.currentTimeMillis(), "加锁时长必须大于当前时间");
        return lock(key, String.valueOf(lockTime));
    }

    /**
     * 加锁
     *
     * @param key redis key 不能为空
     * @return true 加锁成功，false 加锁失败
     */
    public boolean lock(String key) {
        return lock(key, "");
    }

    /**
     * 解锁
     *
     * @param key redis key 不能为空
     */
    public void unlock(String key) {
        Preconditions.checkNotNull(key, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "key"));
        try {
            key = getLockKey(key);
            log.info("unlock key: {}", key);
            stringRedisTemplate.delete(key);
        } catch (Exception e) {
            log.error("redis 解锁异常:", e);
        }
    }

    /**
     * 加锁
     *
     * @param key   ：redis key 不能为空
     * @param value redis value
     * @return true 加锁成功，false 加锁失败
     */
    private boolean lock(String key, String value) {
        Preconditions.checkNotNull(key, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "key"));
        key = getLockKey(key);
        log.info("lock key: {}", key);
        /**
         * value 当前时间+超时时间
         */
        if (StringUtils.isEmpty(value)) {
            value = String.valueOf(System.currentTimeMillis() + redisLockProperties.getTimeout());
        }
        /**
         *   setnx----对应方法 setIfAbsent(key, value)，如果可以加锁返回true，不可以加锁返回false
         */

        if (stringRedisTemplate.opsForValue().setIfAbsent(key, value)) {
            return true;
        }

        /**
         *  为了解决可能出现的死锁情况
         */
        String currentValue = stringRedisTemplate.opsForValue().get(key);
        /**
         *  如果锁过期
         */

        if (!StringUtils.isEmpty(currentValue)
                && Long.parseLong(currentValue) < System.currentTimeMillis()) {
            /**
             *  获取上一个锁的时间：重新设置锁的过期时间value，并返回上一个过期时间
             */
            String oldValue = stringRedisTemplate.opsForValue().getAndSet(key, value);
            /**
             *  currentValue =2021-10-13，两个线程的value=2021-10-14,只会有一个线程拿到锁
             */
            if (!StringUtils.isEmpty(oldValue) && oldValue.equals(currentValue)) {
                return true;
            }
        }
        return false;
    }


    /**
     * 获取redis key
     *
     * @param key
     * @return 前缀+key
     */
    private String getLockKey(String key) {
        return redisLockProperties.getPrefix() + key;
    }


}
