package com.sinosoftgz.starter.redis.utils;


import com.google.common.base.Preconditions;
import com.sinosoftgz.global.common.constants.CommonConstants;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * redis 工具类
 * Created by Roney on 2019/3/21 17:19
 *
 * @author Roney
 */
@Component
@Slf4j
public class RedisUtils {

    @Resource
    private RedisTemplate<String, Object> redisTemplate;

    public RedisTemplate<String, Object> getRedisTemplate() {
        return redisTemplate;
    }

    public void setRedisTemplate(RedisTemplate<String, Object> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }


    /**
     * 指定缓存失效时间
     *
     * @param key  redis key 不能为空
     * @param time 时间(秒)
     * @return true成功 false 失败
     */
    public boolean expire(String key, long time) {
        checkCacheKey(key);
        try {
            if (time > 0) {
                redisTemplate.expire(key, time, TimeUnit.SECONDS);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }
    }

    /**
     * 根据key 获取过期时间
     *
     * @param key redis key 不能为空
     * @return 时间(秒) 返回0代表为永久有效
     */
    public long getExpire(String key) {
        checkCacheKey(key);
        return redisTemplate.getExpire(key, TimeUnit.SECONDS);
    }


    /**
     * 判断key是否存在
     *
     * @param key redis key 不能为空
     * @return true 存在 false不存在
     */
    public boolean hasKey(String key) {
        checkCacheKey(key);
        try {
            return redisTemplate.hasKey(key);
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }
    }

    /**
     * 删除缓存
     *
     * @param key 可以传一个值 或多个 不能为空
     */
    public void delete(String... key) {
        Preconditions.checkNotNull(key);
        if (key != null && key.length > 0) {
            if (key.length == 1) {
                redisTemplate.delete(key[0]);
            } else {
                redisTemplate.delete(Arrays.asList(key));
            }
        }
    }

    // ============================String=============================

    /**
     * 普通缓存获取
     *
     * @param key redis key 不能为空
     * @return 值
     */
    public Object get(String key) {
        checkCacheKey(key);
        return key == null ? null : redisTemplate.opsForValue().get(key);
    }

    /**
     * 普通缓存放入
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @return true成功 false失败
     */
    public boolean set(String key, Object value) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForValue().set(key, value);
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }

    }

    /**
     * 普通缓存放入并设置时间
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @param time  时间(秒) time要大于0 如果time小于等于0 将设置无限期
     * @return true成功 false 失败
     */
    public boolean set(String key, Object value, long time) {
        checkCacheKey(key);
        try {
            if (time > 0) {
                redisTemplate.opsForValue().set(key, value, time, TimeUnit.SECONDS);
            } else {
                set(key, value);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 普通缓存放入并设置时间
     *
     * @param key      键
     * @param value    值
     * @param time     时间
     * @param timeUnit 时间单位
     * @return true成功 false 失败
     */
    public boolean set(String key, Object value, long time, TimeUnit timeUnit) {
        checkCacheKey(key);
        try {
            if (time > 0) {
                redisTemplate.opsForValue().set(key, value, time, timeUnit);
            } else {
                set(key, value);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }
    }

    /**
     * 递增
     *
     * @param key   redis key 不能为空
     * @param delta 要增加几(大于0) 递增因子必须大于0
     * @return 返回递增后的值
     */
    public long incr(String key, long delta) {
        checkCacheKey(key);
        Preconditions.checkArgument(delta > 0, "递增因子必须大于0");
        return redisTemplate.opsForValue().increment(key, delta);
    }

    /**
     * 递减
     *
     * @param key   redis key 不能为空
     * @param delta 要减少几(小于0) 递减因子必须大于0
     * @return 返回递减后的值
     */
    public long decr(String key, long delta) {
        checkCacheKey(key);
        Preconditions.checkArgument(delta > 0, "递减因子必须大于0");
        return redisTemplate.opsForValue().increment(key, -delta);
    }

    // ================================Map=================================

    /**
     * Hash Get
     *
     * @param key  redis key 不能为空
     * @param item 项 不能为空
     * @return 值
     */
    public Object hashGet(String key, String item) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "item"));
        return redisTemplate.opsForHash().get(key, item);
    }

    /**
     * 获取hashKey对应的所有键值
     *
     * @param key redis key 不能为空
     * @return 对应的多个键值
     */
    public Map<Object, Object> hashMapGet(String key) {
        checkCacheKey(key);
        return redisTemplate.opsForHash().entries(key);
    }

    /**
     * Hash Set
     *
     * @param key redis key 不能为空
     * @param map 对应多个键值
     * @return true 成功 false 失败
     */
    public boolean hashMapSet(String key, Map<String, Object> map) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForHash().putAll(key, map);
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }
    }

    /**
     * Hash Set 并设置时间
     *
     * @param key  redis key 不能为空
     * @param map  对应多个键值
     * @param time 时间(秒)
     * @return true成功 false失败
     */
    public boolean hashMapSet(String key, Map<String, Object> map, long time) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForHash().putAll(key, map);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 向一张hash表中放入数据,如果不存在将创建
     *
     * @param key   redis key 不能为空
     * @param item  项
     * @param value 值
     * @return true 成功 false失败
     */
    public boolean hashSet(String key, String item, Object value) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "item"));
        try {
            redisTemplate.opsForHash().put(key, item, value);
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }
    }

    /**
     * 向一张hash表中放入数据,如果不存在将创建
     *
     * @param key   redis key 不能为空
     * @param item  项 不能为空
     * @param value 值
     * @param time  时间(秒) 注意:如果已存在的hash表有时间,这里将会替换原有的时间
     * @return true 成功 false失败
     */
    public boolean hashSet(String key, String item, Object value, long time) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "item"));
        try {
            redisTemplate.opsForHash().put(key, item, value);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return false;
        }
    }

    /**
     * 删除hash表中的值
     *
     * @param key  redis key 不能为空
     * @param item 项 可以使多个 不能为空
     */
    public void hashDelete(String key, Object... item) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item);
        redisTemplate.opsForHash().delete(key, item);
    }

    /**
     * 判断hash表中是否有该项的值
     *
     * @param key  redis key 不能为空
     * @param item 项  不能为空
     * @return true 存在 false不存在
     */
    public boolean hashHasKey(String key, String item) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "item"));
        return redisTemplate.opsForHash().hasKey(key, item);
    }

    /**
     * hash递增 如果不存在,就会创建一个 并把新增后的值返回
     *
     * @param key   redis key 不能为空
     * @param item  项 不能为空
     * @param delta 要增加几(大于0) 递增因子必须大于0
     * @return 返回递增后的值
     */
    public long hashIncr(String key, String item, long delta) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "item"));
        Preconditions.checkArgument(delta > 0, "递增因子必须大于0");
        return redisTemplate.opsForHash().increment(key, item, delta);
    }

    /**
     * hash递减
     *
     * @param key   redis key 不能为空
     * @param item  项 不能为空
     * @param delta 要减少几(大于0) 递减因子必须大于0
     * @return 返回递减后的值
     */
    public long hashDecr(String key, String item, long delta) {
        checkCacheKey(key);
        Preconditions.checkNotNull(item, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "item"));
        Preconditions.checkArgument(delta > 0, "递减因子必须大于0");
        return redisTemplate.opsForHash().increment(key, item, -delta);
    }

    // ============================set=============================

    /**
     * 根据key获取Set中的所有值
     *
     * @param key redis key 不能为空
     * @return 返回set集合
     */
    public Set<Object> setGet(String key) {
        checkCacheKey(key);
        try {
            return redisTemplate.opsForSet().members(key);
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return null;
        }
    }

    /**
     * 根据value从一个set中查询,是否存在
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @return true 存在 false不存在
     */
    public boolean setHasKey(String key, Object value) {
        checkCacheKey(key);
        try {
            return redisTemplate.opsForSet().isMember(key, value);
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 将数据放入set缓存
     *
     * @param key    redis key 不能为空
     * @param values 值 可以是多个
     * @return 成功个数
     */
    public long setSet(String key, Object... values) {
        checkCacheKey(key);
        try {
            return redisTemplate.opsForSet().add(key, values);
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return 0;
        }
    }

    /**
     * 将set数据放入缓存
     *
     * @param key    redis key 不能为空
     * @param time   时间(秒)
     * @param values 值 可以是多个
     * @return 成功个数
     */
    public long setSetTime(String key, long time, Object... values) {
        checkCacheKey(key);
        try {
            Long count = redisTemplate.opsForSet().add(key, values);
            if (time > 0) {
                expire(key, time);
            }
            return count;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return 0;
        }
    }

    /**
     * 获取set缓存的长度
     *
     * @param key redis key 不能为空
     * @return 返回缓存长度
     */
    public long getSetSize(String key) {
        checkCacheKey(key);
        try {
            return redisTemplate.opsForSet().size(key);
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return 0;
        }
    }

    /**
     * 移除值为value的
     *
     * @param key    redis key 不能为空
     * @param values 值 可以是多个
     * @return 移除的个数
     */
    public long setRemove(String key, Object... values) {
        checkCacheKey(key);
        try {
            Long count = redisTemplate.opsForSet().remove(key, values);
            return count;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return 0;
        }
    }
    // ===============================list=================================

    /**
     * 获取list缓存的内容
     *
     * @param key   redis key 不能为空
     * @param start 开始
     * @param end   结束 0 到 -1代表所有值
     * @return 返回list缓存的内容
     */
    public List<Object> listGet(String key, long start, long end) {

        checkCacheKey(key);
        try {
            return redisTemplate.opsForList().range(key, start, end);
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return null;
        }
    }

    /**
     * 获取list缓存的长度
     *
     * @param key redis key 不能为空
     * @return 返回list缓存的长度
     */
    public long listGetSize(String key) {
        checkCacheKey(key);
        try {
            return redisTemplate.opsForList().size(key);
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return 0;
        }
    }

    /**
     * 通过索引 获取list中的值
     *
     * @param key   redis key 不能为空
     * @param index 索引 index>=0时， 0 表头，1 第二个元素，依次类推；index<0时，-1，表尾，-2倒数第二个元素，依次类推
     * @return 返回list中的值
     */
    public Object listGetIndex(String key, long index) {
        checkCacheKey(key);
        try {
            return redisTemplate.opsForList().index(key, index);
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return null;
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @return true 成功 false失败
     */
    public boolean listSet(String key, Object value) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForList().rightPush(key, value);
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @param time  时间(秒)
     * @return true 成功 false失败
     */
    public boolean listSet(String key, Object value, long time) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForList().rightPush(key, value);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @return true 成功 false失败
     */
    public boolean listSet(String key, List<Object> value) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForList().rightPushAll(key, value);
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 将list放入缓存
     *
     * @param key   redis key 不能为空
     * @param value 值
     * @param time  时间(秒)
     * @return true 成功 false失败
     */
    public boolean listSet(String key, List<Object> value, long time) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForList().rightPushAll(key, value);
            if (time > 0) {
                expire(key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 根据索引修改list中的某条数据
     *
     * @param key   redis key 不能为空
     * @param index 索引
     * @param value 值
     * @return true 成功 false失败
     */
    public boolean listUpdateByIndex(String key, long index, Object value) {
        checkCacheKey(key);
        try {
            redisTemplate.opsForList().set(key, index, value);
            return true;
        } catch (Exception e) {
            log.error("操作redis异常:", e);

            return false;
        }
    }

    /**
     * 移除N个值为value
     *
     * @param key   redis key 不能为空
     * @param count 移除多少个
     * @param value 值
     * @return 移除的个数
     */
    public long listRemove(String key, long count, Object value) {
        checkCacheKey(key);
        try {
            Long remove = redisTemplate.opsForList().remove(key, count, value);
            return remove;
        } catch (Exception e) {
            log.error("操作redis异常:", e);
            return 0;
        }
    }

    private void checkCacheKey(String key) {
        Preconditions.checkNotNull(key, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "key"));
    }
}
