package com.sinosoftgz.starter.exception.handler;

import com.sinosoftgz.global.common.response.BaseResponse;
import com.sinosoftgz.global.common.response.enums.CommonResponseCodeEnum;
import com.sinosoftgz.starter.exception.biz.ArgumentNotValidException;
import com.sinosoftgz.starter.exception.biz.BusinessException;
import com.sinosoftgz.starter.utils.lang.Lang;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.NoHandlerFoundException;


/**
 * Created by Roney on 2021/1/4 14:01.
 */
@ControllerAdvice
@ResponseBody
@Slf4j
public class WebExceptionHandler {

    /**
     * 所有的统一返回
     *
     * @param e
     * @return
     */
    @ExceptionHandler(Exception.class)
    public BaseResponse unknownException(Exception e) {
        log.error("发生了未知异常", e);
        if (e instanceof NoHandlerFoundException) {
            return BaseResponse.notFind("未找到页面: " + e.getMessage());
        } else {
            return BaseResponse.error(CommonResponseCodeEnum.UNKNOWN_ERROR.getResultCode(), CommonResponseCodeEnum.UNKNOWN_ERROR.getResultMsg());
        }
    }


    /**
     * 处理自定义的业务异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = BusinessException.class)
    public BaseResponse bizExceptionHandler(BusinessException e) {
        log.error("发生业务异常！原因是：{}", e.getErrorMsg());
        return BaseResponse.error(e.getErrorCode(), e.getErrorMsg());
    }

    /**
     * 处理空指针的异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = NullPointerException.class)
    public BaseResponse nullPointerExceptionHandler(NullPointerException e) {
        log.error("发生空指针异常！原因是:", e);
        return BaseResponse.error(CommonResponseCodeEnum.BAD_REQUEST.getResultCode(), CommonResponseCodeEnum.BAD_REQUEST.getResultMsg());
    }

    /**
     * spring校验错误拦截处理
     *
     * @param exception 错误信息集合
     * @return 错误信息
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public BaseResponse validationBodyException(MethodArgumentNotValidException exception) {
        log.error("发生参数校验异常！原因是:", exception);
        return BaseResponse.error(CommonResponseCodeEnum.PARAMS_VALIDATION_EMTITY.getResultCode(), exception.getMessage());
    }

    /**
     * 自定义校验错误拦截处理
     *
     * @param exception
     * @return
     */
    @ExceptionHandler(ArgumentNotValidException.class)
    public BaseResponse validationBodyException(ArgumentNotValidException exception) {
        log.error("发生自定义参数校验异常！原因是:", exception);
        return BaseResponse.error(CommonResponseCodeEnum.PARAMS_VALIDATION_EMTITY.getResultCode(), exception.getMessage());
    }


    @ExceptionHandler(IllegalArgumentException.class)
    public BaseResponse rpcException(IllegalArgumentException exception) {
        log.error("参数不合法异常！原因是:", exception);
        String message = CommonResponseCodeEnum.PARAMS_VALIDATION_EMTITY.getResultMsg();
        if (!Lang.isEmpty(exception.getLocalizedMessage())) {
            message = exception.getLocalizedMessage();
        }
        return BaseResponse.error(CommonResponseCodeEnum.PARAMS_VALIDATION_EMTITY.getResultCode(), message);
    }


}
