package com.sinosoftgz.starter.sequence.utils;


import com.sinosoftgz.starter.utils.concurrent.ThreadPoolUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.support.rowset.SqlRowSet;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import redis.clients.jedis.exceptions.JedisDataException;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Created by Roney on 2019/3/26 16:50
 *
 * @author Roney
 */

@Slf4j
@Component
@ConfigurationProperties(prefix = "sequence")
public class SequenceUtils implements InitializingBean, DisposableBean {
    private static Map<String, String> createSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> insertSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> querySequenceSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> queryMaxSequenceSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> queryGroupSequenceSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> updateSequenceSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> deleteSequenceSqlMap = new ConcurrentHashMap<String, String>();
    private static Map<String, String> dropSequenceSqlMap = new ConcurrentHashMap<String, String>();

    static {
        createSqlMap
                .put("h2",
                        "CREATE TABLE %s (sequence_key varchar(255) ,sequence number)");
        createSqlMap
                .put("mysql",
                        "CREATE TABLE %s (sequence_key varchar(255) ,sequence bigint)");
        createSqlMap
                .put("informix",
                        "CREATE TABLE %s (sequence_key varchar(255) ,sequence number)");
        createSqlMap
                .put("oracle",
                        "CREATE TABLE %s (sequence_key varchar(255) ,sequence number)");
    }

    static {
        insertSqlMap
                .put("h2",
                        "insert into %s(sequence_key,sequence) values(?,?)");
        insertSqlMap
                .put("mysql",
                        "insert into %s(sequence_key,sequence) values(?,?)");
        insertSqlMap
                .put("informix",
                        "insert into %s(sequence_key,sequence) values(?,?)");
        insertSqlMap
                .put("oracle",
                        "insert into %s(sequence_key,sequence) values(?,?)");
    }

    static {
        querySequenceSqlMap
                .put("h2",
                        "select sequence from  %s where sequence_key=? ");
        querySequenceSqlMap
                .put("mysql",
                        "select sequence from  %s where sequence_key=? ");
        querySequenceSqlMap
                .put("informix",
                        "select sequence from  %s where sequence_key=? ");
        querySequenceSqlMap
                .put("oracle",
                        "select sequence from  %s where sequence_key=? ");
    }

    static {
        queryMaxSequenceSqlMap
                .put("h2",
                        "select max(sequence) from  %s where sequence_key=? ");
        queryMaxSequenceSqlMap
                .put("mysql",
                        "select max(sequence) from  %s where sequence_key=? ");
        queryMaxSequenceSqlMap
                .put("informix",
                        "select max(sequence) from  %s where sequence_key=? ");
        queryMaxSequenceSqlMap
                .put("oracle",
                        "select max(sequence) from  %s where sequence_key=? ");
    }

    static {
        queryGroupSequenceSqlMap
                .put("h2",
                        "select sequence_key  from   %s group by sequence_key having count(*)>1");
        queryGroupSequenceSqlMap
                .put("mysql",
                        "select sequence_key  from   %s group by sequence_key having count(*)>1");
        queryGroupSequenceSqlMap
                .put("informix",
                        "select sequence_key  from   %s group by sequence_key having count(*)>1");
        queryGroupSequenceSqlMap
                .put("oracle",
                        "select sequence_key  from   %s group by sequence_key having count(*)>1");
    }

    static {
        updateSequenceSqlMap
                .put("h2",
                        "update  %s  set sequence=? where (sequence<? or sequence>=?) and sequence_key=? ");
        updateSequenceSqlMap
                .put("mysql",
                        "update  %s  set sequence=? where (sequence<? or sequence>=?) and sequence_key=? ");
        updateSequenceSqlMap
                .put("informix",
                        "update  %s  set sequence=? where (sequence<? or sequence>=?) and sequence_key=? ");
        updateSequenceSqlMap
                .put("oracle",
                        "update  %s  set sequence=? where (sequence<? or sequence>=?) and sequence_key=? ");
    }

    static {
        deleteSequenceSqlMap
                .put("h2",
                        "delete from  %s where sequence_key=? ");
        deleteSequenceSqlMap
                .put("mysql",
                        "delete from  %s where sequence_key=? ");
        deleteSequenceSqlMap
                .put("informix",
                        "delete from  %s where sequence_key=? ");
        deleteSequenceSqlMap
                .put("oracle",
                        "delete from  %s where sequence_key=? ");
    }

    static {
        dropSequenceSqlMap
                .put("h2",
                        "drop table %s ");
        dropSequenceSqlMap
                .put("mysql",
                        "drop table %s ");
        dropSequenceSqlMap
                .put("informix",
                        "drop table %s ");
        dropSequenceSqlMap
                .put("oracle",
                        "drop table %s ");
    }

    private final String ORDERNO_REDIS_KEY = "mall:sequenceNo";
    private Long maxCycle = 1000000L;
    private ExecutorService executorService = null;
    private RedisTemplate redisTemplate;
    private JdbcTemplate jdbcTemplate;

    private final String SEPARATOR = ":";
    /**
     * 数据库类型
     */
    private String databaseType;
    /**
     * 表名
     */
    private String tableName;

    public String getDatabaseType() {
        return databaseType;
    }

    public void setDatabaseType(String databaseType) {
        this.databaseType = databaseType;
    }

    public String getTableName() {
        return tableName;
    }

    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    private String CREATE_SEQUECENCE_TABLE_SQL = null;
    private String INIT_SEQUECENCE_TABLE_SQL = null;
    private String QUERY_SEQUECENCE_TABLE_SQL = null;
    private String QUERY_MAX_SEQUECENCE_TABLE_SQL = null;
    private String QUERY_GROUP_SEQUECENCE_TABLE_SQL = null;
    private String UPDATE_SEQUECENCE_TABLE_SQL = null;
    private String DELETE_SEQUECENCE_TABLE_SQL = null;
    private String DROP_SEQUECENCE_TABLE_SQL = null;

    public SequenceUtils(RedisTemplate redisTemplate, JdbcTemplate jdbcTemplate) {
        this.redisTemplate = redisTemplate;
        this.jdbcTemplate = jdbcTemplate;

    }

    /**
     * 创建序列 默认从0开始,长度为8
     *
     * @param sequenceKey
     * @return
     */
    public String generateSequence(String sequenceKey) {
        return generateSequence(sequenceKey, 8, 0, "");
    }

    /**
     * 创建序列 默认从0开始,长度为8
     *
     * @param sequenceKey
     * @return
     */
    public String generateSequence(String sequenceKey, String prefix) {
        return generateSequence(sequenceKey, 8, prefix);
    }

    /**
     * 创建序列 默认从0开始
     *
     * @param key
     * @param length
     * @return
     */
    public String generateSequence(String key, int length, String prefix) {
        return generateSequence(key, length, 0, prefix);
    }

    /**
     * 生成对连续性无要求，对唯一性有要求的序列号。
     * 没有数据库操作，只使用redis做并发控制，保证唯一性
     *
     * @param pre
     * @return
     */
    public String getRequestId(String pre) {
        DateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        String dateTime = dateFormat.format(new Date());

        String redisKey = ORDERNO_REDIS_KEY + ":REQUESTID:" + pre + ":" + dateTime;
        BoundValueOperations<String, Long> operations = redisTemplate.boundValueOps(redisKey);
        Long next = operations.increment(1);
        if (next == 1) {//新增
            operations.expire(10, TimeUnit.MINUTES);//10分钟后过期，最多会有600个数据，10分钟是为了避免服务器之间时间可能存在的差异
        }
        StringBuilder builder = new StringBuilder();
        builder.append(pre).append(dateTime);
        //补全0
        for (long i = next; i * 10 < maxCycle; i = i * 10) {
            builder.append(0);
        }
        builder.append(next);
        return builder.toString();
    }


    private String generateSequence(String sequenceKey, int length, long start, String prefix) {
        if (ObjectUtils.isEmpty(sequenceKey)) {
            throw new RuntimeException("序列号key不允许为空");
        }
        if (length <= start) {
            log.error("长度小于开始值，序列号生成失败!length：{},start:{}", length, start);
            throw new RuntimeException("长度小于开始值，序列号生成失败!");
        }
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":" + sequenceKey);
        /**
         * 如果redis值为空，初始化redis
         */
        if (boundValueOperations.get() == null) {
            BoundValueOperations sequenceTableLockOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":TABLE:LOCK:" + sequenceKey);
            if (!sequenceTableLockOperations.setIfAbsent(true)) {
                try {
                    Thread.sleep(300);
                } catch (InterruptedException e) {
                    log.error("InterruptedException:", e);
                }
                sequenceTableLockOperations.expire(10, TimeUnit.SECONDS);
                return generateSequence(sequenceKey, length, start, prefix);
            }
            SqlRowSet existSequenceRowSet = jdbcTemplate.queryForRowSet(QUERY_SEQUECENCE_TABLE_SQL, sequenceKey);
            Long existSequence = 0L;
            if (existSequenceRowSet.next()) {
                existSequence = existSequenceRowSet.getLong(1);
            }
            boundValueOperations.set(existSequence);
        }
        long startTime = System.currentTimeMillis();
        int t = 0;
        String sequence = null;
        Exception exception = null;
        /**
         * 超过5次生成序列号失败则抛出异常
         */
        while (t < 5) {
            try {
                Long increment = boundValueOperations.increment(1);
                if (increment >= maxCycle) {
                    BoundValueOperations sequenceLockOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + "CIRCULATE:LOCK:" + sequenceKey);
                    if (sequenceLockOperations.setIfAbsent(true)) {
                        boundValueOperations.set(start);
                        sequenceLockOperations.expire(300, TimeUnit.MICROSECONDS);
                        return generateSequence(sequenceKey, length, start, prefix);
                    }
                    Thread.sleep(200);
                    return generateSequence(sequenceKey, length, start, prefix);
                }

                sequence = prefix + fillLeftZero(increment, length);
                /**
                 * 同步现有sequence到数据库
                 */
                executorService.execute(new Runnable() {
                    @Override
                    public void run() {
                        List lastSequenceList = jdbcTemplate.queryForList(QUERY_SEQUECENCE_TABLE_SQL, sequenceKey);
                        /**
                         * 如果数据库无该sequenceKey记录，初始化
                         */
                        if (CollectionUtils.isEmpty(lastSequenceList)) {
                            BoundValueOperations initDbSequenceLockOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":INIT:LOCK:" + sequenceKey);
                            if (initDbSequenceLockOperations.setIfAbsent(true)) {
                                initDbSequenceLockOperations.expire(1, TimeUnit.MINUTES);
                                jdbcTemplate.update(INIT_SEQUECENCE_TABLE_SQL, sequenceKey, increment);
                            } else {
                                try {
                                    Thread.sleep(1000);
                                } catch (InterruptedException e) {
                                    log.error("InterruptedException:", e);
                                }
                                updateSequence(increment, sequenceKey);
                            }
                        } else {
                            updateSequence(increment, sequenceKey);
                        }
                    }
                });
                break;
            } catch (Exception e) {
                if (e instanceof JedisDataException) {
                    /**
                     * 数据异常，初始化
                     */
                    boundValueOperations.set(0L);
                }
                exception = e;
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e1) {
                    e1.printStackTrace();
                }
                t++;
            }
        }
        if (t >= 5) {
            throw new RuntimeException("生成序列号失败", exception);
        }
        log.debug("生成序列号耗时：{} ", (System.currentTimeMillis() - startTime));
        log.info("sequence--->{}", sequence);
        return sequence;
    }

    private String fillLeftZero(Long max, int length) {
        StringBuilder sb = new StringBuilder();
        sb.append(max);
        if (sb.length() >= length) {
            return sb.substring(sb.length() - length);
        } else {
            for (int i = sb.length(); i < length; i++) {
                sb.insert(0, '0');
            }
        }
        return sb.toString();
    }

    private void updateSequence(Long increment, String sequenceKey) {
        jdbcTemplate.update(UPDATE_SEQUECENCE_TABLE_SQL, new Object[]{increment, increment, maxCycle - 1, sequenceKey});
    }

    @Override
    public void destroy() throws Exception {
        executorService.shutdown();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        /**
         * 需要使用单线程队列方式
         */
        executorService = ThreadPoolUtils.createThreadPool(3, "序列号同步");

        log.debug("初始化sql信息，以及表结构");
        if (ObjectUtils.isEmpty(databaseType) || ObjectUtils.isEmpty(tableName)) {
            throw new RuntimeException("数据库类型和表结构不允许为空");
        }
        CREATE_SEQUECENCE_TABLE_SQL = String.format(createSqlMap.get(databaseType), tableName);
        INIT_SEQUECENCE_TABLE_SQL = String.format(insertSqlMap.get(databaseType), tableName);
        QUERY_SEQUECENCE_TABLE_SQL = String.format(querySequenceSqlMap.get(databaseType), tableName);
        QUERY_MAX_SEQUECENCE_TABLE_SQL = String.format(queryMaxSequenceSqlMap.get(databaseType), tableName);
        QUERY_GROUP_SEQUECENCE_TABLE_SQL = String.format(queryGroupSequenceSqlMap.get(databaseType), tableName);
        UPDATE_SEQUECENCE_TABLE_SQL = String.format(updateSequenceSqlMap.get(databaseType), tableName);
        DELETE_SEQUECENCE_TABLE_SQL = String.format(deleteSequenceSqlMap.get(databaseType), tableName);
        DROP_SEQUECENCE_TABLE_SQL = String.format(dropSequenceSqlMap.get(databaseType), tableName);

        List overSequences = null;
        /**
         * 如果表不存在创建表
         */
        try {
            overSequences = jdbcTemplate.queryForList(QUERY_GROUP_SEQUECENCE_TABLE_SQL, String.class);
        } catch (Exception e) {
            if (e instanceof BadSqlGrammarException) {
                try {
                    jdbcTemplate.execute(DROP_SEQUECENCE_TABLE_SQL);
                } catch (Exception ex) {
                    log.error("Exception:", ex);
                }
                jdbcTemplate.execute(CREATE_SEQUECENCE_TABLE_SQL);
            }
        }
        /**
         * 清除多条相同sequenceKey数据
         */
        if (!CollectionUtils.isEmpty(overSequences)) {
            overSequences.forEach(sequenceKey -> {
                Long maxSequence = jdbcTemplate.queryForObject(QUERY_MAX_SEQUECENCE_TABLE_SQL, Long.class, sequenceKey);
                jdbcTemplate.update(DELETE_SEQUECENCE_TABLE_SQL, sequenceKey);
                jdbcTemplate.update(INIT_SEQUECENCE_TABLE_SQL, sequenceKey, maxSequence);
            });

        }
    }


}
