package com.sinosoftgz.starter.swagger.config;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.sinosoftgz.starter.swagger.properties.Swagger2Properties;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import springfox.documentation.RequestHandler;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.oas.annotations.EnableOpenApi;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;

/**
 * Created by Roney on 2019/8/16.
 *
 * @author Roney
 * @date 2019-08-16 15:18
 */

@Configuration
@EnableOpenApi
@ConditionalOnProperty(prefix = Swagger2Properties.SWAGGER_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = false)
@EnableConfigurationProperties({Swagger2Properties.class})
public class SwaggerConfiguration {
    /**
     * 定义分隔符,配置Swagger多包
     */

    private static final String SPLITOR = ",";

    @Autowired
    Swagger2Properties swagger2Properties;

    @Bean
    public Docket api() {

        return new Docket(DocumentationType.SWAGGER_2)
                .apiInfo(apiInfo())
                .select()
                .apis(basePackage(swagger2Properties.getBasePackage()))
                /**
                 * 加了ApiOperation注解的类，才生成接口文档
                 */
                .apis(RequestHandlerSelectors.withMethodAnnotation(ApiOperation.class))
                .paths(PathSelectors.any())
                .build();
    }

    private ApiInfo apiInfo() {
        return new ApiInfoBuilder()
                .title(swagger2Properties.getTitle())
                .description(swagger2Properties.getDescription())
                .termsOfServiceUrl(swagger2Properties.getTermsOfServiceUrl())
                .version(swagger2Properties.getVersion())
                .contact(swagger2Properties.getContact())
                .build();
    }

    /**
     * 重写basePackage方法，使能够实现多包访问
     *
     * @param basePackage
     * @return com.google.common.base.Predicate<springfox.documentation.RequestHandler>
     */
    private Predicate<RequestHandler> basePackage(final String basePackage) {

        return input -> declaringClass(input).transform(handlerPackage(basePackage)).or(true);
    }

    private Function<Class<?>, Boolean> handlerPackage(final String basePackage) {
        return input -> {
            // 循环判断匹配
            for (String strPackage : basePackage.split(SPLITOR)) {
                boolean isMatch = input.getPackage().getName().startsWith(strPackage);
                if (isMatch) {
                    return true;
                }
            }
            return false;
        };
    }

    private Optional<? extends Class<?>> declaringClass(RequestHandler input) {
        return Optional.fromNullable(input.declaringClass());
    }

}
