package com.sinosoftgz.starter.swagger.controller;

import com.sinosoftgz.starter.swagger.properties.Swagger2Properties;
import com.sinosoftgz.starter.swagger.service.WordService;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.thymeleaf.context.Context;
import org.thymeleaf.spring5.SpringTemplateEngine;


import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Map;

/**
 * Created by Roney on 2020/12/2 10:57.
 */
@Controller
@Api(tags = "To Word API")
@RequestMapping("/swagger")
@ConditionalOnProperty(prefix = Swagger2Properties.SWAGGER_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = false)
@Slf4j
public class Swagger2DocController {

    @Resource
    private WordService tableService;

    @Autowired
    private SpringTemplateEngine springTemplateEngine;

    private String DEFAULT_FILE_NAME = "swagger2word";

    /**
     * 将swagger文档下载为doc文档
     *
     * @param model
     * @param url      需要转换成 word 文档的资源地址
     * @param response
     */
    @ApiOperation(value = "将swagger文档下载为doc文档")
    @ApiResponses(value = {@ApiResponse(code = 200, message = "请求成功。")})
    @RequestMapping(value = "/downloadWord", method = {RequestMethod.GET})
    public void downloadWord(Model model, @ApiParam(value = "资源地址", required = true) @RequestParam(required = true) String url, HttpServletResponse response, @ApiParam(value = "保存文件名称", required = false) @RequestParam(required = false) String fileName) {
        generateModelData(model, url, 1);
        writeContentToResponse(model, fileName, response);
    }

    private void generateModelData(Model model, String url, Integer download) {
        Map<String, Object> result = tableService.tableList(url);
        model.addAttribute("url", url);
        model.addAttribute("download", download);
        model.addAllAttributes(result);
    }

    private void writeContentToResponse(Model model, String fileName, HttpServletResponse response) {
        if (StringUtils.isEmpty(fileName)) {
            fileName = DEFAULT_FILE_NAME;
        }
        Context context = new Context();
        context.setVariables(model.asMap());
        String content = springTemplateEngine.process("word", context);
        response.setContentType("application/octet-stream;charset=utf-8");
        response.setCharacterEncoding("utf-8");
        try (BufferedOutputStream bos = new BufferedOutputStream(response.getOutputStream())) {
            response.setHeader("Content-disposition", "attachment;filename=" + URLEncoder.encode(fileName + ".doc", "utf-8"));
            byte[] bytes = content.getBytes();
            bos.write(bytes, 0, bytes.length);
            bos.flush();
        } catch (IOException e) {
            log.error("异常信息：", e);
        }
    }
}
