package com.sinosoftgz.starter.utils.date;


import cn.hutool.core.date.DatePattern;
import org.joda.time.DateTime;
import org.joda.time.Days;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;

/**
 * 操作日期工具类
 * 操作日期，统一使用joda-time
 * 官网：https://www.joda.org/joda-time/
 * Created by Roney on 2019/3/22 11:19
 *
 * @author Roney
 */
@SuppressWarnings("ALL")
public class DateUtils {
    /**
     * 各种时间格式
     */
    public static final String YYYY_MM_DD = DatePattern.NORM_DATE_PATTERN;
    public static final String YYYY_MM_DDHHMMSS = DatePattern.NORM_DATETIME_PATTERN;
    public static final String YYYYMMDD = DatePattern.PURE_DATE_PATTERN;
    public static final String YYYYMMDDHHMMSSSSS = DatePattern.PURE_DATETIME_MS_PATTERN;
    public static final String YYYYMMDDHHMMSS = DatePattern.PURE_DATETIME_PATTERN;
    public static final String YYYY_MM_DDHHMM = DatePattern.NORM_DATETIME_MINUTE_PATTERN;
    public static final String YYYYMMDDHHMMSS_CHINA = DatePattern.CHINESE_DATE_TIME_PATTERN;
    public static final String YYYY_MM_DD_HHMMSSSSS = DatePattern.ISO8601_PATTERN;

    private DateUtils() {
    }

    /**
     * 新的一年
     *
     * @param fromDate
     * @return
     */
    public static Days daysToNewYear(LocalDate fromDate) {
        LocalDate newYear = fromDate.plusYears(1).withDayOfYear(1);
        return Days.daysBetween(fromDate, newYear);
    }

    /**
     * 获取当前时间
     *
     * @param dateTime
     * @param format
     * @return
     */
    public static String now(DateTime dateTime, String format) {
        DateTimeFormatter formatter = DateTimeFormat.forPattern(format);
        String dateTimeStr = formatter.print(dateTime);
        return dateTimeStr;
    }

    /**
     * 获取当前时间
     *
     * @param format
     * @return
     */
    public static String now(String format) {
        DateTimeFormatter formatter = DateTimeFormat.forPattern(format);
        String dateTimeStr = formatter.print(new DateTime());
        return dateTimeStr;
    }


    /**
     * 明天
     *
     * @param dateTime
     * @param format
     * @return
     */
    public static String tomorrow(String format) {
        DateTime dateTime = new DateTime();
        dateTime = dateTime.plusDays(1);
        DateTimeFormatter formatter = DateTimeFormat.forPattern(format);
        String dateTimeStr = formatter.print(dateTime);
        return dateTimeStr;
    }

    /**
     * 明年
     *
     * @param dateTime
     * @param format
     * @return
     */
    public static String nextYear(String format) {
        DateTime dateTime = new DateTime();
        dateTime = dateTime.plusYears(1);
        DateTimeFormatter formatter = DateTimeFormat.forPattern(format);
        String dateTimeStr = formatter.print(dateTime);
        return dateTimeStr;
    }


    /**
     * 获取日期
     *
     * @return
     */
    public static Date now() {
        DateTime dateTime = new DateTime();
        return dateTime.toDate();
    }


    /**
     * 当天凌晨 00:00:00
     *
     * @return
     */
    public static Date startOfDay() {
        DateTime nowTime = new DateTime();
        DateTime startOfDay = nowTime.withTimeAtStartOfDay();
        return startOfDay.toDate();
    }

    /**
     * 当天结束时间 23:59:59
     *
     * @return
     */
    public static Date endOfDay() {
        DateTime nowTime = new DateTime();
        DateTime endOfDay = nowTime.millisOfDay().withMaximumValue();
        return endOfDay.toDate();
    }

    /**
     * timestamp转成LocalDateTime
     *
     * @param timestamp
     * @return
     */
    public static LocalDateTime getDateTimeOfTimestamp(long timestamp) {
        Instant instant = Instant.ofEpochMilli(timestamp);
        ZoneId zone = ZoneId.systemDefault();
        return LocalDateTime.ofInstant(instant, zone);
    }

}
