package com.sinosoftgz.starter.utils.lang;


import lombok.extern.slf4j.Slf4j;
import com.sinosoftgz.starter.utils.convert.Converters;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Java通用工具类
 *
 * @author Roney
 */
@SuppressWarnings("ALL")
@Slf4j
public class Lang {

    private static final String DATE_FORMAT_YYYYMMDDHHMMSS_SSS = "yyyyMMddHHmmssSSS";


    private Lang() {
    }

    /**
     * 空对象
     */
    public static final Object EMPTY = new Object();
    /**
     * 空数组
     */
    public static final Object[] EMPTY_ARRAY = new Object[]{};

    /**
     * 获取对象系统引用哈希值（不为负数）
     *
     * @param x
     * @return
     */
    public static long identityHashCode(Object x) {
        return (long) System.identityHashCode(x) + (long) Integer.MAX_VALUE;
    }

    /**
     * 将UUID转换为22位字符串，依据Base64编码（URL安全）
     *
     * @return
     */
    public static String id() {
        OID oid = new OID();
        return oid.toString();
    }

    /**
     * 将CheckedException转换为RuntimeException.
     */
    public static RuntimeException unchecked(Throwable e) {
        if (e instanceof RuntimeException) {
            return (RuntimeException) e;
        } else {
            return new RuntimeException(e);
        }
    }

    /**
     * 将CheckedException转换为RuntimeException.
     */
    public static RuntimeException unchecked(Throwable e, String message,
                                             Object... args) {
        return new RuntimeException(String.format(message, args), e);
    }

    /**
     * 判断一个对象是否是空对象
     *
     * @param obj
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static boolean isEmpty(Object obj) {
        if (obj == null) {
            return true;
        }
        if (obj instanceof CharSequence) {
            return obj.toString().trim().length() == 0;
        }
        if (obj.getClass().equals(Object.class)) {
            return true;
        }
        if (isBaseType(obj.getClass())) {
            return false;
        }
        if (obj instanceof Map) {
            return ((Map) obj).isEmpty();
        }
        if (obj instanceof Collection) {
            return ((Collection) obj).isEmpty();
        }
        if (obj.getClass().isArray()) {
            return Array.getLength(obj) == 0;
        }
        return Object.class.equals(obj.getClass());
    }

    public static boolean isNotEmpty(Object object) {
        if (object != null && !"".equals(object) && !"null".equals(object)) {
            return (true);
        }
        return (false);
    }

    /**
     * 判断一个类型是否是基本类型
     *
     * @param type
     * @return
     */
    public static boolean isBaseType(Class<?> type) {
        if (type.isPrimitive()) {
            return true;
        }
        if (CharSequence.class.isAssignableFrom(type)) {
            return true;
        }
        if (Number.class.isAssignableFrom(type)) {
            return true;
        }
        if (Date.class.isAssignableFrom(type)) {
            return true;
        }
        if (Boolean.class.equals(type)) {
            return true;
        }
        if (Character.class.equals(type)) {
            return true;
        }
        if (Class.class.equals(type)) {
            return true;
        }
        if (StringBuilder.class.equals(type)) {
            return true;
        }
        if (StringBuffer.class.equals(type)) {
            return true;
        }
        if (Object.class.equals(type)) {
            return true;
        }
        if (Void.class.equals(type)) {
            return true;
        }
        return false;
    }

    /**
     * 判断是否是数字类型
     *
     * @param type
     * @return
     */
    public static boolean isNumber(Class<?> type) {
        if (Number.class.isAssignableFrom(type)) {
            return true;
        }
        if (type.equals(int.class)) {
            return true;
        }
        if (type.equals(short.class)) {
            return true;
        }
        if (type.equals(long.class)) {
            return true;
        }
        if (type.equals(float.class)) {
            return true;
        }
        if (type.equals(double.class)) {
            return true;
        }
        if (type.equals(byte.class)) {
            return true;
        }
        return false;
    }

    /**
     * 获得本源异常信息
     *
     * @param e
     * @return
     */
    public static Throwable getCause(Throwable e) {
        return e.getCause() == null ? e : getCause(e.getCause());
    }

    /**
     * 输出对象字符串格式
     *
     * @param obj
     * @return
     */
    public static String toString(Object obj) {
        return toString(obj, null);
    }

    /**
     * 输出对象字符串格式
     *
     * @param obj
     * @return
     */
    public static String toString(Object obj, String format) {
        if (obj == null) {
            return "null";
        }
        if (obj instanceof Throwable) {
            Throwable throwable = (Throwable) obj;
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            throwable.printStackTrace(pw);
            pw.flush();
            pw.close();
            sw.flush();
            return sw.toString();
        }
        if (obj instanceof Date) {
            return new SimpleDateFormat(
                    format == null || format.trim().length() == 0 ? DATE_FORMAT_YYYYMMDDHHMMSS_SSS
                            : format).format((Date) obj);
        }
        if (isNumber(obj.getClass())) {
            if (format != null && format.trim().length() != 0) {
                return new DecimalFormat(format).format(obj);
            }
        }
        return String.valueOf(obj);
    }

    /**
     * 将一个对象转换为指定类型
     *
     * @param value
     * @param type
     * @return
     */
    public static <T> T convert(Object value, Class<T> type) {
        return convert(value, type, null);
    }

    /**
     * 将一个对象转换为指定类型
     *
     * @param value
     * @param type
     * @return
     */
    public static <T> T convert(Object value, Class<T> type, String format) {
        return Converters.BASE.convert(value, type, format);
    }

    /**
     * 新建一个Set
     *
     * @param args
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> Set<T> newSet(T... args) {
        int length = args == null ? 1 : args.length;
        Set<T> set = new HashSet<T>(length);
        if (args == null) {
            set.add(null);
        } else {
            for (int i = 0; i < args.length; i++) {
                set.add(args[i]);
            }
        }
        return set;
    }

    /**
     * 新建一个List
     *
     * @param args
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> List<T> newList(T... args) {
        int length = args == null ? 1 : args.length;
        List<T> list = new ArrayList<T>(length);
        if (args == null) {
            list.add(null);
        } else {
            for (int i = 0; i < args.length; i++) {
                list.add(args[i]);
            }
        }
        return list;
    }

    /**
     * 抛出一个带消息的异常
     *
     * @param type
     * @param message
     * @param args
     * @return
     */
    public static <T extends Throwable> T newThrowable(Class<T> type,
                                                       String message, Object... args) {
        try {
            return type.getConstructor(String.class).newInstance(
                    String.format(message, args));
        } catch (InstantiationException e) {
            throw Lang.unchecked(e, message, args);
        } catch (IllegalAccessException e) {
            throw Lang.unchecked(e, message, args);
        } catch (IllegalArgumentException e) {
            throw Lang.unchecked(e, message, args);
        } catch (InvocationTargetException e) {
            throw Lang.unchecked(e, message, args);
        } catch (NoSuchMethodException e) {
            throw Lang.unchecked(e, message, args);
        } catch (SecurityException e) {
            throw Lang.unchecked(e, message, args);
        }
    }

    /**
     * 抛出一个带消息的运行时异常
     *
     * @param message
     * @param args
     * @return
     */
    public static IllegalStateException newThrowable(String message,
                                                     Object... args) {
        return newThrowable(IllegalStateException.class, message, args);
    }

    /**
     * 新建一个Map，必须是偶数个参数
     *
     * @param args
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <K, V> Map<K, V> newMap(Object... args) {
        Map<K, V> map = new HashMap<K, V>();
        if (args != null) {
            if (args.length % 2 != 0) {
                throw new IllegalArgumentException(
                        "The number of arguments must be an even number");
            }
            for (int i = 0; i < args.length; i += 2) {
                map.put((K) args[i], (V) args[i + 1]);
            }
        }
        return map;
    }


    /**
     * 比较两个对象是否相同，对于数字、日期等按照大小进行比较，自动兼容包装器实例
     *
     * @param a
     * @param b
     * @return
     */
    public static boolean equals(Object a, Object b) {
        if (a == b) {
            return true;
        }
        if (a == null || b == null) {
            return false;
        }
        if (a.equals(b)) {
            return true;
        }
        // 比较大数字
        if (isNumber(a.getClass()) && isNumber(b.getClass())) {
            return new BigDecimal(a.toString()).compareTo(new BigDecimal(b
                    .toString())) == 0;
        }
        // 比较日期
        if (a instanceof Date && b instanceof Date) {
            return ((Date) a).compareTo((Date) b) == 0;
        }
        return false;
    }

    /**
     * 计时执行
     *
     * @return 返回runnable的执行时间
     */
    public static long timing(Runnable runnable) {
        long begin = System.currentTimeMillis();
        try {
            runnable.run();
            return System.currentTimeMillis() - begin;
        } catch (Throwable e) {
            throw unchecked(e);
        }
    }


    /**
     * 判断是否为真，不为真则抛出异常
     *
     * @param flag    真假标志位
     * @param message 消息体，可带格式，将使用String.format进行格式化
     * @param args    格式化参数，可为空
     */
    public static void isTrue(boolean flag, String message, Object... args) {
        if (!flag) {
            throw new IllegalArgumentException(String.format(message, args));
        }
    }

    /**
     * 判断是否非null，为null则抛出异常
     *
     * @param object  要判断的对象
     * @param message 消息体，可带格式，将使用String.format进行格式化
     * @param args    格式化参数，可为空
     */
    public static void notNull(Object object, String message, Object... args) {
        isTrue(object != null, message, args);
    }

    /**
     * 判断是否非空，为空则抛出异常
     *
     * @param object  要判断的对象
     * @param message 消息体，可带格式，将使用String.format进行格式化
     * @param args    格式化参数，可为空
     */
    public static void notEmpty(Object object, String message, Object... args) {
        isTrue(!Lang.isEmpty(object), message, args);
    }


    /**
     * 获取最初的消息异常
     *
     * @param e
     * @return
     */
    public static Throwable getMessageCause(Throwable e) {
        while (e != null && e.getMessage() == null && e.getCause() != null) {
            e = e.getCause();
        }
        return e;
    }


    public static String generateDynamic(int len) {
        boolean isDigit = false;
        boolean isLetter = false;
        final int maxNum = 36;
        int m = 0;
        StringBuffer pwd = null;
        while (!isDigit || !isLetter) {
            isDigit = false;
            isLetter = false;
            pwd = new StringBuffer("");
            int i; // 生成的随机数
            int count = 0;
            char[] str = {'4', 'b', 'c', '8', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
                    'l', '6', 'n', 'o', 'p', 'q', 'r', 's', '0', 'u', 'v', 'w',
                    '1', 'y', 'z', 't', 'x', '2', '3', 'a', '5', 'm', '7', 'd', '9'};

            Random r = new Random();
            while (count < len) {
                // 生成随机数，取绝对值，防止生成负数，
                // 生成的数最大为36-1

                i = Math.abs(r.nextInt(maxNum));
                if (i >= 0 && i < str.length) {
                    pwd.append(str[i]);
                    count++;
                }
            }
            //循环遍历字符串
            for (int j = 0; j < pwd.toString().length(); j++) {
                //用char包装类中的判断数字的方法判断每一个字符
                if (Character.isDigit(pwd.toString().charAt(j))) {
                    isDigit = true;
                }
                //用char包装类中的判断字母的方法判断每一个字符
                if (Character.isLetter(pwd.toString().charAt(j))) {
                    isLetter = true;
                }
            }
            m++;
            log.info("--------------" + "第" + m + "次生成密码：" + pwd.toString() + "--------------------");
        }

        return pwd.toString();
    }

    public static BigDecimal fourCutFiveIn(double d, int decimalPlace) {
        BigDecimal bigDecimal = new BigDecimal(d);
        return bigDecimal.setScale(decimalPlace, BigDecimal.ROUND_HALF_UP);
    }

    public static BigDecimal fourCutFiveIn(BigDecimal bigDecimal, int decimalPlace) {
        return bigDecimal.setScale(decimalPlace, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * 从列表中提取属性集合，支持Map列表及JavaBean列表
     *
     * @param list   支持Map列表（List<Map>）及JavaBean列表(List<Bean>)
     * @param key:   String 查找属性名
     * @param value: T 返回集合中的类型
     * @param <T>
     * @return
     */
    public static <T> List<T> grepList(List list, String key, T value) {
        List<T> grepedList = null;
        Object v;
        if (!isEmpty(list)) {
            grepedList = new ArrayList();
            Object obj = list.get(0);
            if (obj != null) {
                /** 判断成员类型 */
                if (obj instanceof Map) {
                    for (Object o : list) {
                        v = ((Map) o).get(key);
                        if (v != null) {
                            grepedList.add((T) v);
                        }
                    }
                } else {
                    /** 处理Bean */
                    Class clz = obj.getClass();
                    try {
                        BeanInfo beanInfo = Introspector.getBeanInfo(clz);
                        PropertyDescriptor[] propDescs = beanInfo.getPropertyDescriptors();
                        for (Object o : list) {
                            for (int i = 0; i < propDescs.length; i++) {
                                if (propDescs[i].getName().equals(key)) {
                                    v = propDescs[i].getReadMethod().invoke(o);
                                    if (v != null) {
                                        grepedList.add((T) v);
                                    }
                                }
                            }
                        }

                    } catch (InvocationTargetException e) {
                        throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                    } catch (IllegalAccessException e) {
                        throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                    } catch (IntrospectionException e) {
                        throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                    }

                }
            }
        }

        return grepedList;
    }

    /**
     * 从列表中查找，支持Map列表及JavaBean列表
     * 使用：
     * 1. 从List<Map>查找
     * List<Map> list = new ArrayList() {{
     * add(new HashMap(){{
     * put("key1", "key1_v1");
     * put("key2", "key2_v1");
     * }});
     * <p>
     * add(new HashMap(){{
     * put("key1", "key1_v2");
     * put("key2", "key2_v2");
     * }});
     * }}
     *
     * <b>Lang.findInBeanList(list, "key1", "key2_v2");</b>
     * <p>
     * 2. 从List<JavaBean>查找
     * List<TestBean> list = new ArrayList() {{
     * add(new TestBean(){{
     * setId("id1");
     * setName("bean1");
     * }});
     * <p>
     * add(new TestBean(){{
     * setId("id2");
     * setName("bean2");
     * }});
     * }}
     * <b>Lang.findInBeanList(list, "id", "id2");</b>
     *
     * @param list   支持Map列表（List<Map>）及JavaBean列表(List<Bean>)
     * @param key:   String 查找属性名，即Map的key或JavaBean的属性名
     * @param value: Object 需匹配的属性值
     * @return
     */
    public static <T> T findInBeanList(List<T> list, String key, Object value) {
        T result = null;
        Object v;
        if (!isEmpty(list)) {
            Object obj = list.get(0);
            /** 判断成员类型 */
            if (obj instanceof Map) {
                for (Object o : list) {
                    v = ((Map) o).get(key);
                    if (v != null) {
                        if (v.equals(value)) {
                            result = (T) o;
                            break;
                        }
                    }
                }
            } else {
                /** 处理Bean */
                try {
                    BeanInfo beanInfo = Introspector.getBeanInfo(obj.getClass());
                    PropertyDescriptor[] propDescs = beanInfo.getPropertyDescriptors();
                    for (Object o : list) {
                        for (int i = 0; i < propDescs.length; i++) {
                            if (propDescs[i].getName().equals(key)) {
                                v = propDescs[i].getReadMethod().invoke(o);
                                if (v != null && v.equals(value)) {
                                    result = (T) o;
                                    break;
                                }
                            }
                        }
                    }
                } catch (IntrospectionException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (InvocationTargetException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (IllegalAccessException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (ClassCastException e) {
                    throw new RuntimeException("ClassCastException !", e);
                } catch (Exception e) {
                    throw new RuntimeException("findInBeanList !", e);
                }

            }
        }

        return result;
    }

    public static <T> T findInBeanList(T[] beans, String key, Object value) {
        if (beans == null) {
            return null;
        }
        return findInBeanList(Arrays.asList(beans), key, value);
    }

    /**
     * 从列表中查找，支持Map列表及JavaBean列表
     * 使用：
     * 1. 从List<Map>查找
     * List<Map> list = new ArrayList() {{
     * add(new HashMap(){{
     * put("key1", "key1_v1");
     * put("key2", "key2_v1");
     * }});
     * <p>
     * add(new HashMap(){{
     * put("key1", "key1_v2");
     * put("key2", "key2_v2");
     * }});
     * }}
     *
     * <b>Lang.findInBeanList(list, "key1", "key2_v2");</b>
     * <p>
     * 2. 从List<JavaBean>查找
     * List<TestBean> list = new ArrayList() {{
     * add(new TestBean(){{
     * setId("id1");
     * setName("bean1");
     * }});
     * <p>
     * add(new TestBean(){{
     * setId("id2");
     * setName("bean2");
     * }});
     * }}
     * <b>Lang.findInBeanList(list, "id", "id2");</b>
     *
     * @param list   支持Map列表（List<Map>）及JavaBean列表(List<Bean>)
     * @param key:   String 查找属性名，即Map的key或JavaBean的属性名
     * @param value: Object 需匹配的属性值
     * @return
     */
    public static <T> List<T> findAllInBeanList(List<T> list, String key, Object value) {
        List<T> result = new ArrayList();
        Object v;
        if (!isEmpty(list)) {
            Object obj = list.get(0);
            /** 判断成员类型 */
            if (obj instanceof Map) {
                for (Object o : list) {
                    v = ((Map) o).get(key);
                    if (v != null) {
                        if (v.equals(value)) {
                            result.add((T) o);
                            break;
                        }
                    }
                }
            } else {
                /** 处理Bean */
                try {
                    BeanInfo beanInfo = Introspector.getBeanInfo(obj.getClass());
                    PropertyDescriptor[] propDescs = beanInfo.getPropertyDescriptors();
                    for (Object o : list) {
                        for (int i = 0; i < propDescs.length; i++) {
                            if (propDescs[i].getName().equals(key)) {
                                v = propDescs[i].getReadMethod().invoke(o);
                                if (v.equals(value)) {
                                    result.add((T) o);
                                    break;
                                }
                            }
                        }
                    }
                } catch (IntrospectionException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (InvocationTargetException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (IllegalAccessException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                }

            }
        }

        return result;
    }

    public static <T> List<T> findAllInBeanList(T[] beans, String key, Object value) {
        if (beans == null) {
            return null;
        }
        return findAllInBeanList(Arrays.asList(beans), key, value);
    }

    /**
     * 在JavaBean集合中，提取指定key值相同的记录，放到Map中，以key值作为索引
     *
     * @param list
     * @param key
     * @param <T>
     * @return Map<String, List < T>>
     */
    public static <T> Map<Object, List<T>> beanListGroupBy(List<T> list, Object key) {
        Map<Object, List<T>> result = null;
        Object v;
        List<T> groupedBeanList;
        if (!isEmpty(list)) {
            result = new HashMap();
            Object obj = list.get(0);
            /** 判断成员类型 */
            if (obj instanceof Map) {
                for (Object o : list) {
                    v = ((Map) o).get(key);
                    if (v != null) {
                        groupedBeanList = result.get(v);
                        if (groupedBeanList == null) {
                            groupedBeanList = new ArrayList();
                        }
                        groupedBeanList.add((T) o);
                        result.put(v, groupedBeanList);
                    }
                }
            } else {
                /** 处理Bean */
                try {
                    BeanInfo beanInfo = Introspector.getBeanInfo(obj.getClass());
                    PropertyDescriptor[] propDescs = beanInfo.getPropertyDescriptors();
                    for (Object o : list) {
                        for (int i = 0; i < propDescs.length; i++) {
                            if (propDescs[i].getName().equals(key)) {
                                v = propDescs[i].getReadMethod().invoke(o);
                                if (v != null) {
                                    groupedBeanList = result.get(v);
                                    if (groupedBeanList == null) {
                                        groupedBeanList = new ArrayList();
                                    }
                                    groupedBeanList.add((T) o);
                                    result.put(v, groupedBeanList);
                                }
                            }
                        }
                    }
                } catch (IntrospectionException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (InvocationTargetException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                } catch (IllegalAccessException e) {
                    throw new RuntimeException("Not getter for key [" + key + "] in bean list !", e);
                }

            }
        }

        return result;
    }

    /**
     * 在List中查找指定值
     *
     * @param list
     * @param value
     * @param <T>
     * @return
     */
    public static <T> T findInList(Iterable<T> list, T value) {
        Iterator<T> it = list.iterator();
        T tmp;
        while (it.hasNext()) {
            tmp = it.next();
            if (tmp.equals(value)) {
                return tmp;
            }
        }

        return null;
    }


    /**
     * 去除Map中的空值
     *
     * @param source
     * @param casde  是否级联
     * @return
     */
    public static <K, V> Map<K, V> filterNullMap(Map<K, V> source, Boolean casde) {
        Iterator<Map.Entry<K, V>> it = source.entrySet().iterator();
        V val;
        while (it.hasNext()) {
            Map.Entry<K, V> entry = it.next();
            val = entry.getValue();
            if (val == null) {
                it.remove();
            } else {
                if (casde) {
                    if (val instanceof Iterable) {
                        Iterator<Object> cit = ((Iterable) val).iterator();
                        Object tmp;
                        while (cit.hasNext()) {
                            tmp = cit.next();
                            if (tmp instanceof Map) {
                                filterNullMap((Map<Object, Object>) tmp, casde);
                            }
                        }
                    } else if (val instanceof Map) {
                        filterNullMap((Map<Object, Object>) val, casde);
                    }
                }
            }
        }

        return source;
    }

    /**
     * 将集合按指定数量分组
     *
     * @param list
     * @param quantity
     * @return 返回分组后的List -> List<List<T>>
     */
    public static List groupListByQuantity(List list, int quantity) {
        if (list == null || list.size() == 0) {
            return list;
        }

        if (quantity <= 0) {
            new IllegalArgumentException("Wrong quantity.");
        }

        List wrapList = new ArrayList();
        int count = 0;
        while (count < list.size()) {
            wrapList.add(new ArrayList(list.subList(count, (count + quantity) > list.size() ? list.size() : count + quantity)));
            count += quantity;
        }

        return wrapList;
    }

    /**
     * 拆分列表
     *
     * @param list     List<T>
     * @param pageSize 每个拆分单元列表最大大小
     * @param <T>      泛型
     * @return List<List < T>>
     */
    public static <T> List<List<T>> splitList(List<T> list, int pageSize) {
        int listSize = list.size();
        int page = (listSize + (pageSize - 1)) / pageSize;

        List<List<T>> listArray = new ArrayList<>();
        for (int i = 0; i < page; i++) {
            List<T> subList = new ArrayList<>();
            for (int j = 0; j < listSize; j++) {
                int pageIndex = ((j + 1) + (pageSize - 1)) / pageSize;
                if (pageIndex == (i + 1)) {
                    subList.add(list.get(j));
                }
                if ((j + 1) == ((j + 1) * pageSize)) {
                    break;
                }
            }
            listArray.add(subList);
        }
        return listArray;
    }

    /**
     * 将驼峰命名转化成下划线
     *
     * @param para
     * @return
     */
    public static String camelToUnderline(String para) {
        StringBuilder sb = new StringBuilder(para);
        int temp = 0;//定位
        for (int i = 0; i < para.length(); i++) {
            if (Character.isUpperCase(para.charAt(i))) {
                sb.insert(i + temp, "_");
                temp += 1;
            }
        }
        return sb.toString().toLowerCase();
    }
}
