package com.sinosoftgz.starter.utils.web;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.sinosoftgz.starter.utils.web.enums.UserPlatformType;
import org.springframework.util.StringUtils;
import com.sinosoftgz.starter.utils.web.vo.UserAgent;

import javax.servlet.http.HttpServletRequest;


/**
 * Created by Roney on 2019/3/26 16:50
 *
 * @author Roney
 */
public class UserAgentUtils {

    private static final String USER_AGENT = "USER-AGENT";

    // \b 是单词边界(连着的两个(字母字符 与 非字母字符) 之间的逻辑上的间隔),
    // 字符串在编译时会被转码一次,所以是 "\\b"
    // \B 是单词内部逻辑间隔(连着的两个字母字符之间的逻辑上的间隔)
    private static final String PHONE_REG = "\\b(ip(hone|od)|android|opera m(ob|in)i" + "|windows (phone|ce)|blackberry" + "|s(ymbian|eries60|amsung)|p(laybook|alm|rofile/midp" + "|laystation portable)|nokia|fennec|htc[-_]" + "|mobile|up.browser|[1-4][0-9]{2}x[1-4][0-9]{2})\\b";
    private static final String TABLE_REG = "\\b(ipad|tablet|(Nexus 7)|up.browser" + "|[1-4][0-9]{2}x[1-4][0-9]{2})\\b";

    public static final Map<String, String> PLATFORM_TYPE_MAP = new LinkedHashMap<String, String>() {{
        put("nexus", "安卓平板");
        put("droid", "安卓平板");
        put("xoom", "安卓平板");
        put("playbook", "PlayBook");
        put("ipad", "ios平版");
        put("iphone", "苹果手机");
        put("symbian", "塞班手机");
        put("windows phone", "winphone");
        put("android", "安卓手机");
        put("mobile", "手机版");
        put("windows", "微软电脑");
        put("mac os x", "苹果电脑");
    }};


    public static final Map<String, String> BROWSER_TYPE_MAP = new LinkedHashMap<String, String>() {{
        put("micromessenger", "微信浏览器");
        put("opera", "opera浏览器");
        put("360se", "360浏览器");
        put("qqbrowser", "QQ浏览器");
        put("lbbrowser", "猎豹浏览器");
        put("baidubrowser", "百度浏览器");
        put("bidubrowser", "百度浏览器");
        put("baiduboxapp", "百度手机APP");
        put("maxthon", "遨游浏览器");
        put("sogoumobilebrowser", "搜狗手机浏览器");
        put("metasr", "搜狗浏览器");
        put("ucweb", "uc浏览器");
        put("ucbrowser", "uc浏览器");
        put("The World", "世界之窗");
        put("avant", "AvantBrowser");
        put("tencenttraveler", "腾讯TT");
        put("msie 8", "IE8浏览器");
        put("msie", "IE浏览器");
        put("firefox", "火狐");
        put("chrome", "谷歌浏览器");
        put("safari", "safari");
    }};

    private UserAgentUtils() {
    }

    /**
     * @param request
     * @return
     */
    public static UserAgent getUserAgent(HttpServletRequest request) {
        String userAgent = request.getHeader(USER_AGENT);
        return getUserAgent(userAgent);
    }

    /**
     * 用途：根据客户端 User Agent Strings 判断其浏览器、操作平台
     * if 判断的先后次序：
     * 根据设备的用户使用量降序排列，这样对于大多数用户来说可以少判断几次即可拿到结果：
     * >>操作系统:windows > 苹果 > 安卓 > Linux > ...
     * >>Browser:Chrome > FF > IE > ...
     *
     * @param userAgent
     * @return
     */
    public static UserAgent getUserAgent(String userAgent) {
        if (StringUtils.isEmpty(userAgent)) {
            return null;
        }
        String tUserAgent = userAgent.toLowerCase();

        String platformType = null;
        for (String key : PLATFORM_TYPE_MAP.keySet()) {
            if (tUserAgent.contains(key)) {
                platformType = PLATFORM_TYPE_MAP.get(key);
                break;
            }
        }
        UserAgent obj = judgeBrowser(tUserAgent, platformType);
        return obj;
    }

    /**
     * 用途：根据客户端 User Agent Strings 判断其浏览器
     * if 判断的先后次序：
     * 根据浏览器的用户使用量降序排列，这样对于大多数用户来说可以少判断几次即可拿到结果：
     * >>Browser:Chrome > FF > IE > ...
     *
     * @param userAgent:user  agent
     * @param platformType:平台
     * @return
     */
    private static UserAgent judgeBrowser(String userAgent, String platformType) {

        UserAgent obj = new UserAgent();
        obj.setUserAgent(userAgent);
        obj.setPlatformType(platformType);

        String browserType = null;
        for (String key : BROWSER_TYPE_MAP.keySet()) {
            if (userAgent.contains(key)) {
                browserType = BROWSER_TYPE_MAP.get(key);
                break;
            }
        }
        obj.setBrowserType(browserType);
        obj.setHeadMessage(userAgent);
        generatorBuyType(userAgent, obj);


        return obj;

    }

    private static void generatorBuyType(String userAgent, UserAgent obj) {
        // 移动设备正则匹配：手机端、平板
        Pattern phonePat = Pattern.compile(PHONE_REG, Pattern.CASE_INSENSITIVE);
        Pattern tablePat = Pattern.compile(TABLE_REG, Pattern.CASE_INSENSITIVE);
        // 匹配
        Matcher matcherPhone = phonePat.matcher(userAgent);
        Matcher matcherTable = tablePat.matcher(userAgent);
        if (matcherTable.find()) {
            obj.setUserPlatformType(UserPlatformType.IPAD.getType());
        } else if (matcherPhone.find()) {
            obj.setUserPlatformType(UserPlatformType.MOBILE.getType());
        } else {
            obj.setUserPlatformType(UserPlatformType.PC.getType());
        }
    }
}
