package com.sinosoftgz.starter.apollo.refresh;

import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfig;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.sinosoftgz.starter.apollo.cache.BeanCache;
import com.sinosoftgz.starter.apollo.utils.ArraysUtils;
import com.sinosoftgz.starter.apollo.utils.LogChangeKeyUtils;
import com.sinosoftgz.starter.apollo.annotation.RefreshBean;
import com.sinosoftgz.starter.apollo.properties.InsConfigProperties;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;


/**
 * Created by Roney on 2020/12/1 18:52.
 * 刷新带有@ConditionalOnProperty注解的bean
 * 1、当满足条件注解时，则手动创建bean，然后配合@ApolloConfigChangeListener监听该bean的属性变化。当该bean属性有变化时，手动把属性注入bean。同时刷新依赖该bean的其他bean
 * 2、当不满足条件注解时，则手动从spring容器中移除bean，同时刷新依赖该bean的其他bean
 */
@Component
@Slf4j
public class ConditionalOnPropertyRefreshConfig implements ApplicationContextAware {

    private static final String LOG_TITLE = "刷新带有@ConditionalOnProperty注解的bean";
    private ApplicationContext applicationContext;

    @ApolloConfig
    private Config config;

    @Resource
    private BeanCache beanCache;

    private final InsConfigProperties insConfigProperties;

    public ConditionalOnPropertyRefreshConfig(final InsConfigProperties insConfigProperties, final ApplicationContext applicationContext) {
        this.insConfigProperties = insConfigProperties;
        //this.beanCache = beanCache;
        this.applicationContext = applicationContext;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @ApolloConfigChangeListener()
    private void refresh(ConfigChangeEvent changeEvent) {
        log.info(LOG_TITLE + " The to refresh bean name is {}. ", this.insConfigProperties.getToRefreshBeanNames());
        Optional<String[]> beanArrayOptional = this.insConfigProperties.toRefreshBeanNameArray();
        if (beanArrayOptional.isPresent()) {
            Collection<Class> conditionalClasses = beanCache.getConditionalClassesMap().values();
            if (!CollectionUtils.isEmpty(conditionalClasses)) {
                for (Class conditionalClass : conditionalClasses) {
                    ConditionalOnProperty conditionalOnProperty = (ConditionalOnProperty) conditionalClass.getAnnotation(ConditionalOnProperty.class);
                    if (conditionalOnProperty != null) {
                        String[] conditionalOnPropertyKeys = conditionalOnProperty.name();
                        String beanChangeCondition = this.getChangeKey(changeEvent, conditionalOnPropertyKeys);
                        String conditionalOnPropertyValue = conditionalOnProperty.havingValue();
                        boolean isChangeBean = this.changeBean(conditionalClass, beanChangeCondition, conditionalOnPropertyValue);
                        if (!isChangeBean) {
                            // 更新相应的bean的属性值，主要是存在@ConfigurationProperties注解的bean
                            applicationContext.publishEvent(new EnvironmentChangeEvent(changeEvent.changedKeys()));
                        }
                    }
                }
            }
            LogChangeKeyUtils.printChange(LOG_TITLE, insConfigProperties.isEnabledLog(), changeEvent);
        }
    }

    private boolean isChangeKey(ConfigChangeEvent changeEvent, String conditionalOnPropertyKey) {
        Set<String> changeKeys = changeEvent.changedKeys();
        if (!CollectionUtils.isEmpty(changeKeys) && changeKeys.contains(conditionalOnPropertyKey)) {
            return true;
        }
        return false;
    }

    private String getChangeKey(ConfigChangeEvent changeEvent, String[] conditionalOnPropertyKeys) {
        if (ArraysUtils.isEmpty(conditionalOnPropertyKeys)) {
            return null;
        }
        String changeKey = null;
        for (String conditionalOnPropertyKey : conditionalOnPropertyKeys) {
            if (isChangeKey(changeEvent, conditionalOnPropertyKey)) {
                changeKey = conditionalOnPropertyKey;
                break;
            }
        }

        return changeKey;
    }

    /**
     * 根据条件对bean进行注册或者移除
     *
     * @param conditionalClass
     * @param beanChangeCondition        bean发生改变的条件
     * @param conditionalOnPropertyValue
     */
    private boolean changeBean(Class conditionalClass, String beanChangeCondition, String conditionalOnPropertyValue) {
        boolean isNeedRegisterBeanIfKeyChange = this.isNeedRegisterBeanIfKeyChange(beanChangeCondition, conditionalOnPropertyValue);
        boolean isNeedRemoveBeanIfKeyChange = this.isNeedRemoveBeanIfKeyChange(beanChangeCondition, conditionalOnPropertyValue);
        String beanName = StringUtils.uncapitalize(conditionalClass.getSimpleName());
        if (isNeedRegisterBeanIfKeyChange) {
            boolean isAlreadyRegisterBean = this.isExistBean(beanName);
            if (!isAlreadyRegisterBean) {
                this.registerBean(beanName, conditionalClass);
                return true;
            }
        } else if (isNeedRemoveBeanIfKeyChange) {
            this.unregisterBean(beanName);
            this.refreshBeanDependChangeBean(conditionalClass);
            return true;
        }
        return false;
    }

    /**
     * bean注册
     *
     * @param beanName
     * @param beanClass
     */
    public void registerBean(String beanName, Class beanClass) {
        log.info("registerBean->beanName:{},beanClass:{}", beanName, beanClass);
        BeanDefinitionBuilder beanDefinitionBurinilder = BeanDefinitionBuilder.genericBeanDefinition(beanClass);
        BeanDefinition beanDefinition = beanDefinitionBurinilder.getBeanDefinition();
        setBeanField(beanClass, beanDefinition);
        getBeanDefinitionRegistry().registerBeanDefinition(beanName, beanDefinition);
        refreshBeanDependChangeBean(beanClass);

    }

    /**
     * 设置bean字段值
     *
     * @param beanClass
     * @param beanDefinition
     */
    private void setBeanField(Class beanClass, BeanDefinition beanDefinition) {
        ConfigurationProperties configurationProperties = (ConfigurationProperties) beanClass.getAnnotation(ConfigurationProperties.class);
        if (!ObjectUtils.isEmpty(configurationProperties)) {
            String prefix = configurationProperties.prefix();
            for (String propertyName : config.getPropertyNames()) {
                String fieldPrefix = prefix + ".";
                if (propertyName.startsWith(fieldPrefix)) {
                    String fieldName = propertyName.substring(fieldPrefix.length());
                    String fieldVal = config.getProperty(propertyName, null);
                    log.info("setBeanField-->fieldName:{},fieldVal:{}", fieldName, fieldVal);
                    beanDefinition.getPropertyValues().add(fieldName, fieldVal);
                }
            }
        }
    }


    /**
     * bean移除
     *
     * @param beanName
     */
    public void unregisterBean(String beanName) {
        log.info("unregisterBean->beanName:{}", beanName);
        getBeanDefinitionRegistry().removeBeanDefinition(beanName);
    }


    public boolean isExistBean(String beanName) {
        return applicationContext.containsBean(beanName);
    }


    public boolean isNeedRegisterBeanIfKeyChange(String changeKey, String conditionalOnPropertyValue) {
        if (StringUtils.isEmpty(changeKey)) {
            return false;
        }
        String apolloConfigValue = config.getProperty(changeKey, null);
        return conditionalOnPropertyValue.equals(apolloConfigValue);
    }

    private boolean isNeedRemoveBeanIfKeyChange(String changeKey, String conditionalOnPropertyValue) {
        if (!StringUtils.isEmpty(changeKey)) {
            String apolloConfigValue = config.getProperty(changeKey, null);
            return !conditionalOnPropertyValue.equals(apolloConfigValue);
        }

        return false;

    }

    /**
     * 刷新依赖依赖的bean
     *
     * @param conditionalClass
     */
    private void refreshBeanDependChangeBean(final Class conditionalClass) {
        beanCache.getRefreshBeanClassesMap().forEach((refreshBeanName, refreshBeanClass) -> {
            RefreshBean refreshBean = (RefreshBean) refreshBeanClass.getAnnotation(RefreshBean.class);
            for (Class fieldBeanClass : refreshBean.refreshFieldBeans()) {
                if (conditionalClass.getName().equals(fieldBeanClass.getName())) {
                    log.info("refreshBeanDependChangeBean-->refreshBeanName:{},refreshBeanClass:{}", refreshBeanName, refreshBeanClass);
                    BeanDefinition beanDefinition = getBeanDefinitionRegistry().getBeanDefinition(refreshBeanName);
                    beanDefinition.setBeanClassName(refreshBeanClass.getName());
                    getBeanDefinitionRegistry().registerBeanDefinition(refreshBeanName, beanDefinition);
                }
            }

        });

    }

    private BeanDefinitionRegistry getBeanDefinitionRegistry() {
        ConfigurableApplicationContext configurableContext = (ConfigurableApplicationContext) applicationContext;
        BeanDefinitionRegistry beanDefinitionRegistry = (DefaultListableBeanFactory) configurableContext.getBeanFactory();
        return beanDefinitionRegistry;
    }
}
