package com.sinosoftgz.starter.jdbc.page;


import com.github.drinkjava2.jdialects.Dialect;
import com.sinosoftgz.global.common.request.page.PageQueryRequest;
import com.sinosoftgz.global.common.response.page.ResultPage;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;

/**
 * Created by Roney on 2019/9/6.
 * JDBC分页查询
 *
 * @author Roney
 * @date 2019-09-06 21:05
 */
@Component
@Slf4j
public class JdbcPageUtils {

    private JdbcTemplate jdbcTemplate;

    public JdbcPageUtils(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    /**
     * 分页查询，返回对象
     *
     * @param sql
     * @param elementType      返回需要封装的java 对象
     * @param pageQueryRequest 分页参数
     * @param args             sql参数
     * @return ResultPage 分页对象
     */
    public <T> ResultPage<T> query(@NotNull String sql, @NotNull Class<T> elementType, @NotNull PageQueryRequest pageQueryRequest, Map<String, Object> args) {
        long time = System.currentTimeMillis();
        NamedParameterJdbcTemplate namedParameterJdbcTemplate = new NamedParameterJdbcTemplate(jdbcTemplate.getDataSource());
        SqlReCreateFactory sqlReCreateFactory = new SqlReCreateFactory<Object>(sql, pageQueryRequest.getPageNum(), pageQueryRequest.getPageSize(), args).invoke();
        sql = sqlReCreateFactory.getSql();
        ResultPage<T> resultPage = sqlReCreateFactory.getResultPage();
        log.debug("获取数据库连接耗时------：{} ms", System.currentTimeMillis() - time);
        long startTime = System.currentTimeMillis();
        List<T> result = namedParameterJdbcTemplate.query(sql, args, new BeanPropertyRowMapper<>(elementType));
        log.debug("执行查询sql耗时------：{} ms", System.currentTimeMillis() - startTime);
        resultPage.setData(result);
        return resultPage;
    }

    /**
     * 分页查询，返回List<Map<String,Object>>
     *
     * @param sql              默认封装成List<Map<String,Object>>
     * @param pageQueryRequest 分页参数
     * @param args             sql参数
     * @return ResultPage 分页对象
     */
    public ResultPage query(@NotNull String sql, @NotNull PageQueryRequest pageQueryRequest, Map<String, Object> args) {
        NamedParameterJdbcTemplate namedParameterJdbcTemplate = new NamedParameterJdbcTemplate(jdbcTemplate.getDataSource());
        SqlReCreateFactory sqlReCreateFactory = new SqlReCreateFactory<Object>(sql, pageQueryRequest.getPageNum(), pageQueryRequest.getPageSize(), args).invoke();
        sql = sqlReCreateFactory.getSql();
        ResultPage<Map<String, Object>> resultPage = sqlReCreateFactory.getResultPage();
        List<Map<String, Object>> result = namedParameterJdbcTemplate.queryForList(sql, args);
        resultPage.setData(result);
        return resultPage;
    }

    @Data
    private class SqlReCreateFactory<T> {
        private String sql;
        private int pageNum;
        private int pageSize;
        private Map<String, Object> args;
        private ResultPage<T> resultPage;


        public SqlReCreateFactory(String sql, int pageNum, int pageSize, Map<String, Object> args) {
            this.sql = sql;
            this.pageNum = pageNum;
            this.pageSize = pageSize;
            this.args = args;
        }


        public SqlReCreateFactory invoke() {
            NamedParameterJdbcTemplate namedParameterJdbcTemplate = new NamedParameterJdbcTemplate(jdbcTemplate.getDataSource());
            resultPage = new ResultPage<T>(new Long(0), new Long(pageSize));
            String sqlCount = "select count(*) from (" + sql + ")  count_table";
            Integer count = namedParameterJdbcTemplate.queryForObject(sqlCount, args, Integer.class); //查询页数
            resultPage.setTotalCount(count);
            int pageNum = 0;
            if (count != null && count > 0) {
                if (count % pageSize == 0) {
                    pageNum = count / pageSize;
                } else {
                    pageNum = count / pageSize + 1;
                }
                resultPage.setPageNo(pageNum);
            }

            Dialect dialect = Dialect.guessDialect(jdbcTemplate.getDataSource());
            sql = dialect.pagin(this.pageNum, pageSize, sql);
            log.info("sql:{}", sql);
            return this;
        }
    }
}
