package com.sinosoftgz.starter.jdbc.utils;


import cn.hutool.core.bean.BeanUtil;
import com.google.common.collect.Lists;

import com.sinosoftgz.starter.utils.lang.NamingUtils;
import javafx.util.Pair;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.NotNull;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * Created by Roney on 2020/12/4 10:13.
 */
@Component
@Slf4j
@Data
public class JdbcTemplateUtils {

    private JdbcTemplate jdbcTemplate;


    /**
     * @param jdbcTemplate
     */
    public JdbcTemplateUtils(JdbcTemplate jdbcTemplate) {
        this(jdbcTemplate, "");
    }

    /**
     * @param jdbcTemplate
     * @param tableName
     */
    public JdbcTemplateUtils(JdbcTemplate jdbcTemplate, @NotNull String tableName) {
        this.jdbcTemplate = jdbcTemplate;
    }

    /**
     * 新增数据
     *
     * @param map 字段和Value值，放入Map中入库
     */
    public Integer insert(@NotNull String tableName, Map<String, Object> map) {
        Pair<String, Object[]> pair = SqlBuilderUtils.buildInsert(tableName, map);
        return getJdbcTemplate().update(pair.getKey(), pair.getValue());
    }

    /**
     * 批量插入数据
     *
     * @param listData
     */
    public int[] batchInsert(@NotNull String tableName, final List<Map<String, Object>> listData) {
        Pair<String, List<Object[]>> pair = SqlBuilderUtils.buildBatchInsert(tableName, listData);
        return jdbcTemplate.batchUpdate(pair.getKey(), pair.getValue());
    }

    /**
     * 批量插入数据，不考虑是否重复等因素
     *
     * @param tableName 表名
     * @param fields    字段列表
     * @param list      数据列表
     * @throws Exception
     */
    public void batchInsert(@NotNull String tableName, final String[] fields, final List<Map<String, Object>> list) throws Exception {
        batchInsert(tableName, fields, list, false);
    }

    /**
     * 批量新增，默认不使用insert ignore的形式插入数据
     *
     * @param tableName
     * @param fields
     * @param list
     * @param ignore    是否重复，如果重复的话，会使用insert ignore忽略插入的形式进行数据插入
     * @throws Exception
     */
    public void batchInsert(@NotNull String tableName, final String[] fields, final List<Map<String, Object>> list, boolean ignore) {

        StringBuffer sql = new StringBuffer("insert ")
                .append(ignore ? " ignore into " : " into ")
                .append(tableName).append(" (").append(SqlBuilderUtils.buildFields(fields)).append(") values (")
                .append(SqlBuilderUtils.buildQuestion(fields)).append(");");

        getJdbcTemplate().batchUpdate(sql.toString(), new BatchPreparedStatementSetter() {

            @Override
            public void setValues(PreparedStatement ps, int i) throws SQLException {
                for (int j = 0, len = fields.length; j < len; j++) {
                    ps.setObject(j + 1, list.get(i).get(fields[j]));
                }
            }

            @Override
            public int getBatchSize() {
                return list.size();
            }
        });
    }

    /**
     * 批量插入数据，由于JDBCTemplate不支持批量插入后返回批量id，所以此处使用jdbc原生的方法实现此功能
     *
     * @param tableName
     * @param fields
     * @param list
     * @return
     * @throws Exception
     */
    public List<Long> batchAddWithId(@NotNull String tableName, String[] fields, final List<Map<String, Object>> list) throws Exception {

        StringBuffer sql = new StringBuffer("insert into ")
                .append(tableName)
                .append("(")
                .append(SqlBuilderUtils.buildFields(fields))
                .append(") values (")
                .append(SqlBuilderUtils.buildQuestion(fields))
                .append(");");

        Connection con = getJdbcTemplate().getDataSource().getConnection();
        con.setAutoCommit(false);
        PreparedStatement pstmt = con.prepareStatement(sql.toString(), PreparedStatement.RETURN_GENERATED_KEYS);

        for (Map<String, Object> map : list) {
            for (int j = 0, len = fields.length; j < len; j++) {
                pstmt.setObject(j + 1, map.get(fields[j]));
            }
            pstmt.addBatch();
        }

        pstmt.executeBatch();
        con.commit();
        ResultSet rs = pstmt.getGeneratedKeys(); //获取结果
        List<Long> ids = new ArrayList<Long>();
        while (rs.next()) {
            ids.add(rs.getLong(1));//取得ID
        }
        con.close();
        pstmt.close();
        rs.close();
        return ids;
    }

    /**
     * 更新
     *
     * @param sql    自定义更新sql
     * @param params 查询条件对应的参数(List<Object>)
     * @return int 更新的数量
     */
    public int update(String sql, List<Object> params) {
        return getJdbcTemplate().update(sql, params.toArray());
    }

    /**
     * 修改数据
     *
     * @param map      修改的字段集合
     * @param whereMap where 参数集合
     */
    public int update(@NotNull String tableName, Map<String, Object> map, LinkedHashMap<String, Object> whereMap) {
        Pair<String, Object[]> pair = SqlBuilderUtils.buildUpdateSql(tableName, map, whereMap);
        return getJdbcTemplate().update(pair.getKey(), pair.getValue());
    }


    /**
     * 自定义查询
     *
     * @param sql
     * @return
     */
    public Map<String, Object> findOne(@NotNull String sql) {
        return getJdbcTemplate().queryForMap(sql);
    }

    /**
     * 自定义传参查询
     *
     * @param sql
     * @param params
     * @return
     */
    public Map<String, Object> findOne(@NotNull String sql, Object[] params) {
        return getJdbcTemplate().queryForMap(sql, params);
    }

    /**
     * 根据id字段进行查询
     *
     * @param tableName
     * @param idFiledName
     * @param id
     * @return
     */
    public Map<String, Object> findById(@NotNull String tableName, @NotNull String idFiledName, @NotNull String id) {
        return getJdbcTemplate().queryForMap("SELECT * FROM " + SqlBuilderUtils.buildTableName(tableName) + " WHERE " + NamingUtils.camelToUnderline(idFiledName) + " = ?", id);
    }

    /**
     * 根据ID查询 返回实体对象
     *
     * @param tableName
     * @param idFiledName
     * @param id
     * @param elementType
     * @param <T>
     * @return
     */
    public <T> T findById(@NotNull String tableName, @NotNull String idFiledName, @NotNull String id, Class<T> elementType) {
        Map<String, Object> queryResultMap = this.findById(tableName, idFiledName, id);
        return BeanUtil.mapToBean(queryResultMap, elementType, true);
    }

    /**
     * 原生的 queryForList查询
     *
     * @param sql
     * @return
     */
    public List<Map<String, Object>> queryForList(@NotNull String sql) {
        log.debug("queryForList查询语句为：{}", sql);
        return getJdbcTemplate().queryForList(sql);
    }

    /**
     * 原生的 queryForList查询 自定义传参
     *
     * @param sql
     * @param params
     * @return
     */
    public List<Map<String, Object>> queryForList(@NotNull String sql, Object[] params) {
        log.debug("queryForList查询语句为：{}", sql);
        return getJdbcTemplate().queryForList(sql, params);
    }


    /**
     * 原生的 queryForList 不能直接转到实体，会产生异常，原生只支持基本类型转换，不支持实体类型转换
     *
     * @param sql
     * @param elementType
     * @param <T>
     * @return
     */
    public <T> List<T> queryForList(@NotNull String sql, Class<T> elementType) {
        List<T> listResult = Lists.newArrayList();
        List<Map<String, Object>> listQueryResultMap = queryForList(sql);
        if (CollectionUtils.isEmpty(listQueryResultMap)) {
            return null;
        }
        listQueryResultMap.forEach(item -> {
            listResult.add(BeanUtil.mapToBean(item, elementType, true));
        });
        return listResult;
    }

    /**
     * 原生的 queryForList 不能直接转到实体，会产生异常，原生只支持基本类型转换，不支持实体类型转换 自定义传参
     *
     * @param sql
     * @param params
     * @param elementType
     * @param <T>
     * @return
     */
    public <T> List<T> queryForList(@NotNull String sql, Object[] params, Class<T> elementType) {
        List<T> listResult = Lists.newArrayList();
        List<Map<String, Object>> listQueryResultMap = queryForList(sql, params);
        if (CollectionUtils.isEmpty(listQueryResultMap)) {
            return null;
        }
        listQueryResultMap.forEach(item -> {
            listResult.add(BeanUtil.mapToBean(item, elementType, true));
        });
        return listResult;
    }

    /**
     * 直接用sql查询转实体
     *
     * @param sql
     * @param params
     * @param beanPropertyRowMapper
     * @param <T>
     * @return
     */
    public <T> List<T> queryForList(@NotNull String sql, Object[] params, BeanPropertyRowMapper<T> beanPropertyRowMapper) {
        List<T> listResult = jdbcTemplate.query(sql, params, beanPropertyRowMapper);
        return listResult;
    }

    /**
     * 删除数据
     *
     * @param id
     */
    public Integer delete(@NotNull String tableName, @NotNull String idFiledName, @NotNull String id) {
        StringBuffer sql = new StringBuffer("DELETE FROM ")
                .append(SqlBuilderUtils.buildTableName(tableName)).append(" WHERE " + idFiledName + "= ? ");
        return getJdbcTemplate().update(sql.toString(), new Object[]{id});
    }


    /**
     * 总数
     *
     * @param tableName 表名
     * @param alias     主表别名
     * @param where     查询条件
     * @return
     * @throws Exception
     */
    public long count(@NotNull String tableName, @NotNull String alias, Map<String, String> where) {
        String sql = SqlBuilderUtils.buildCountSql(tableName, alias, where);
        return getJdbcTemplate().queryForObject(sql, Long.class);
    }

    /**
     * 根据Where条件求Count
     *
     * @param tableName
     * @param where
     * @return
     */
    public long count(String tableName, Map<String, String> where) {
        return count(tableName, "1", where);
    }
}
