package com.sinosoftgz.global.common.request.page;


import com.sinosoftgz.global.common.constants.CommonConstants;
import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;


/**
 * Created by Roney on 2019/4/3 18:12
 *
 * @author Roney
 * <p>
 * 分页查询带请求实体查询基础类
 */
@NoArgsConstructor
public class PageQueryRequest<T> extends AbstractPageRequest<T> implements Serializable {


    public static final String ASC = "ASC";
    public static final String DESC = "DESC";


    /**
     * 排序列
     */
    Map<String, String> orderByColumn = new LinkedHashMap();


    public PageQueryRequest(@NotNull Integer pageNum, @NotNull Integer pageSize) {
        super(pageNum, pageSize);

    }

    public PageQueryRequest(@NotNull Integer pageNum, @NotNull Integer pageSize, Map<String, String> orderByColumn) {
        super(pageNum, pageSize);
        this.orderByColumn = orderByColumn;
    }


    public static PageQueryRequest of() {
        return new PageQueryRequest(DEFAULT_PAGE_NUM, DEFAULT_PAGE_SIZE);
    }

    public static PageQueryRequest of(@NotNull Integer pageNum) {
        return new PageQueryRequest(pageNum, DEFAULT_PAGE_SIZE);
    }

    public static PageQueryRequest of(@NotNull Integer pageNum, @NotNull Integer pageSize) {
        return new PageQueryRequest(pageNum, pageSize);
    }


    /**
     * 将pageQueryRequest转成Pageable
     *
     * @param pageQueryRequest
     * @return
     */
    public static Pageable convertToPageable(PageQueryRequest pageQueryRequest) {
        Assert.notNull(pageQueryRequest, CommonConstants.MessageString.ASSERT_NOT_NULL_OBJECT_MESSAGE);
        Map<String, String> orderByColumn = pageQueryRequest.getOrderByColumn();
        Pageable pageable = null;
        List<Sort.Order> orders = new ArrayList<>();
        if (!ObjectUtils.isEmpty(orderByColumn) && orders.size() > 0) {
            for (String key : orderByColumn.keySet()) {
                String value = orderByColumn.get(key);
                Sort.Order order = null;
                if (PageQueryRequest.DESC.equals(value)) {
                    order = new Sort.Order(Sort.Direction.DESC, key);
                } else {
                    order = new Sort.Order(Sort.Direction.ASC, key);
                }
                orders.add(order);
            }
        }
        if (!ObjectUtils.isEmpty(orders) && orders.size() > 0) {
            Sort sort = Sort.by(orders);
            pageable = PageRequest.of(pageQueryRequest.getPageNum(), pageQueryRequest.getPageSize(), sort);
        } else {
            pageable = PageRequest.of(pageQueryRequest.getPageNum(), pageQueryRequest.getPageSize());
        }

        return pageable;
    }


    public Map<String, String> getOrderByColumn() {
        return orderByColumn;
    }

    public void setOrderByColumn(Map<String, String> orderByColumn) {
        this.orderByColumn = orderByColumn;
    }
}
