package com.sinosoftgz.global.common.response;

import com.sinosoftgz.global.common.response.enums.CommonResponseCodeEnum;

import lombok.Data;
import lombok.ToString;

import java.io.Serializable;

/**
 * @author Roney
 * @date 2017-09-26
 */
@SuppressWarnings("ALL")
@Data
@ToString
public class BaseResponse<T> extends AbstractBaseResponse implements Serializable {
    private static final long serialVersionUID = 1L;

    /**
     * 成功标志
     */
    private boolean success = true;

    /**
     * 返回处理消息
     */
    private String message = CommonResponseCodeEnum.SUCCESS.getResultMsg();

    /**
     * 业务返回信息描述
     */
    private String businessMessage;

    /**
     * 返回代码
     */
    private String code = CommonResponseCodeEnum.SUCCESS.getResultCode();

    /**
     * 返回数据对象 data
     */
    private T data;


    public BaseResponse() {

    }

    /**
     * 请求成功
     *
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> ok() {
        return ok(null);
    }

    /**
     * 请求成功
     *
     * @param result
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> ok(T result) {
        BaseResponse<T> r = new BaseResponse<T>();
        r.setData(result);
        return r;
    }

    public static <T> BaseResponse<T> ok(String code, String message) {
        BaseResponse<T> r = new BaseResponse<T>();
        r.setCode(code);
        r.setMessage(message);
        return r;
    }

    public static <T> BaseResponse<T> ok(String code, String message, T result) {
        BaseResponse<T> r = ok(code, message);
        r.setData(result);
        return r;
    }

    /**
     * 请求失败（如：请求参数类型不匹配）
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> bad(String message) {
        return error(CommonResponseCodeEnum.BAD_REQUEST.getResultCode(), message);
    }

    /**
     * 请求失败（如：未登录或未通过验证）
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> unauthorized(String message) {
        return error(CommonResponseCodeEnum.UNAUTHORIZED.getResultCode(), message);
    }

    /**
     * 请求失败（如：权限不足）
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> forbidden(String message) {
        return error(CommonResponseCodeEnum.FORBIDDEN.getResultCode(), message);
    }

    /**
     * 请求失败（如：请求资源不存在）
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> notFind(String message) {
        return error(CommonResponseCodeEnum.NOT_FOUND.getResultCode(), message);
    }

    /**
     * 请求失败（如：查询数据库报错）
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> error(String message) {
        return error(CommonResponseCodeEnum.SERVER_ERROR.getResultCode(), message);
    }


    public static <T> BaseResponse<T> error(T result) {
        BaseResponse<T> r = new BaseResponse<T>();
        r.setCode(CommonResponseCodeEnum.SERVER_ERROR.getResultCode());
        r.setMessage(CommonResponseCodeEnum.SERVER_ERROR.getResultMsg());
        r.setData(result);
        return r;
    }

    public static <T> BaseResponse<T> error() {
        return error(CommonResponseCodeEnum.SERVER_ERROR.getResultCode(), CommonResponseCodeEnum.SERVER_ERROR.getResultMsg());
    }

    /**
     * 请求失败（如：请求频繁或系统维护）
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> BaseResponse<T> unavailable(String message) {
        return error(CommonResponseCodeEnum.SERVER_UNAVAILABLE.getResultCode(), message);
    }


    public static <T> BaseResponse<T> error(String code, String message) {
        BaseResponse<T> r = new BaseResponse<T>();
        r.setSuccess(Boolean.FALSE);
        r.setCode(code);
        r.setMessage(message);
        return r;
    }


    public static <T> BaseResponse<T> error(String code, String message, T result) {
        BaseResponse<T> r = error(code, message);
        r.setData(result);
        return r;
    }


    // 类方法
    public BaseResponse<T> code(String code) {
        this.code = code;
        return this;
    }

    public BaseResponse<T> message(String message) {
        this.message = message;
        return this;
    }

    public BaseResponse<T> result(T result) {
        this.data = result;
        return this;
    }

}
