package com.sinosoftgz.starter.okhttp.config;

import com.sinosoftgz.starter.okhttp.properties.OkHttpProperties;
import com.sinosoftgz.starter.okhttp.properties.ProxyProperties;
import com.sinosoftgz.starter.okhttp.utils.OkHttpUtils;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.util.concurrent.TimeUnit;

/**
 * Created by Roney on 2019/3/20 19:29
 *
 * @author Roney
 */
@EnableConfigurationProperties({ProxyProperties.class, OkHttpProperties.class})
@Configuration
@Slf4j
public class OkHttpConfiguration {

    private ProxyProperties proxyProperties;

    private OkHttpProperties okHttpProperties;

    public OkHttpConfiguration(ProxyProperties proxyProperties, OkHttpProperties okHttpProperties) {
        this.proxyProperties = proxyProperties;
        this.okHttpProperties = okHttpProperties;
    }

    @Bean
    OkHttpClient okHttpClient() {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        customizeOkHttp(builder);
        builder.addInterceptor(new Interceptor() {
            @Override
            public Response intercept(Chain chain) throws IOException {
                //这个chain里面包含了request和response，所以你要什么都可以从这里拿
                Request request = chain.request();
                //请求发起的时间
                long t1 = System.nanoTime();
                log.info(String.format("发送请求: [%s] on %s%n请求头:%s%n请求参数:%s%n",
                        request.url(), chain.connection(), request.headers(), request.body()));

                Response response = chain.proceed(request);
                //收到响应的时间
                long t2 = System.nanoTime();

                //这里不能直接使用response.body().string()的方式输出日志
                //因为response.body().string()之后，response中的流会被关闭，程序会报错，我们需要创建出一
                //个新的response给应用层处理
                ResponseBody responseBody = response.peekBody(1024 * 1024);

                log.info(String.format("接收响应:[%s] %n返回json:[%s]%n请求耗时:%.1fms%n%s",
                        response.request().url(),
                        responseBody.string(),
                        (t2 - t1) / 1e6d,
                        response.headers()));

                return response;
            }
        });
        return builder.build();
    }

    /**
     * 自定义一些参数
     *
     * @param builder
     */
    private void customizeOkHttp(OkHttpClient.Builder builder) {
        if (okHttpProperties.getEnabled()) {
            builder.connectTimeout(okHttpProperties.getConnectTimeout(), TimeUnit.SECONDS).readTimeout(okHttpProperties.getReadTimeout(), TimeUnit.SECONDS).writeTimeout(okHttpProperties.getWriteTimeout(), TimeUnit.SECONDS);
            builder.connectionPool(pool());
        }
        if (proxyProperties.getEnabled()) {
            builder.proxy(new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyProperties.getHost(), proxyProperties.getPort())));
        }
        if (proxyProperties.getEnabledAuthenticator()) {
            builder.proxyAuthenticator(new Authenticator() {
                @Override
                public Request authenticate(Route route, Response response) throws IOException {
                    //设置代理服务器账号密码
                    String credential = Credentials.basic(proxyProperties.getUsername(), proxyProperties.getPassword());
                    return response.request().newBuilder()
                            .header("Proxy-Authorization", credential)
                            .build();
                }
            });
        }
    }

    @Bean
    public ConnectionPool pool() {
        return new ConnectionPool(okHttpProperties.getMaxIdle(), okHttpProperties.getKeepAliveDuration(), TimeUnit.MINUTES);
    }

    @Bean
    @ConditionalOnMissingBean
    public OkHttpUtils okHttpUtils(OkHttpClient okHttpClient) {
        OkHttpUtils okHttpUtils = new OkHttpUtils();
        okHttpUtils.setOkHttpClient(okHttpClient);
        return okHttpUtils;
    }
}
