package com.sinosoftgz.starter.mail.core.biz.impl;

import com.sinosoftgz.global.common.request.BaseRequest;
import com.sinosoftgz.starter.mail.core.biz.MailSendBiz;
import com.sinosoftgz.starter.mail.core.enums.MailBodyType;
import com.sinosoftgz.starter.mail.core.enums.MailPlatformType;
import com.sinosoftgz.starter.mail.core.handler.AbstractMailHandler;
import com.sinosoftgz.starter.mail.core.handler.MailHandlerContext;
import com.sinosoftgz.starter.mail.core.properties.MailPlatformProperties;
import com.sinosoftgz.starter.mail.core.request.JavaMailClientReq;
import com.sinosoftgz.starter.mail.core.request.MailSendReq;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author: Roney
 * @date: 2022-05-09 11:18
 * @description: 发送邮件业务逻辑实现
 */
@Component
@Slf4j
public class MailSendBizImpl implements MailSendBiz {

    @Resource
    MailHandlerContext mailHandlerContext;

    @Autowired
    MailPlatformProperties mailPlatformProperties;


    @Override
    public boolean sendSimpleMail(String recipientTo, String subject, String content) {
        checkRecipientTo(recipientTo);
        try {
            AbstractMailHandler handler = getAbstractMailHandler();
            JavaMailClientReq javaMailClientReq = new JavaMailClientReq.JavaMailClientReqBuilder()
                    .subject(subject)
                    .content(content)
                    .recipientTos(Arrays.asList(recipientTo))
                    .mailBodyType(MailBodyType.Text)
                    .mailPlatformType(getMailPlatformType().getType())
                    .build();
            return handler.send(javaMailClientReq);
        } catch (Exception e) {
            log.error("发送邮件异常", e);
            return false;
        }


    }


    @Override
    public boolean sendHtmlMail(String recipientTo, String subject, String content) {
        checkRecipientTo(recipientTo);
        try {
            AbstractMailHandler handler = getAbstractMailHandler();
            JavaMailClientReq javaMailClientReq = new JavaMailClientReq.JavaMailClientReqBuilder()
                    .subject(subject)
                    .content(content)
                    .recipientTos(Arrays.asList(recipientTo))
                    .mailBodyType(MailBodyType.HTML)
                    .mailPlatformType(getMailPlatformType().getType())
                    .build();
            return handler.send(javaMailClientReq);
        } catch (Exception e) {
            log.error("发送邮件异常", e);
            return false;
        }

    }


    @Override
    public boolean sendMail(BaseRequest<MailSendReq> mailSendReq) {
        checkMailSendReq(mailSendReq);
        try {
            AbstractMailHandler handler = getAbstractMailHandler();
            MailSendReq mailSendReqReq = mailSendReq.getData();
            MailBodyType mailBodyType = MailBodyType.Text;
            if (!Objects.isNull(mailSendReqReq.getMailBodyType())) {
                mailBodyType = mailSendReqReq.getMailBodyType();
            }
            JavaMailClientReq javaMailClientReq = new JavaMailClientReq.JavaMailClientReqBuilder()
                    .subject(mailSendReqReq.getSubject())
                    .content(mailSendReqReq.getContent())
                    .recipientTos(mailSendReqReq.getRecipientTos())
                    .recipientCc(mailSendReqReq.getRecipientCc())
                    .recipientBcc(mailSendReqReq.getRecipientBcc())
                    .attachments(mailSendReqReq.getFiles())
                    .images(mailSendReqReq.getImages())
                    .mailBodyType(mailBodyType)
                    .mailPlatformType(getMailPlatformType().getType())
                    .build();
            return handler.send(javaMailClientReq);
        } catch (Exception e) {
            log.error("发送邮件异常", e);
            return false;
        }

    }

    /**
     * 校验参数信息
     *
     * @param mailSendReq
     */
    private void checkMailSendReq(BaseRequest<MailSendReq> mailSendReq) {
        Assert.notNull(mailSendReq, "邮件发送请求对象不能为空");
        Assert.notNull(mailSendReq.getData(), "邮件发送请求对象不能为空");
        Assert.notEmpty(mailSendReq.getData().getRecipientTos(), "收件人不能为空");
        if (!CollectionUtils.isEmpty(mailSendReq.getData().getImages())) {
            AtomicBoolean imageDataNullFlag = new AtomicBoolean(false);
            mailSendReq.getData().getImages().parallelStream().forEach(image -> {
                boolean checkDataFlag = !StringUtils.hasLength(image.getBase64Str()) && (null == image.getData() || image.getData().length == 0);
                if (checkDataFlag) {
                    imageDataNullFlag.set(true);

                }
            });
            if (imageDataNullFlag.get()) {
                throw new IllegalArgumentException("邮件图片信息存在空");
            }
        }

        if (!CollectionUtils.isEmpty(mailSendReq.getData().getFiles())) {
            AtomicBoolean fileDataNullFlag = new AtomicBoolean(false);
            mailSendReq.getData().getFiles().parallelStream().forEach(attachment -> {
                boolean checkDataFlag = !StringUtils.hasLength(attachment.getBase64Str()) && (null == attachment.getData() || attachment.getData().length == 0);
                if (checkDataFlag) {
                    fileDataNullFlag.set(true);
                }
            });
            if (fileDataNullFlag.get()) {
                throw new IllegalArgumentException("邮件附件信息存在空");
            }
        }
    }

    @Override
    public boolean batchSendMail(BaseRequest<List<MailSendReq>> mailSendReqs) {
        Assert.notNull(mailSendReqs, "邮件发送请求对象不能为空");
        Assert.notEmpty(mailSendReqs.getData(), "邮件发送请求对象不能为空");
        try {
            List<MailSendReq> mailSendReqList = mailSendReqs.getData();
            if (!CollectionUtils.isEmpty(mailSendReqList)) {
                mailSendReqList.parallelStream().forEach((mailSendReq -> {
                    this.sendMail(BaseRequest.simple(mailSendReq));
                }));
            }
        } catch (Exception e) {
            log.error("发送邮件异常", e);
            return false;
        }
        return true;
    }

    /**
     * 校验收件人信息
     *
     * @param recipientTo
     */
    private void checkRecipientTo(String recipientTo) {
        Assert.notNull(recipientTo, "收件人不能为空");
    }

    /**
     * 获取处理器信息
     *
     * @return
     */
    private AbstractMailHandler getAbstractMailHandler() {
        return (AbstractMailHandler) mailHandlerContext.getInstance(getMailPlatformType());
    }

    /**
     * 获取邮件平台
     *
     * @return
     */
    private MailPlatformType getMailPlatformType() {
        String type = mailPlatformProperties.getType();
        MailPlatformType mailPlatformType = MailPlatformType.MOCK;
        if (StringUtils.hasLength(type)) {
            if (!Objects.isNull(MailPlatformType.mailPlatformTypeMap.get(type))) {
                mailPlatformType = MailPlatformType.mailPlatformTypeMap.get(type);
            }
        }
        return mailPlatformType;
    }

}
