package com.sinosoftgz.starter.mail.core.request;

import com.sinosoftgz.starter.mail.core.enums.MailBodyType;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.NotEmpty;
import java.io.Serializable;
import java.util.*;

/**
 * @author: Roney
 * @date: 2022-05-09 11:55
 * @description: java 邮件发送统一请求对象
 */
@Data
public class JavaMailClientReq implements Serializable {

    /**
     * 邮件平台类型
     */
    private String mailPlatformType;
    /**
     * 标题
     */
    private final String subject;
    /**
     * 收件人列表
     */
    @NotEmpty(message = "收件人列表不能为空")
    private final List<String> recipientTos;
    /**
     * 抄送人列表
     */
    private final List<String> recipientCc;
    /**
     * 密送人列表
     */
    private final List<String> recipientBcc;
    /**
     * 附件列表
     */
    private final List<FileData> attachments;
    /**
     * 图片列表
     */
    private final List<FileData> images;
    /**
     * 邮件类型
     */
    private final MailBodyType mailBodyType;
    /**
     * 邮件内容
     */
    private final String content;

    private JavaMailClientReq(JavaMailClientReqBuilder builder) {
        this.mailPlatformType = builder.mailPlatformType;
        this.subject = builder.subject;
        this.recipientTos = builder.recipientTos;
        this.recipientCc = builder.recipientCc;
        this.recipientBcc = builder.recipientBcc;
        this.attachments = builder.attachments;
        this.images = builder.images;
        this.mailBodyType = builder.mailBodyType;
        this.content = builder.content;
    }

    public static class JavaMailClientReqBuilder {

        private String mailPlatformType;
        private String subject;
        private List<String> recipientTos;
        private List<String> recipientCc;
        private List<String> recipientBcc;
        private List<FileData> attachments;
        private List<FileData> images;
        private MailBodyType mailBodyType;
        private String content;


        public JavaMailClientReqBuilder() {
            this.mailPlatformType = "";
            this.subject = "";
            this.recipientTos = new ArrayList<>(1);
            this.recipientCc = new ArrayList<>(0);
            this.recipientBcc = new ArrayList<>(0);
            this.attachments = new ArrayList<>(0);
            this.images = new ArrayList<>(0);
            this.mailBodyType = MailBodyType.Text;
            this.content = "";
        }

        /**
         * @param mailPlatformType
         * @return
         * @Desc 邮件平台类型
         */
        public JavaMailClientReqBuilder mailPlatformType(String mailPlatformType) {
            this.mailPlatformType = mailPlatformType;
            return this;
        }


        /**
         * @param subject
         * @return
         * @Desc 邮件主题
         */
        public JavaMailClientReqBuilder subject(String subject) {
            this.subject = subject;
            return this;
        }

        /**
         * @param recipientTos
         * @return
         * @Desc 收件人
         */
        public JavaMailClientReqBuilder recipientTos(List<String> recipientTos) {
            this.recipientTos = recipientTos;
            return this;
        }

        /**
         * @param recipientCc
         * @return
         * @Desc 抄送人
         */
        public JavaMailClientReqBuilder recipientCc(List<String> recipientCc) {
            this.recipientCc = recipientCc;
            return this;
        }

        /**
         * @param recipientBcc
         * @return
         * @Desc 密送人
         */
        public JavaMailClientReqBuilder recipientBcc(List<String> recipientBcc) {
            this.recipientBcc = recipientBcc;
            return this;
        }


        /**
         * @param mailFileReqs
         * @return
         * @Desc 附件
         */
        public JavaMailClientReqBuilder attachments(List<MailFileReq> mailFileReqs) {
            if (!CollectionUtils.isEmpty(mailFileReqs)) {
                this.attachments.addAll(MailFileReqToFileData(mailFileReqs));
            }
            return this;
        }

        /**
         * 将MailFileReq转成FileData
         *
         * @param mailFileReqs
         * @return
         */
        private List<FileData> MailFileReqToFileData(List<MailFileReq> mailFileReqs) {
            List<FileData> fileDatas = new ArrayList<>();
            for (MailFileReq mailFileReq : mailFileReqs) {
                if (mailFileReq != null) {
                    FileData data = new FileData(mailFileReq.getFileName(), mailFileReq.getData(), mailFileReq.getBase64Str());
                    fileDatas.add(data);
                }
            }
            return fileDatas;
        }


        /**
         * @param mailImageReqs
         * @return
         * @Desc 图片
         */
        public JavaMailClientReqBuilder images(List<MailImageReq> mailImageReqs) {
            if (!CollectionUtils.isEmpty(mailImageReqs)) {
                this.images.addAll(MailImageReqToFileData(mailImageReqs));
            }
            return this;
        }

        /**
         * 将MailImageReq转成FileData
         *
         * @param mailImageReqs
         * @return
         */
        private List<FileData> MailImageReqToFileData(List<MailImageReq> mailImageReqs) {
            List<FileData> fileDatas = new ArrayList<>();
            for (MailImageReq mailImageReq : mailImageReqs) {
                if (mailImageReq != null) {
                    FileData data = new FileData(mailImageReq.getFileName(), mailImageReq.getData(), mailImageReq.getBase64Str());
                    fileDatas.add(data);
                }
            }
            return fileDatas;
        }

        /**
         * @param mailBodyType
         * @return
         * @Desc 正文类型
         */
        public JavaMailClientReqBuilder mailBodyType(MailBodyType mailBodyType) {
            this.mailBodyType = mailBodyType;
            return this;
        }

        /**
         * @param content
         * @return
         * @Desc 邮件正文
         */
        public JavaMailClientReqBuilder content(String content) {
            this.content = content;
            return this;
        }

        /**
         * @return
         * @Desc 创建邮件
         */
        public JavaMailClientReq build() {
            return new JavaMailClientReq(this);
        }
    }


}


