package com.sinosoftgz.starter.mail.core.handler.biz;

import cn.hutool.core.codec.Base64;
import com.sinosoftgz.starter.mail.core.enums.MailBodyType;
import com.sinosoftgz.starter.mail.core.enums.MailPlatformType;
import com.sinosoftgz.starter.mail.core.handler.annotation.MailHandlerType;
import com.sinosoftgz.starter.mail.core.properties.MailPlatformProperties;
import com.sinosoftgz.starter.mail.core.request.FileData;
import com.sinosoftgz.starter.mail.core.request.JavaMailClientReq;
import com.sinosoftgz.starter.utils.date.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.mail.MailProperties;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.InputStreamSource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import javax.validation.Valid;
import java.util.List;


/**
 * @author: Roney
 * @date: 2022-05-09 14:44
 * @description: SpringBootMailHandler
 */

@Component
@MailHandlerType(MailPlatformType.SPRING_BOOT_MAIL)
@Slf4j
public class SpringBootMailHandler extends DefaultMailHandler {

    @Autowired
    MailProperties mailProperties;

    @Autowired
    MailPlatformProperties mailPlatformProperties;

    @Autowired
    JavaMailSender javaMailSender;


    @Override
    public void testStrategyPattern() {
        log.debug("SpringBootMailHandler");
    }

    @Override
    public boolean send(@Valid JavaMailClientReq javaMailClientReq) {
        checkJavaMailClient(javaMailClientReq);
        /**
         * todo： 增加重复拦截，10秒内，不允许向同一个用户发送相同的邮件，服务未连接redis，暂时不做防重
         */
        MimeMessage mimeMessage = javaMailSender.createMimeMessage();
        MimeMessageHelper helper = null;
        try {
            helper = new MimeMessageHelper(mimeMessage, true);
        } catch (MessagingException e) {
            log.error("设置带附件参数错误", e);
        }
        try {
            String[] recipientTos = javaMailClientReq.getRecipientTos().toArray((new String[0]));
            helper.setFrom(mailProperties.getUsername());
            helper.setTo(recipientTos);
            helper.setSubject(javaMailClientReq.getSubject());
            helper.setSentDate(DateUtils.now());
            updateText(javaMailClientReq, helper);
            /**
             * 设置抄送人
             */
            updateCc(helper, javaMailClientReq.getRecipientCc());
            /**
             * 设置密送人
             */
            updateCc(helper, javaMailClientReq.getRecipientBcc());
            /**
             * 设置附件
             */
            updateAttachment(javaMailClientReq, helper);
            /**
             * 设置图片
             */
            updateImage(javaMailClientReq, helper);
        } catch (MessagingException e) {
            log.error("发送邮件异常", e);
            return false;
        } catch (Exception e) {
            log.error("发送邮件异常", e);
            return false;
        }
        javaMailSender.send(mimeMessage);
        return true;
    }

    /**
     * 设置邮件类型
     *
     * @param javaMailClient
     * @param helper
     * @throws MessagingException
     */
    private void updateText(JavaMailClientReq javaMailClient, MimeMessageHelper helper) throws MessagingException {
        if (MailBodyType.HTML.equals(javaMailClient.getMailBodyType())) {
            /**
             * 设置true，代表发送html
             */
            helper.setText(javaMailClient.getContent(), true);
        } else {
            helper.setText(javaMailClient.getContent());
        }
    }

    /**
     * 设置抄送人信息
     *
     * @param helper
     * @param recipientCc
     * @throws MessagingException
     */
    private void updateCc(MimeMessageHelper helper, List<String> recipientCc) throws MessagingException {

        if (!CollectionUtils.isEmpty(recipientCc)) {
            helper.setCc(recipientCc.toArray((new String[0])));
        }
    }

    /**
     * 设置附件信息
     *
     * @param javaMailClient
     * @param helper
     */
    private void updateAttachment(JavaMailClientReq javaMailClient, MimeMessageHelper helper) {
        if (!CollectionUtils.isEmpty(javaMailClient.getAttachments())) {
            javaMailClient.getAttachments().stream().forEach(fileData -> {
                InputStreamSource source = null;
                if (StringUtils.hasLength(fileData.getBase64Str())) {
                    source = new ByteArrayResource(Base64.decode(fileData.getBase64Str()));
                } else {
                    source = new ByteArrayResource(fileData.getData());
                }

                try {
                    helper.addAttachment(fileData.getFileName(), source);
                } catch (MessagingException e) {
                    log.error("设置邮件附件发生异常", e);
                }
            });
        }
    }

    /**
     * 设置图片信息
     *
     * @param javaMailClient
     * @param helper
     */
    private void updateImage(JavaMailClientReq javaMailClient, MimeMessageHelper helper) {
        if (!CollectionUtils.isEmpty(javaMailClient.getImages())) {

            //请在正文设置img标签加载图片
            List<FileData> images = javaMailClient.getImages();
            for (int i = 0; i < images.size(); i++) {
                try {
                    FileData fileData = images.get(i);
                    InputStreamSource source = null;
                    if (StringUtils.hasLength(fileData.getBase64Str())) {
                        source = new ByteArrayResource(Base64.decode(fileData.getBase64Str()));
                    } else {
                        source = new ByteArrayResource(fileData.getData());
                    }
                    String contentId = mailPlatformProperties.getMailImageContentIdPrefix() + i;
                    helper.addInline(contentId, source, "image/jpeg");
                } catch (MessagingException e) {
                    log.error("设置邮件图片发生异常", e);
                }
            }
        }
    }

}
