package com.sinosoftgz.starter.activemq.config;

import com.sinosoftgz.starter.activemq.utils.ActiveMqUtils;
import org.apache.activemq.ActiveMQConnectionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.jms.DefaultJmsListenerContainerFactoryConfigurer;
import org.springframework.boot.autoconfigure.jms.JmsProperties;
import org.springframework.boot.autoconfigure.jms.activemq.ActiveMQProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.jms.annotation.EnableJms;
import org.springframework.jms.config.DefaultJmsListenerContainerFactory;
import org.springframework.jms.config.JmsListenerContainerFactory;

import org.springframework.jms.core.JmsTemplate;
import org.springframework.jms.support.converter.MappingJackson2MessageConverter;
import org.springframework.jms.support.converter.MessageConverter;
import org.springframework.jms.support.converter.MessageType;

import javax.annotation.Resource;
import javax.jms.ConnectionFactory;

/**
 * Created by Roney on 2019/8/19.
 *
 * @author Roney
 * @date 2019-08-19 15:56
 */
@Configuration
@EnableConfigurationProperties({ActiveMQProperties.class, JmsProperties.class})
@EnableJms
public class ActiveMqConfiguration {
    final public static String QUEUE_LISTENER_FACTORY = "queueListenerFactory";
    final public static String TOPIC_LISTENER_FACTORY = "topicListenerFactory";

    @Autowired
    ActiveMQProperties activeMQProperties;

    @Bean
    public ConnectionFactory connectionFactory() {
        return new ActiveMQConnectionFactory(activeMQProperties.getUser(), activeMQProperties.getPassword(), activeMQProperties.getBrokerUrl());
    }

    @Bean
    public JmsListenerContainerFactory<?> queueListenerFactory(ConnectionFactory connectionFactory,
                                                               DefaultJmsListenerContainerFactoryConfigurer configurer) {
        DefaultJmsListenerContainerFactory factory = new DefaultJmsListenerContainerFactory();
        // This provides all boot's default to this factory, including the message converter
        configurer.configure(factory, connectionFactory);
        return factory;
    }

    @Bean
    public JmsListenerContainerFactory<?> topicListenerFactory(ConnectionFactory connectionFactory,
                                                               DefaultJmsListenerContainerFactoryConfigurer configurer) {
        DefaultJmsListenerContainerFactory factory = new DefaultJmsListenerContainerFactory();
        // This provides all boot's default to this factory, including the message converter
        configurer.configure(factory, connectionFactory);
        // You could still override some of Boot's default if necessary.
        /**
         * 为True时，使用topic广播模式，所有消费者都可以收到消息
         */
        factory.setPubSubDomain(true);
        return factory;
    }


    @Bean(name = "topicJmsTemplate")
    public JmsTemplate topicJmsTemplate(ConnectionFactory connectionFactory) {
        JmsTemplate topicJmsTemplate = new JmsTemplate(connectionFactory);
        topicJmsTemplate.setDeliveryPersistent(true);
        topicJmsTemplate.setPubSubDomain(true);
        MessageConverter messageConverter = jacksonJmsMessageConverter();
        topicJmsTemplate.setMessageConverter(messageConverter);
        return topicJmsTemplate;
    }

    @Primary
    @Bean(name = "queueJmsTemplate")
    public JmsTemplate queueJmsTemplate(ConnectionFactory connectionFactory) {
        JmsTemplate jmsTemplate = new JmsTemplate(connectionFactory);
        MessageConverter messageConverter = jacksonJmsMessageConverter();
        jmsTemplate.setMessageConverter(messageConverter);
        return jmsTemplate;
    }

    /**
     * Serialize message content to json using TextMessage
     *
     * @return
     */
    @Bean("jacksonJmsMessageConverter")
    public MessageConverter jacksonJmsMessageConverter() {
        MappingJackson2MessageConverter converter = new MappingJackson2MessageConverter();
        converter.setTargetType(MessageType.TEXT);
        converter.setTypeIdPropertyName("_type");
        return converter;
    }

    @Bean
    ActiveMqUtils activeMqUtils(ConnectionFactory connectionFactory) {
        return new ActiveMqUtils(queueJmsTemplate(connectionFactory), topicJmsTemplate(connectionFactory));
    }


}
