package com.sinosoftgz.starter.rabbitmq.utils;

import cn.hutool.json.JSONUtil;
import com.rabbitmq.client.Channel;
import com.sinosoftgz.starter.rabbitmq.constant.MessageDelayLevel;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.*;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.amqp.RabbitProperties;
import org.springframework.util.ObjectUtils;

import java.io.IOException;

/**
 * Created by Roney on 2019/8/20.
 *
 * @author Roney
 * @date 2019-08-20 11:54
 */
@Slf4j
public class RabbitMqUtils {

    private final RabbitAdmin rabbitAdmin;

    private final RabbitTemplate rabbitTemplate;

    @Autowired
    RabbitProperties rabbitProperties;

    public RabbitMqUtils(RabbitAdmin rabbitAdmin, RabbitTemplate rabbitTemplate) {
        this.rabbitAdmin = rabbitAdmin;
        this.rabbitTemplate = rabbitTemplate;
    }

    /**
     * 创建Exchange
     *
     * @param exchange
     */
    public void addExchange(AbstractExchange exchange) {
        rabbitAdmin.declareExchange(exchange);
    }

    /**
     * 删除一个Exchange
     *
     * @param exchangeName
     */
    public boolean deleteExchange(String exchangeName) {
        return rabbitAdmin.deleteExchange(exchangeName);
    }


    /**
     * Declare a queue whose name is automatically named. It is created with exclusive = true, autoDelete=true, and
     * durable = false.
     *
     * @return Queue
     */
    public Queue addQueue() {
        return rabbitAdmin.declareQueue();
    }

    /**
     * 创建一个指定的Queue
     *
     * @param queue
     * @return queueName
     */
    public String addQueue(Queue queue) {
        return rabbitAdmin.declareQueue(queue);
    }

    /**
     * Delete a queue.
     *
     * @param queueName the name of the queue.
     * @param unused    true if the queue should be deleted only if not in use.
     * @param empty     true if the queue should be deleted only if empty.
     */
    public void deleteQueue(String queueName, boolean unused, boolean empty) {
        rabbitAdmin.deleteQueue(queueName, unused, empty);
    }

    /**
     * 删除一个queue
     *
     * @param queueName
     * @return true if the queue existed and was deleted.
     */
    public boolean deleteQueue(String queueName) {
        return rabbitAdmin.deleteQueue(queueName);
    }

    /**
     * 绑定一个队列到一个匹配型交换器使用一个routingKey
     *
     * @param queue
     * @param exchange
     * @param routingKey
     */
    public void addBinding(Queue queue, TopicExchange exchange, String routingKey) {
        Binding binding = BindingBuilder.bind(queue).to(exchange).with(routingKey);
        rabbitAdmin.declareBinding(binding);
    }

    /**
     * 绑定一个Exchange到一个匹配型Exchange 使用一个routingKey
     *
     * @param exchange
     * @param topicExchange
     * @param routingKey
     */
    public void addBinding(Exchange exchange, TopicExchange topicExchange, String routingKey) {
        Binding binding = BindingBuilder.bind(exchange).to(topicExchange).with(routingKey);
        rabbitAdmin.declareBinding(binding);
    }

    /**
     * 去掉一个binding
     *
     * @param binding
     */
    public void removeBinding(Binding binding) {
        rabbitAdmin.removeBinding(binding);
    }

    /**
     * 通知 MQ 消息已被成功消费,可以ACK了
     *
     * @param message
     * @param channel
     */
    public void basicAck(Message message, Channel channel) {
        this.checkAcknowledgeMode();
        final long deliveryTag = message.getMessageProperties().getDeliveryTag();
        // 通知 MQ 消息已被成功消费,可以ACK了
        try {
            channel.basicAck(deliveryTag, false);
        } catch (IOException e) {
            log.error("通知 MQ 消息已被成功消费,可以ACK了异常信息", e);
        }
    }

    /**
     * 处理失败,重新压入MQ
     *
     * @param channel
     */
    private void basicRecover(Channel channel) {
        try {
            channel.basicRecover();
        } catch (IOException e) {
            log.error("处理失败,重新压入MQ发生异常，异常信息", e);
        }
    }

    /**
     * 通知 MQ 消息已被成功消费,可以ACK了同时将处理失败,重新压入MQ
     * 手动ack是必要步骤，是否重新压入MQ由业务方决定
     *
     * @param message
     * @param channel
     */
    public void basicAckAndRecover(Message message, Channel channel) {
        this.checkAcknowledgeMode();
        final long deliveryTag = message.getMessageProperties().getDeliveryTag();
        // 通知 MQ 消息已被成功消费,可以ACK了
        try {
            channel.basicAck(deliveryTag, false);
        } catch (IOException e) {
            log.error("通知 MQ 消息已被成功消费,可以ACK发生异常，异常信息", e);
            this.basicRecover(channel);
        }
    }


    /**
     * 检查消息确认模式
     */
    private void checkAcknowledgeMode() {
        AcknowledgeMode acknowledgeMode = rabbitProperties.getListener().getSimple().getAcknowledgeMode();
        if (!ObjectUtils.isEmpty(acknowledgeMode)) {
            if (AcknowledgeMode.MANUAL != acknowledgeMode) {
                log.error("配置的消息确认模式不是手动模式，不允许手动确认，配置的消息确认模式为：{} ", acknowledgeMode);
                throw new RuntimeException("配置的消息确认模式不是手动模式，不允许手动确认。");
            } else {
                if (AcknowledgeMode.AUTO == acknowledgeMode) {
                    log.info("配置的消息确认为：{} 。请确保消费方没有手动确认消息，否则将抛出如下异常：Caused by: org.springframework.amqp.AmqpException: PublisherCallbackChannel is closed", AcknowledgeMode.AUTO);
                }
            }
        } else {
            log.info("未配置的消息确认模式，默认为：{} 。请确保消费方没有手动确认消息，否则将抛出如下异常：Caused by: org.springframework.amqp.AmqpException: PublisherCallbackChannel is closed", AcknowledgeMode.AUTO);
        }
    }

    /**
     * 直接模式发送
     *
     * @param queueName
     * @param message
     */
    public void directQueue(String queueName, Object message) {
        log.debug("directQueue queueName:{},message:{}", queueName, message);
        rabbitTemplate.convertAndSend(queueName, message);
    }

    /**
     * 分列模式队列
     *
     * @param queueName
     * @param routingKey
     * @param message
     */
    public void fanoutExchange(String queueName, String routingKey, Object message) {
        log.debug("fanoutExchange queueName:{},routingKey:{},message:{}", queueName, routingKey, message);
        rabbitTemplate.convertAndSend(queueName, routingKey, message);
    }

    /**
     * 主题模式队列
     *
     * @param queueName
     * @param routingKey
     * @param message
     */
    public void topicExchange(String queueName, String routingKey, Object message) {
        log.debug("topicExchange queueName:{},routingKey:{},message:{}", queueName, routingKey, message);
        rabbitTemplate.convertAndSend(queueName, routingKey, message);
    }

    /**
     * 发送延迟队列
     *
     * @param exchange
     * @param queueName
     * @param message
     * @param messageDelayLevel
     */
    public void delayQueue(String exchange, String queueName, Object message, MessageDelayLevel messageDelayLevel) {
        log.debug("delayQueue exchange:{},queueName:{},message:{}", exchange, queueName, message);
        rabbitTemplate.convertAndSend(exchange, queueName, message, messagePostProcessor -> {
            messagePostProcessor.getMessageProperties().setHeader("x-delay", messageDelayLevel.getValue());
            return messagePostProcessor;
        });
    }

}
