package com.sinosoftgz.starter.rocketmq.utils;

import com.sinosoftgz.starter.rocketmq.constant.MessageDelayLevel;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.spring.core.RocketMQTemplate;
import org.springframework.messaging.Message;
import org.springframework.util.Assert;

/**
 * Created by Roney on 2019/11/4.
 *
 * @author Roney
 * @date 2019-11-04 09:59
 */
@Slf4j
public class RocketMqUtils {

    RocketMQTemplate rocketMQTemplate;

    public RocketMqUtils(RocketMQTemplate rocketMQTemplate) {
        this.rocketMQTemplate = rocketMQTemplate;
    }

    /**
     * 发送topic
     *
     * @param topicName
     * @param message
     */
    public void sendTopic(String topicName, Object message) {
        Assert.notNull(topicName, "topicName is null ");
        Assert.notNull(message, "message is null ");
        log.debug("sendTopic topicName:{},message:{}", topicName, message);
        rocketMQTemplate.convertAndSend(topicName, message);
    }

    /**
     * 同步发送队列
     * 可靠同步发送
     * 主要运用在比较重要一点消息传递/通知等业务
     *
     * @param queueName
     * @param message
     */
    public SendResult syncSendQueue(String queueName, Object message) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        log.debug("syncSendQueue queueName:{},message:{}", queueName, message);
        SendResult result = rocketMQTemplate.syncSend(queueName, message);
        log.debug("syncSendQueue result:{}", result);
        return result;
    }

    /**
     * 异步发送队列（不需要异步处理返回结果）
     * 可靠异步发送
     * 通常用于对发送消息响应时间要求更高/更快的场景
     *
     * @param queueName
     * @param message
     */
    public void asyncSendQueue(String queueName, Object message) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        log.debug("asyncSendQueue queueName:{},message:{}", queueName, message);
        rocketMQTemplate.asyncSend(queueName, message, new SendCallback() {
            // 实现消息发送成功的后续处理
            @Override
            public void onSuccess(SendResult var1) {
                log.debug("async onSuccess SendResult={}", var1);
            }

            // 实现消息发送失败的后续处理
            @Override
            public void onException(Throwable var1) {
                log.debug("async onException Throwable={}", var1);
            }
        });

    }

    /**
     * 异步发送队列（需要异步处理返回结果）
     * 可靠异步发送
     * 通常用于对发送消息响应时间要求更高/更快的场景
     *
     * @param queueName
     * @param message
     * @param sendCallback
     */
    public void asyncSendQueue(String queueName, Object message, SendCallback sendCallback) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        log.debug("asyncSendQueue queueName:{},message:{},sendCallback:{}", queueName, message, sendCallback);
        rocketMQTemplate.asyncSend(queueName, message, sendCallback);
    }

    /**
     * 单向发送
     * 适用于某些耗时非常短，但对可靠性要求并不高的场景，例如日志收集。
     * 只发送消息，不等待服务器响应，只发送请求不等待应答。此方式发送消息的过程耗时非常短，一般在微秒级别。
     *
     * @param queueName
     * @param message
     */
    public void sendOneWay(String queueName, Object message) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        log.debug("sendOneway queueName:{},message:{}", queueName, message);
        rocketMQTemplate.sendOneWay(queueName, message);
    }


    /**
     * 异步发送延迟队列（不需要异步处理返回结果）
     * 可靠异步发送
     * 通常用于对发送消息响应时间要求更高/更快的场景
     * "1s 5s 10s 30s 1m 2m 3m 4m 5m 6m 7m 8m 9m 10m 20m 30m 1h 2h"
     *
     * @param queueName
     * @param message
     * @param messageDelayLevel
     */
    public void asyncDelayQueue(String queueName, Message<?> message, MessageDelayLevel messageDelayLevel) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        Assert.notNull(messageDelayLevel, "messageDelayLevel is null ");
        log.debug("asyncDelayQueue queueName:{},message:{},delayLevel:{}", queueName, message, messageDelayLevel);
        rocketMQTemplate.asyncSend(queueName, message, new SendCallback() {
            // 实现消息发送成功的后续处理
            @Override
            public void onSuccess(SendResult var1) {
                log.debug("asyncDelayQueue onSuccess SendResult={}", var1);
            }

            // 实现消息发送失败的后续处理
            @Override
            public void onException(Throwable var1) {
                log.debug("asyncDelayQueue onException Throwable={}", var1);
            }
        }, rocketMQTemplate.getProducer().getSendMsgTimeout(), messageDelayLevel.getValue());

    }

    /***
     * 异步发送延迟队列（需要异步处理返回结果）
     * 可靠异步发送
     * 通常用于对发送消息响应时间要求更高/更快的场景
     * "1s 5s 10s 30s 1m 2m 3m 4m 5m 6m 7m 8m 9m 10m 20m 30m 1h 2h"
     * @param queueName
     * @param message
     * @param messageDelayLevel
     * @param sendCallback
     */
    public void asyncDelayQueue(String queueName, Message<?> message, MessageDelayLevel messageDelayLevel, SendCallback sendCallback) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        Assert.notNull(messageDelayLevel, "messageDelayLevel is null ");
        log.debug("asyncDelayQueue queueName:{},message:{},delayLevel:{},sendCallback:{}", queueName, message, messageDelayLevel, sendCallback);
        rocketMQTemplate.asyncSend(queueName, message, sendCallback, rocketMQTemplate.getProducer().getSendMsgTimeout(), messageDelayLevel.getValue());

    }

    /**
     * 同步发送延迟队列
     * 可靠同步发送
     * 主要运用在比较重要一点消息传递/通知等业务
     *
     * @param queueName
     * @param message
     * @param messageDelayLevel
     * @return
     */
    public SendResult syncDelayQueue(String queueName, Message<?> message, MessageDelayLevel messageDelayLevel) {
        Assert.notNull(queueName, "queueName is null ");
        Assert.notNull(message, "message is null ");
        Assert.notNull(messageDelayLevel, "messageDelayLevel is null ");
        log.debug("syncDelayQueue queueName:{},message:{},delayLevel:{}", queueName, message, messageDelayLevel);
        SendResult result = rocketMQTemplate.syncSend(queueName, message, rocketMQTemplate.getProducer().getSendMsgTimeout(), messageDelayLevel.getValue());
        log.debug("syncDelayQueue result:{}", result);
        return result;
    }
}
