package com.sinosoftgz.starter.minio.oss.config;

import com.sinosoftgz.starter.minio.oss.properties.MinioProperties;
import io.minio.BucketExistsArgs;
import io.minio.MakeBucketArgs;
import io.minio.MinioClient;
import io.minio.errors.*;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

/**
 * @author: Roney
 * @date: 2021-10-15 11:48
 * @description: MinioConfiguration
 */
@Configuration
@Import({

})
@EnableConfigurationProperties({MinioProperties.class})
@ConditionalOnProperty(prefix = MinioProperties.OSS_MINIO_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@ConditionalOnClass(MinioClient.class)
@Slf4j
public class MinioConfiguration {

    @Autowired
    private MinioProperties minioProperties;

    @Bean
    @ConditionalOnMissingBean
    MinioClient minioClient(OkHttpClient okHttpClient) {
        log.info("init the minio client user args :{}", minioProperties);
        MinioClient client = MinioClient.builder()
                .httpClient(okHttpClient)
                .credentials(minioProperties.getAccessKey(), minioProperties.getSecretKey())
                .endpoint(minioProperties.getUrl()).build();
        client.setTimeout(minioProperties.getConnectTimeout().toMinutes(),
                minioProperties.getWriteTimeout().toMinutes(),
                minioProperties.getReadTimeout().toMinutes());

        return client;
    }

    @Bean
    @ConditionalOnMissingBean(OkHttpClient.class)
    OkHttpClient httpClient() {
        log.info("minio init by the default http client");
        return new OkHttpClient.Builder().build();
    }

    @PostConstruct
    void init() throws IOException, InvalidKeyException, InvalidResponseException, InsufficientDataException, NoSuchAlgorithmException, ServerException, InternalException, XmlParserException, ErrorResponseException {

        if (minioProperties.getBucket() == null || "".equals(minioProperties.getBucket().trim())) {
            return;
        }
        if (minioProperties.isCreateBucket()) {
            MinioClient minioClient = minioClient(httpClient());
            boolean bucketExists = minioClient.bucketExists(BucketExistsArgs.builder()
                    .bucket(minioProperties.getBucket())
                    .build());
            if (!bucketExists) {
                log.info("create minio bucket : {} ", minioProperties.getBucket());
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(minioProperties.getBucket()).build());
            }
        }
    }
}
