package com.sinosoft.starter.dubbo.cluster.loadbalance;

import com.sinosoft.starter.dubbo.constant.DevDubboConstants;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.RpcContext;
import org.apache.dubbo.rpc.cluster.loadbalance.RandomLoadBalance;
import org.springframework.util.CollectionUtils;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Created by Roney on 2021/5/10 16:00.
 */
@Slf4j
public class DevLoadBalance extends RandomLoadBalance {

    public static final String NAME = "dev";

    public static final String METHOD_CALL_LOCAL_MAP_KEY = "%s.%s.%s";


    @Override
    protected <T> Invoker<T> doSelect(List<Invoker<T>> invokers, URL url, Invocation invocation) {
        ConcurrentMap<String, Invoker> invokerLocalIpConcurrentMap = new ConcurrentHashMap<String, Invoker>();
        ConcurrentMap<String, Invoker> invokerMethodNameConcurrentMap = new ConcurrentHashMap<String, Invoker>();
        //log.info("loadBalance select start");
        //获取起始ip信息
        Object fromIp = RpcContext.getContext().getAttachment(DevDubboConstants.START_LOCAL_IP);
        if (fromIp == null) {
            try {
                //如果没有获取到起始ip，则优先选择自己的ip
                fromIp = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException e) {
                e.printStackTrace();
            }
        }
        //log.info("loadBalance select ip is:" + fromIp);
        /**
         *  选择起始ip的服务，或者自己的服务进行调用
         *  优先选择自己本地的ip和配置的sit环境的ip进行调用
         */
        String localIp = fromIp.toString();
        List<Invoker<T>> invokerList = getInvokerAddrList(invokers, localIp, invocation, invokerLocalIpConcurrentMap, invokerMethodNameConcurrentMap);
        //log.info("local ip : {}", localIp);
        if (StringUtils.isNoneEmpty(localIp)) {
            if (invokerLocalIpConcurrentMap.containsKey(localIp)) {
                Invoker invoker = invokerLocalIpConcurrentMap.get(localIp);
                String invokerKey = generatorInvokerKey(invocation, invoker);
                log.info("invokerKey:{}", invokerKey);
                if (invoker.isAvailable() && invokerMethodNameConcurrentMap.containsKey(invokerKey)) {
                    // log.info("invokerLocalIpConcurrentMap:{}",invokerLocalIpConcurrentMap);
                    // log.info("invokerMethodNameConcurrentMap:{}",invokerMethodNameConcurrentMap);
                    return callInvoker(invoker);
                }
            }
        }

        if (!CollectionUtils.isEmpty(invokerList)) {
            invokerList.parallelStream().forEach(invoker -> {
                log.debug("获取到测试环境相关的服务列表：{},状态：{}", invoker.getUrl(), invoker.isAvailable());
            });
            log.debug("select call RandomLoadBalance");
            return super.doSelect(invokerList, url, invocation);
        }

        log.debug("select super loadBalance");
        //如果起始ip没有启动服务，自己也没有启动，则使用默认规则
        return super.doSelect(invokers, url, invocation);
    }


    private <T> List<Invoker<T>> getInvokerAddrList(List<Invoker<T>> invokers, String localIp, Invocation invocation, ConcurrentMap<String, Invoker> invokerLocalIpConcurrentMap, ConcurrentMap<String, Invoker> invokerMethodNameConcurrentMap) {
        List<Invoker<T>> invokerList = new ArrayList<>();
        for (Invoker<T> invoker : invokers) {
            /**
             * 将测试环境IP加入新list
             * 将本地IP和SITip 加入到invokerConcurrentMap
             */
            if (invoker.getUrl().getIp().equals(localIp) && invoker.isAvailable()) {

                String invokerKey = generatorInvokerKey(invocation, invoker);
                invokerMethodNameConcurrentMap.put(invokerKey, invoker);
                invokerLocalIpConcurrentMap.put(localIp, invoker);
            }
            if (invoker.getUrl().getIp().contains(DevDubboConstants.START_SIT_IP) && invoker.isAvailable()) {
                invokerList.add(invoker);
            }
        }
        return invokerList;
    }

    private <T> String generatorInvokerKey(Invocation invocation, Invoker<T> invoker) {
        String remoteApplicationKey = invoker.getUrl().getParameter(CommonConstants.REMOTE_APPLICATION_KEY);
        String serviceKey = invoker.getUrl().getServiceKey();
        String methodName = invocation.getMethodName();
        String invokerKey = String.format(METHOD_CALL_LOCAL_MAP_KEY, remoteApplicationKey, serviceKey, methodName);
        return invokerKey;
    }


    private <T> Invoker<T> callInvoker(Invoker<T> invoker) {
        log.info("call local invoker url :{}", invoker.getUrl());
        return invoker;
    }
}
