package com.sinosoftgz.starter.jwt.biz.impl;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.sinosoftgz.global.common.constants.CommonConstants;
import com.sinosoftgz.starter.jwt.biz.JwtAuthBiz;
import com.sinosoftgz.starter.jwt.model.JwtPrincipal;
import com.sinosoftgz.starter.jwt.model.JwtToken;
import com.sinosoftgz.starter.jwt.model.JwtUserInfo;
import com.sinosoftgz.starter.jwt.properties.JwtProperties;
import com.sinosoftgz.starter.jwt.utils.JwtUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * jwt mock实现
 *
 * @author: Roney
 * @date: 2022-06-07 14:11
 * @description: jwt mock实现
 */
@Slf4j
@Component("mockJwtAuthBizImpl")
@ConditionalOnProperty(prefix = JwtProperties.JWT_PREFIX, name = "enableMock", havingValue = "true", matchIfMissing = false)
public class MockJwtAuthBizImpl extends AbstractJwtAuthBizImpl {

    @Resource
    JwtUtils jwtUtils;

    @Override
    public JwtPrincipal createToken(JwtUserInfo jwtUserInfo) {
        checkJwtUserInfo(jwtUserInfo);
        String token = jwtUtils.sign(jwtUserInfo.getAccount(), jwtUserInfo.getSecret());
        long expiresAt = jwtUtils.getExpiresAt(token);
        String account = jwtUtils.getAccount(token);
        return new JwtPrincipal(account, expiresAt, token);
    }

    @Override
    public boolean verify(JwtToken jwtToken) {
        checkJwtToken(jwtToken);
        return jwtUtils.verify(jwtToken.getToken(), jwtToken.getSecret());
    }


    @Override
    public JwtPrincipal renewalToken(JwtToken jwtToken) {
        checkJwtToken(jwtToken);
        DecodedJWT decodedJWT = jwtUtils.renewalToken(jwtToken.getToken(), jwtToken.getSecret());
        if (!Objects.isNull(decodedJWT)) {
            String token = decodedJWT.getToken();
            long expiresAt = decodedJWT.getExpiresAt().getTime();
            String account = jwtUtils.getAccount(token);
            return new JwtPrincipal(account, expiresAt, token);
        }
        return new JwtPrincipal();
    }

    @Override
    public boolean destroyToken(JwtToken jwtToken) {
        checkJwtToken(jwtToken);
        return jwtUtils.destroyToken(jwtToken.getToken(), jwtToken.getSecret());
    }

    @Override
    public String getAccount(String token) {
        checkToken(token);
        return jwtUtils.getAccount(token);
    }


    @Override
    public boolean expire(String token) {
        checkToken(token);
        return jwtUtils.expire(token);
    }
}
