package com.sinosoftgz.starter.jwt.utils;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.sinosoftgz.global.common.constants.CommonConstants;
import com.sinosoftgz.starter.jwt.exception.JwtAccountException;
import com.sinosoftgz.starter.jwt.exception.JwtVerificationFailedException;
import com.sinosoftgz.starter.jwt.properties.JwtProperties;
import com.sinosoftgz.starter.utils.lang.Lang;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Date;

/**
 * Created by Roney on 2021/1/4 20:29.
 */
@Slf4j
@Component
public class JwtUtils {


    private final JwtProperties jwtProperties;

    public JwtUtils(final JwtProperties jwtProperties) {
        this.jwtProperties = jwtProperties;
    }

    /**
     * 销毁token
     *
     * @param token
     * @param secret
     * @return
     */
    public boolean destroyToken(String token, String secret) {
        if (!this.verify(token, secret)) {
            return true;
        }
        return false;
    }

    /**
     * 续签token
     */
    public DecodedJWT renewalToken(String token, String secret) {
        /**
         * 如果token已经过期，无法续签
         */
        if (Lang.isEmpty(this.verifySafetyPeriod(token, secret))) {
            throw new JwtVerificationFailedException("token已经过期，无法续签");
        }
        /**
         * 生成新的token
         */
        String newToken = this.sign(this.getAccount(token), secret);
        DecodedJWT jwt = this.checkToken(newToken, secret);
        return jwt;
    }


    /**
     * 校验token 增加安全期
     *
     * @param token
     * @param secret
     * @return
     */
    public DecodedJWT verifySafetyPeriod(String token, String secret) {
        try {
            Algorithm algorithm = Algorithm.HMAC256(secret);
            JWTVerifier verifier = JWT.require(algorithm)
                    //在token失效前提供一个安全窗口期，使前端有机会刷新token
                    //注意这里的单位为秒
                    .acceptExpiresAt(this.jwtProperties.getMaxIdleMinute() * 60)
                    .build();
            DecodedJWT jwt = verifier.verify(token);
            return jwt;
        } catch (Exception exception) {
            log.error("verify token error", exception);
            throw new JwtVerificationFailedException("verify token error");
        }
    }

    /**
     * 校验token是否正确
     *
     * @param token
     * @param secret
     * @return
     */
    public boolean verify(String token, String secret) {
        DecodedJWT jwt = this.checkToken(token, secret);
        if (!Lang.isEmpty(jwt)) {
            return true;
        }
        return false;
    }


    /**
     * 校验token信息
     *
     * @param token
     * @param secret
     * @return
     */
    public DecodedJWT checkToken(String token, String secret) {
        try {
            //根据密码生成JWT效验器
            Algorithm algorithm = Algorithm.HMAC256(secret);
            JWTVerifier verifier = JWT.require(algorithm)
                    .withClaim(this.jwtProperties.getAccountAlias(), this.getAccount(token))
                    .build();
            //效验TOKEN
            DecodedJWT jwt = verifier.verify(token);
            return jwt;
        } catch (Exception exception) {
            log.error("verify token error", exception);
            throw new JwtVerificationFailedException("校验token异常");
        }
    }

    /**
     * 获得token中的信息无需secret解密也能获得
     *
     * @return token中包含的用户名
     */
    public String getAccount(String token) {
        Preconditions.checkNotNull(token, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "token"));
        DecodedJWT jwt = decode(token);
        if (jwt != null) {
            return jwt.getClaim(this.jwtProperties.getAccountAlias()).asString();
        } else {
            throw new JwtAccountException("获取账号异常");
        }

    }

    /**
     * 生成签名
     *
     * @param account            用户名
     * @param secret             用于加密的key
     * @param expireAfterMinutes 指定token在多少分钟后过期
     * @return 加密的token
     */
    public String sign(String account, String secret, long expireAfterMinutes) {
        Preconditions.checkNotNull(account, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "account"));
        Preconditions.checkNotNull(secret, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "secret"));
        if (Lang.isEmpty(expireAfterMinutes)) {
            expireAfterMinutes = jwtProperties.getMaxAliveMinute();
        }
        Date expireAfter = new Date(System.currentTimeMillis() + expireAfterMinutes * 60 * 1000);
        Algorithm algorithm = Algorithm.HMAC256(secret);
        // 附带username信息
        return JWT.create()
                .withClaim(jwtProperties.getAccountAlias(), account)
                .withExpiresAt(expireAfter)
                .sign(algorithm);
    }

    /**
     * 生成签名
     *
     * @param account 用户名
     * @param secret  用于加密的key
     * @return 加密的token
     */
    public String sign(String account, String secret) {
        Preconditions.checkNotNull(account, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "account"));
        Preconditions.checkNotNull(secret, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "secret"));
        return this.sign(account, secret, jwtProperties.getMaxAliveMinute());
    }

    /**
     * 判断token是否过期
     *
     * @param token
     * @return
     */
    public boolean expire(String token) {
        Preconditions.checkNotNull(token, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "token"));
        DecodedJWT jwt = decode(token);
        if (jwt != null) {
            if (jwt.getExpiresAt().getTime() > System.currentTimeMillis()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 获取token有效期时长
     *
     * @param token
     * @return 返回负数或者0都是已经失效
     */
    public long getExpiresAt(String token) {
        Preconditions.checkNotNull(token, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "token"));
        DecodedJWT jwt = decode(token);
        if (jwt != null) {
            return jwt.getExpiresAt().getTime();
        }
        return -1;
    }

    /**
     * 根据token解密jwt信息
     *
     * @param token
     * @return
     */
    private DecodedJWT decode(String token) {
        Preconditions.checkNotNull(token, String.format(CommonConstants.MessageString.ASSERT_NOT_NULL_PROPERTY_NAME_MESSAGE, "token"));
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt;
        } catch (Exception e) {
            log.error("decode token error", e);
            throw new JWTDecodeException("decode token error");
        }

    }
}
