package com.sinosoftgz.starter.shiro.jwt.config;

import com.alibaba.fastjson.JSON;
import com.sinosoftgz.starter.jwt.properties.JwtProperties;
import com.sinosoftgz.starter.jwt.utils.JwtUtils;
import com.sinosoftgz.starter.shiro.jwt.filter.ShiroJwtAccessControlFilter;
import com.sinosoftgz.starter.shiro.jwt.interceptor.TokenRefreshInterceptor;
import com.sinosoftgz.starter.shiro.jwt.biz.UserAuthBiz;
import com.sinosoftgz.starter.shiro.jwt.support.ShiroJwtRealm;
import com.sinosoftgz.starter.shiro.properties.ShiroProperties;
import com.sinosoftgz.starter.shiro.support.CustomDefaultSubjectFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.mgt.*;
import org.apache.shiro.mgt.SecurityManager;
import org.apache.shiro.session.mgt.DefaultSessionManager;
import org.apache.shiro.session.mgt.SessionManager;
import org.apache.shiro.spring.security.interceptor.AuthorizationAttributeSourceAdvisor;
import org.apache.shiro.spring.web.ShiroFilterFactoryBean;
import org.apache.shiro.web.mgt.DefaultWebSecurityManager;
import org.springframework.aop.framework.autoproxy.DefaultAdvisorAutoProxyCreator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;
import org.springframework.web.filter.DelegatingFilterProxy;
import org.springframework.web.servlet.config.annotation.*;

import javax.servlet.Filter;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;


/**
 * Created by Roney on 2021/1/5 16:15.
 */
@Configuration
@EnableConfigurationProperties({ShiroProperties.class, JwtProperties.class})
@Slf4j
@ConditionalOnProperty(prefix = JwtProperties.JWT_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
public class ShiroJwtConfiguration {

    private static final String DEFAULT_URL = "/**";
    private static final String ANON = "anon";
    private static final String SHIRO_JWT_AUTHC = "shiroJwtAuthc";
    private static final String SHIRO_JWT_FILTER_FACTORY_NAME = "shiroJwtFilterFactoryBean";

    private JwtProperties jwtProperties;

    private ShiroProperties shiroProperties;

    public ShiroJwtConfiguration(JwtProperties jwtProperties, ShiroProperties shiroProperties) {
        this.jwtProperties = jwtProperties;
        this.shiroProperties = shiroProperties;
    }

    /****
     * 注入无状态的realm
     *
     * @return
     */
    @Bean
    public ShiroJwtRealm shiroJwtRealm() {
        ShiroJwtRealm realm = new ShiroJwtRealm();
        //设置禁用缓存（因为缓存不适用无状态）
        realm.setCachingEnabled(false);
        return realm;
    }

    /**
     * 自定义的无状态（不创建session）Subject工厂
     *
     * @return
     */
    @Bean
    public CustomDefaultSubjectFactory subjectFactory() {
        return new CustomDefaultSubjectFactory(shiroProperties);
    }

    /**
     * sessionManager通过sessionValidationSchedulerEnabled禁用掉会话调度器，
     * 因为我们禁用掉了会话，所以没必要再定期过期会话了。
     *
     * @return
     */
    @Bean
    public SessionManager sessionManager() {
        DefaultSessionManager sessionManager = new DefaultSessionManager();
        //关闭Session验证（无状态认证不需要session认证）
        sessionManager.setSessionValidationSchedulerEnabled(false);
        return sessionManager;
    }

    /**
     * 注入SessionStorageEvaluator,关闭Session存储
     */
    @Bean
    public SessionStorageEvaluator sessionStorageEvaluator() {
        DefaultSessionStorageEvaluator defaultSessionStorageEvaluator = new DefaultSessionStorageEvaluator();
        defaultSessionStorageEvaluator.setSessionStorageEnabled(false);
        return defaultSessionStorageEvaluator;
    }

    /***
     * 安全管理配置
     *
     * @return
     */
    @Bean
    public SecurityManager securityManager(
            ShiroJwtRealm shiroJwtRealm, SessionStorageEvaluator sessionStorageEvaluator,
            SubjectFactory subjectFactory, SessionManager sessionManager) {
        DefaultWebSecurityManager defaultSecurityManager = new DefaultWebSecurityManager();
        defaultSecurityManager.setRealm(shiroJwtRealm);
        DefaultSubjectDAO defaultSubjectDAO = new DefaultSubjectDAO();
        defaultSubjectDAO.setSessionStorageEvaluator(sessionStorageEvaluator);
        defaultSecurityManager.setSubjectDAO(defaultSubjectDAO);
        defaultSecurityManager.setSubjectFactory(subjectFactory);
        defaultSecurityManager.setSessionManager(sessionManager);
        return defaultSecurityManager;
    }


    /**
     * Add. 访问控制器.
     *
     * @return
     */
    @Bean
    public ShiroJwtAccessControlFilter shiroJwtAccessControlFilter() {
        ShiroJwtAccessControlFilter shiroJwtAccessControlFilter = new ShiroJwtAccessControlFilter(jwtProperties);
        return shiroJwtAccessControlFilter;
    }

    @Bean
    public FilterRegistrationBean registration(ShiroJwtAccessControlFilter shiroJwtAccessControlFilter) {
        FilterRegistrationBean registration = new FilterRegistrationBean(shiroJwtAccessControlFilter);
        registration.setEnabled(false);
        return registration;
    }

    /**
     * 拦截器配置
     */
    @Bean(name = SHIRO_JWT_FILTER_FACTORY_NAME)
    public ShiroFilterFactoryBean shiroFilterFactoryBean(SecurityManager securityManager, ShiroJwtAccessControlFilter shiroJwtAccessControlFilter) {
        ShiroFilterFactoryBean shiroFilterFactoryBean = new ShiroFilterFactoryBean();
        shiroFilterFactoryBean.setSecurityManager(securityManager);
        //添加自定义的拦截器
        Map<String, Filter> filters = new LinkedHashMap<>();
        filters.put(SHIRO_JWT_AUTHC, shiroJwtAccessControlFilter);
        shiroFilterFactoryBean.setFilters(filters);
        //设置自定义的拦截器,拦截所有请求,方法一
        Map<String, String> filterChainDefinitionMap = new LinkedHashMap();
        //获取排除权限及登录认证的路径
        List<String> urlExcludes = shiroProperties.getUrlExcludes();
        if (!CollectionUtils.isEmpty(urlExcludes)) {
            log.info("Shiro URL Excludes : {}", JSON.toJSONString(urlExcludes));
            for (String urlExclude : urlExcludes) {
                filterChainDefinitionMap.put(urlExclude, ANON);
            }
        } else {
            log.info("Shiro URL Excludes Is Empty .");
        }
        /**
         * 标识 /api/**路径走statelessAuthc 过滤认证
         */
        List<String> urlPatterns = shiroProperties.getUrlPatterns();
        if (CollectionUtils.isEmpty(urlPatterns)) {
            log.warn("Shiro urlPatterns is empty,set default /**");
            filterChainDefinitionMap.put(DEFAULT_URL, SHIRO_JWT_AUTHC);
        } else {
            for (String urlPattern : urlPatterns) {
                filterChainDefinitionMap.put(urlPattern, SHIRO_JWT_AUTHC);
            }
        }
        shiroFilterFactoryBean.setFilterChainDefinitionMap(filterChainDefinitionMap);
        return shiroFilterFactoryBean;
    }

    /**
     * 拦截器注册
     */
    @Bean
    public FilterRegistrationBean delegatingFilterProxy() {
        FilterRegistrationBean filterRegistrationBean = new FilterRegistrationBean();
        DelegatingFilterProxy proxy = new DelegatingFilterProxy();
        proxy.setTargetFilterLifecycle(true);
        proxy.setTargetBeanName(SHIRO_JWT_FILTER_FACTORY_NAME);
        filterRegistrationBean.setFilter(proxy);
        return filterRegistrationBean;
    }

    /**
     * *
     * 开启Shiro的注解(如@RequiresRoles,@RequiresPermissions),
     * 需借助SpringAOP扫描使用Shiro注解的类,并在必要时进行安全逻辑验证
     * *
     * 配置以下两个bean(DefaultAdvisorAutoProxyCreator(可选)
     * 和AuthorizationAttributeSourceAdvisor)即可实现此功能
     * *
     */
    @Bean
    public DefaultAdvisorAutoProxyCreator advisorAutoProxyCreator() {
        DefaultAdvisorAutoProxyCreator advisorAutoProxyCreator = new DefaultAdvisorAutoProxyCreator();
        advisorAutoProxyCreator.setProxyTargetClass(true);
        return advisorAutoProxyCreator;
    }

    @Bean
    public AuthorizationAttributeSourceAdvisor authorizationAttributeSourceAdvisor(SecurityManager securityManager) {
        AuthorizationAttributeSourceAdvisor authorizationAttributeSourceAdvisor = new AuthorizationAttributeSourceAdvisor();
        authorizationAttributeSourceAdvisor.setSecurityManager(securityManager);
        return authorizationAttributeSourceAdvisor;
    }

    @Bean
    @ConditionalOnProperty(prefix = JwtProperties.JWT_PREFIX, name = "enable-auto-refresh-token", havingValue = "true", matchIfMissing = false)
    public TokenRefreshInterceptor tokenRefreshInterceptor(JwtProperties jwtProperties, UserAuthBiz userAuthBiz, JwtUtils jwtUtils) {
        return new TokenRefreshInterceptor(jwtProperties, userAuthBiz, jwtUtils);
    }

    @Configuration
    @ConditionalOnProperty(prefix = JwtProperties.JWT_PREFIX, name = "enable-auto-refresh-token", havingValue = "true", matchIfMissing = false)
    public static class ShiroJwtWebMvcConfigurer extends WebMvcConfigurerAdapter {

        @Autowired
        private TokenRefreshInterceptor tokenRefreshInterceptor;

        @Autowired
        private JwtProperties prop;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            InterceptorRegistration reg = registry.addInterceptor(tokenRefreshInterceptor);
            List<String> patterns = prop.getUrlPatterns();
            log.info("启用token自动刷新机制，已注册TokenRefreshInterceptor");
            for (String urlPattern : patterns) {
                log.info("TokenRefreshInterceptor匹配URL规则：" + urlPattern);
                reg.addPathPatterns(urlPattern);
            }
        }

        @Override
        public void addCorsMappings(CorsRegistry registry) {
            //允许访问header中的与token相关属性
            List<String> urls = prop.getUrlPatterns();
            for (String url : urls) {
                registry.addMapping(url).exposedHeaders(prop.getHeaderKeyOfToken());
            }
        }
    }

}
