package com.sinosoftgz.starter.shiro.jwt.interceptor;

import com.sinosoftgz.starter.jwt.model.JwtPrincipal;
import com.sinosoftgz.starter.jwt.properties.JwtProperties;
import com.sinosoftgz.starter.jwt.utils.JwtUtils;
import com.sinosoftgz.starter.shiro.jwt.model.UserInfo;
import com.sinosoftgz.starter.shiro.jwt.biz.UserAuthBiz;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.subject.Subject;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * SpringMVC拦截器，用于检查即将过期的token并及时返回token更新
 * Created by Roney on 2021/1/5 20:26.
 */
public class TokenRefreshInterceptor implements HandlerInterceptor {

    private JwtProperties jwtProperties;

    private UserAuthBiz userAuthBiz;

    private JwtUtils jwtUtils;

    public TokenRefreshInterceptor(JwtProperties jwtProperties, UserAuthBiz userAuthBiz, JwtUtils jwtUtils) {
        this.jwtProperties = jwtProperties;
        this.userAuthBiz = userAuthBiz;
        this.jwtUtils = jwtUtils;
    }

    /**
     * 涉及修改response header的操作只能放到preHandle中，postHandle中的response信息已经被锁定，无法修改
     **/
    @Override
    public boolean preHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o) throws Exception {
        Subject subject = SecurityUtils.getSubject();
        JwtPrincipal principal = (JwtPrincipal) subject.getPrincipal();
        //当前的token已经过期但还未超过maxIdleMinute的情况
        if (principal != null && System.currentTimeMillis() > principal.getExpiresAt()) {
            UserInfo user = userAuthBiz.getUserInfo(principal.getAccount());
            String newToken = jwtUtils.sign(user.getAccount(), user.getSecret());
            httpServletResponse.setHeader(jwtProperties.getHeaderKeyOfToken(), newToken);
        }
        return true;
    }

    @Override
    public void postHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, ModelAndView modelAndView) throws Exception {

    }

    @Override
    public void afterCompletion(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, Exception e) throws Exception {

    }
}
