package com.sinosoftgz.starter.shiro.jwt.support;

import com.auth0.jwt.interfaces.DecodedJWT;
import com.sinosoftgz.starter.jwt.model.JwtPrincipal;
import com.sinosoftgz.starter.jwt.utils.JwtUtils;
import com.sinosoftgz.starter.shiro.jwt.model.UserInfo;
import com.sinosoftgz.starter.shiro.jwt.biz.UserAuthBiz;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.SimpleAuthenticationInfo;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.subject.PrincipalCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.Set;

/**
 * Created by Roney on 2021/1/5 16:20.
 */
@Component
public class ShiroJwtRealm extends AuthorizingRealm {
    private static final Logger logger = LoggerFactory.getLogger(ShiroJwtRealm.class);

    @Autowired
    JwtUtils jwtUtils;

    @Autowired
    UserAuthBiz userAuthBiz;

    /**
     * 启用token支持
     *
     * @param token
     * @return
     */
    @Override
    public boolean supports(AuthenticationToken token) {
        return token instanceof JWTToken;
    }

    /***
     * 获取用户授权
     */
    @Override
    protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection principalCollection) {
        JwtPrincipal principal = (JwtPrincipal) principalCollection.getPrimaryPrincipal();

        // 获取用户名
        UserInfo account = userAuthBiz.getUserInfo(principal.getAccount());
        // 判断用户名是否存在
        if (StringUtils.isEmpty(account)) {
            throw new RuntimeException("获取用户授权信息失败");
        }
        // 创建一个授权对象
        SimpleAuthorizationInfo info = new SimpleAuthorizationInfo();

        // 进行权限设置
        Set<String> permissions = account.getPermissions();
        if (permissions != null && !permissions.isEmpty()) {
            info.addStringPermissions(permissions);
        }
        // 角色设置
        Set<String> roles = account.getRoles();
        if (roles != null) {
            info.addRoles(roles);
        }
        return info;
    }

    /**
     * 获取用户认证信息
     */
    @Override
    protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken authenticationToken)
            throws AuthenticationException {

        String token = (String) authenticationToken.getCredentials();
        // 解密获得username，用于和数据库进行对比
        String username = jwtUtils.getAccount(token);
        if (username == null) {
            throw new AuthenticationException("无效的请求");
        }
        UserInfo user = userAuthBiz.getUserInfo(username);
        if (user == null) {
            throw new AuthenticationException("未找到用户信息");
        }
        DecodedJWT jwt = jwtUtils.checkToken(token, user.getSecret());
        if (jwt == null) {
            throw new AuthenticationException("token已经过期，请重新登录");
        }
        JwtPrincipal principal = new JwtPrincipal();
        principal.setAccount(user.getAccount());
        principal.setExpiresAt(jwt.getExpiresAt().getTime());
        //这里实际上会将AuthenticationToken.getCredentials()与传入的第二个参数credentials进行比较
        //第一个参数是登录成功后，可以通过subject.getPrincipal获取
        return new SimpleAuthenticationInfo(principal, token, this.getName());
    }
}
