package com.sinosoftgz.simpleSession;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import com.sinosoftgz.simpleSession.data.SessionCache;
import com.sinosoftgz.simpleSession.data.SessionCacheManager;

public class SessionSharingFilter implements Filter {
	
	private Logger logger = LoggerFactory.getLogger(SessionSharingFilter.class);
	
	private static final String MAX_ACTIVE_TIME_KEY = "maxActiveTime";
	//tomcat默认的JSESSIONID
	//private static final String JSESSIONID="JSESSIONID";
	
	private ServletContext servletContext;
	
	@Value("#{sessionConfig.session_max_age}")
	private int maxActiveTime;
	
	@Value("#{sessionConfig.cookies_max_age}")
	private int cookiesMaxAge;
	
	@Value("#{sessionConfig.cookies_domain}")
	private String cookiesDomain;

	//注入 自定义 jessionId
	@Value("#{sessionConfig.JsessionId}")
	private String jsessionId;
	
	//session保存时间
	@Value("#{sessionConfig.session_max_age}")
	private int sessionMaxAge;
	
	@Override
	public void init(FilterConfig filterConfig) throws ServletException {
		servletContext = filterConfig.getServletContext();
		String matString = filterConfig.getInitParameter(MAX_ACTIVE_TIME_KEY);
		if(StringUtils.isNotEmpty(matString)){
			maxActiveTime = Integer.valueOf(matString);
		}
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)throws IOException, ServletException {
		HttpServletRequest  httpServletRequest  = (HttpServletRequest)request;
		HttpServletResponse httpServletResponse = (HttpServletResponse)response;
		
		
		SessionCache sessionCache = SessionCacheManager.getSessionCache();
		//如果缓存不为空的情况下，重写
		if (sessionCache != null && sessionCache.isJedisActive()) {
			logger.debug("using cache's session");
			//重写request
			SessionHttpServletRequestWrapper sessionHttpServletRequestWrapper = new SessionHttpServletRequestWrapper(httpServletRequest, httpServletResponse, servletContext, cookiesMaxAge, cookiesDomain, sessionMaxAge, jsessionId);
			
			//从缓存里拿到redis session 续约
			refreshSessionCache(sessionHttpServletRequestWrapper.getSession().getId(), sessionCache);

			try{
	        	chain.doFilter(sessionHttpServletRequestWrapper, httpServletResponse);
			}catch(Exception e){
				e.printStackTrace();
			}finally {
				//session 不可用
				if(sessionHttpServletRequestWrapper.isRequestedSessionIdValid()){
					//清除cookies
					Cookie cookie = new Cookie(jsessionId, null);
					cookie.setMaxAge(0);
					cookie.setPath("/");
					cookie.setDomain(this.cookiesDomain);
					cookie.setHttpOnly(true);
					httpServletResponse.addCookie(cookie);
					
					HttpSession httpSession = sessionHttpServletRequestWrapper.getSession(false);
					//清除缓存
					if(httpSession != null){
						SessionCacheManager.getSessionCache().put(httpSession.getId(), null, 0);
					}
				}

			}
			
			return;
		}
	
		logger.debug("using container's session");
		//使用 容器的 session
		chain.doFilter(request, response);
	}

	@Override
	public void destroy() {
		// TODO Auto-generated method stub
	}
    
	/**
	 * 刷新redis的session
	 * @param sessionId
	 * @param sessionCache
	 */
	private void refreshSessionCache(String sessionId,SessionCache sessionCache){
		if (null != sessionId && null != sessionCache) {
			sessionCache.setMaxInactiveInterval(sessionId, this.sessionMaxAge);
		}
	}

	public int getCookiesMaxAge() {
		return cookiesMaxAge;
	}

	public void setCookiesMaxAge(int cookiesMaxAge) {
		this.cookiesMaxAge = cookiesMaxAge;
	}

	public String getCookiesDomain() {
		return cookiesDomain;
	}

	public void setCookiesDomain(String cookiesDomain) {
		this.cookiesDomain = cookiesDomain;
	}

	public String getJsessionId() {
		return jsessionId;
	}

	public void setJsessionId(String jsessionId) {
		this.jsessionId = jsessionId;
	}

	public int getSessionMaxAge() {
		return sessionMaxAge;
	}

	public void setSessionMaxAge(int sessionMaxAge) {
		this.sessionMaxAge = sessionMaxAge;
	}
	
	
	
    
}
