package com.sinosoftgz.simpleSession;

import javax.servlet.ServletContext;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sinosoftgz.simpleSession.data.SessionCache;
import com.sinosoftgz.simpleSession.data.SessionCacheManager;
import com.sinosoftgz.simpleSession.data.SessionMap;

import java.util.UUID;

public class SessionHttpServletRequestWrapper extends HttpServletRequestWrapper {

	public static final String NAME = "SESSION";
	public Logger logger = LoggerFactory.getLogger(SessionHttpServletRequestWrapper.class);
	
	public static final String CURRENT_SESSION_ATTR = SessionHttpServletRequestWrapper.class.getName();
	
	private ServletContext servletContext;
	private int cookiesMaxAge;
	private String cookiesDomain;
	private int sessionMaxAge;
	private String JsessionId;
	private HttpServletResponse response;
	private String prefix;
	
	
	
	
	public SessionHttpServletRequestWrapper(HttpServletRequest request,
											HttpServletResponse response,
											ServletContext servletContext, int cookiesMaxAge,
											String cookiesDomain, int sessionMaxAge, String jsessionId, String prefix) {
		super(request);
		this.servletContext = servletContext;
		this.cookiesMaxAge = cookiesMaxAge;
		this.cookiesDomain = cookiesDomain;
		this.sessionMaxAge = sessionMaxAge;
		JsessionId = jsessionId;
		this.response = response;
		this.prefix =prefix;
	}


	@Override
	public HttpSession getSession() {
		return getSession(true);
	}
	
    @Override
    public HttpSession getSession(boolean create) {
    	//1.获取当前的session
    	HttpSessionWrapper currentSession = getCurrentSession();
		SessionCache sessionCache = SessionCacheManager.getSessionCache();
		if(currentSession != null) {
			//refreshSessionCache(currentSession.getId(), sessionCache);
			return currentSession;
		}
    	
		//2.从cookies 里拿出JSessionId
    	String cookieSessionId = getCookieSessionID();
    	//3.能拿出sessionId

		if(cookieSessionId != null) {
			//从缓存里拿到redis
			SessionMap sessionMap = sessionCache.get(cookieSessionId);
			if(sessionMap != null && sessionMap.isInvalidated() == false) {
				//从缓存里拿出了sessionMap ,复制session
				currentSession = new HttpSessionWrapper(sessionMap, sessionCache, this.sessionMaxAge, servletContext);
				currentSession.setNew(false);
				setCurrentSession(currentSession);
				return currentSession;
			}
		}
		
		
		
		if(!create) {//是否创建
			return null;
		}

		//4.生成新的session
		String sessionid =new StringBuilder()
							.append(prefix)
							.append(":")
							.append(NAME)
							.append(":")
							.append(UUID.randomUUID().toString())
							.toString();
		SessionMap sessionMap = new SessionMap(this,prefix);//生成新ID
		
		//6.新session
		currentSession = new HttpSessionWrapper(sessionMap,sessionCache, this.sessionMaxAge, servletContext);
		//7.存储到缓存
		if(sessionCache.put(sessionMap.getId(), sessionMap, this.sessionMaxAge)){
			logger.debug(" save session to cache,sessionId is {}",sessionMap.getId());
		}
		setCurrentSession(currentSession);
		//8.设置cookies
		setCookies(currentSession.getId());
		
		return currentSession;
    }
    
	
	/**
	 * 刷新redis的session
	 * @param sessionId
	 * @param sessionCache
	 */
	private void refreshSessionCache(String sessionId,SessionCache sessionCache){
		if (null != sessionId && null != sessionCache) {
			sessionCache.setMaxInactiveInterval(sessionId, this.sessionMaxAge);
		}
	}
    
    /**
     * 设置cookies
     * @param cookiesSessionId
     */
    private void setCookies(String cookiesSessionId){
		Cookie cookie = new Cookie(this.JsessionId, cookiesSessionId);
		cookie.setMaxAge(this.cookiesMaxAge);
		if(this.cookiesDomain != null){
			cookie.setDomain(this.cookiesDomain);
		}
		cookie.setPath("/");
		cookie.setHttpOnly(true);
		this.response.addCookie(cookie);
    }

    /**
     * 获取cookie中的redis使用的id
     * @return
     */
	private String getCookieSessionID() {
		String cookieSessionId = null;//session id 为空
    	Cookie[] cookies = this.getCookies();
    	if(cookies != null && this.JsessionId != null){
    		for(Cookie cookie:cookies){
    			if(this.JsessionId.equals(cookie.getName())){
    				cookieSessionId = cookie.getValue();//从cookies里拿到了 自定义的JSESSIONID
    				logger.debug("get JsessionId from cookies,JsessionId is :{} ",cookieSessionId);
    				break;
    			}
    		}
    	}
		return cookieSessionId;
	}
    
	private HttpSessionWrapper getCurrentSession() {
		return (HttpSessionWrapper) getAttribute(CURRENT_SESSION_ATTR);
	}

	private void setCurrentSession(HttpSessionWrapper currentSession) {
		if(currentSession == null) {
			removeAttribute(CURRENT_SESSION_ATTR);//当前缓存为空，移除内存session所有属性
		} else {
			setAttribute(CURRENT_SESSION_ATTR, currentSession);//把缓存的session重置属性
		}
	}
	
    public boolean isRequestedSessionIdValid() {
    	HttpSessionWrapper httpSessionWrapper = (HttpSessionWrapper) this.getSession(false);
    	if(httpSessionWrapper == null){
    		return true;
    	}
		return httpSessionWrapper.isInvalidated();
    }
	
	
}
