package sinosoftgz.utils.data;

import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.openxml4j.exceptions.NotOfficeXmlFileException;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import java.io.*;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;

import static org.apache.poi.ss.usermodel.CellType.*;


public class ExcelUtils {

    /**
     * 从Excel文件得到二维数组，每个sheet的第一行为标题
     *
     * @param file
     *            Excel文件
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getData(File file) throws FileNotFoundException, IOException {
        return getData(file, 1);
    }

    /**
     * 从Excel文件得到二维数组
     * @param file Excel文件
     * @param ignoreRows 忽略的行数，通常为每个sheet的标题行数
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getData(File file, int ignoreRows) throws FileNotFoundException, IOException {
        BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));
        return getData(in, ignoreRows, 1);
    }

    /**
     * 从Excel文件流得到二维数组
     * @param inputStream Excel文件流
     * @param ignoreRows 忽略的行数，通常为每个sheet的标题行数
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getData(InputStream inputStream, int ignoreRows) throws FileNotFoundException, IOException {
        return getData(inputStream, ignoreRows, 1);
    }

    /**
     * 从Excel文件得到二维数组
     * @param inputStream Excel文件流
     * @param ignoreRows 忽略的行数，通常为每个sheet的标题行数
     * @param index sheet工作表序号
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getData(InputStream inputStream, int ignoreRows, int index) throws FileNotFoundException, IOException {
        String[][] data;

        try {
            data = getDataXssf(inputStream, ignoreRows, index);
        } catch (NotOfficeXmlFileException noxfe) {
            data = getDataHssf(inputStream, ignoreRows, index);
        }
        return data;
    }

    /**
     * 从Excel文件得到二维数组，Excel2003及之前版本使用
     * @param inputStream
     * @param ignoreRows
     * @param index
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getDataHssf(InputStream inputStream, int ignoreRows, int index) throws FileNotFoundException, IOException {
        ArrayList result = new ArrayList();
        int rowSize = 0;
        // 打开HSSFWorkbook
        POIFSFileSystem fs = new POIFSFileSystem(inputStream);
        HSSFWorkbook wb = null;
        wb = new HSSFWorkbook(fs);
        HSSFCell cell = null;
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        DecimalFormat decimalFormat = new DecimalFormat("0");
        StringBuilder value = new StringBuilder();
        HSSFSheet st = wb.getSheetAt(index - 1);
        // 第一行为标题，不取
        for (int rowIndex = ignoreRows; rowIndex <= st.getLastRowNum(); rowIndex++) {
            HSSFRow row = st.getRow(rowIndex);
            if (row == null) {
                continue;
            }
            int tempRowSize = row.getLastCellNum() + 1;
            if (tempRowSize > rowSize) {
                rowSize = tempRowSize;
            }
            String[] values = new String[rowSize];
            Arrays.fill(values, "");
            boolean hasValue = false;
            for (short columnIndex = 0; columnIndex <= row.getLastCellNum(); columnIndex++) {
                value.setLength(0);
                cell = row.getCell(columnIndex);

                if (cell != null) {
                    // 注意：一定要设成这个，否则可能会出现乱码
                    switch (cell.getCellType()) {
                        case STRING:
                            value.append(cell.getStringCellValue());
                            break;
                        case NUMERIC:
                            if (HSSFDateUtil.isCellDateFormatted(cell)) {
                                Date date = cell.getDateCellValue();
                                if (date != null) {
                                    value.append(dateFormat.format(date));
                                }
                            } else {
                                value.append(decimalFormat.format(cell.getNumericCellValue()));
                            }
                            break;
                        case FORMULA:
                            // 导入时如果为公式生成的数据则无值
                            if (cell.getStringCellValue().equals("")) {
                                value.append(cell.getNumericCellValue());
                            } else {
                                value.append(cell.getStringCellValue());
                            }
                            break;
                        case BLANK:
                            break;
                        case ERROR:
                            break;
                        case BOOLEAN:
                            value.append((cell.getBooleanCellValue() == true ? "Y" : "N"));
                            break;
                        default:
                            value.setLength(0);
                    }
                }
                String s = value.toString().trim();
                if (columnIndex == 0 && s.equals("")) {// 若第一列为空，则向后判断5列
                    // 若都为空则不导入该行数据
                    if (row.getCell(new Short("1").shortValue()) == null
                            && row.getCell(new Short("2").shortValue()) == null
                            && row.getCell(new Short("3").shortValue()) == null
                            && row.getCell(new Short("4").shortValue()) == null
                            && row.getCell(new Short("5").shortValue()) == null) {
                        break;
                    }
                }
                values[columnIndex] = value.toString();
                hasValue = true;

                // 到每行的第五列的时候开始判断前5列是否都为空,若都为空则不导入该行数据 (该方法需要与以上方法同时使用)
                if (columnIndex == 4) {
                    if ("".equals(values[0]) && "".equals(values[1]) && "".equals(values[2])
                            && "".equals(values[3]) && "".equals(values[4])) {
                        hasValue = false;
                        break;
                    }
                }
            }

            if (hasValue) {
                result.add(values);
            }
        }
        inputStream.close();
        String[][] returnArray = new String[result.size()][rowSize];
        for (int i = 0; i < returnArray.length; i++) {
            returnArray[i] = (String[]) result.get(i);
        }
        return returnArray;
    }

    /**
     * 从Excel文件得到二维数组，Excel2007及之后版本使用
     * @param inputStream
     * @param ignoreRows
     * @param index
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getDataXssf(InputStream inputStream, int ignoreRows, int index) throws FileNotFoundException, IOException {
        ArrayList result = new ArrayList();
        int rowSize = 0;
        // 打开XSSFWorkbook
        XSSFWorkbook wb = null;
        wb = new XSSFWorkbook(inputStream);

        XSSFCell cell = null;
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        DecimalFormat decimalFormat = new DecimalFormat("0");
        StringBuilder value = new StringBuilder();
        XSSFSheet st = wb.getSheetAt(index - 1);
        // 第一行为标题，不取
        for (int rowIndex = ignoreRows; rowIndex <= st.getLastRowNum(); rowIndex++) {
            XSSFRow row = st.getRow(rowIndex);
            if (row == null) {
                continue;
            }
            int tempRowSize = row.getLastCellNum() + 1;
            if (tempRowSize > rowSize) {
                rowSize = tempRowSize;
            }
            String[] values = new String[rowSize];
            Arrays.fill(values, "");
            boolean hasValue = false;
            for (short columnIndex = 0; columnIndex <= row.getLastCellNum(); columnIndex++) {
                value.setLength(0);
                cell = row.getCell(columnIndex);
                if (cell != null) {
                    // 注意：一定要设成这个，否则可能会出现乱码
                    switch (cell.getCellType()) {
                        case STRING:
                            value.append(cell.getStringCellValue());
                            break;
                        case NUMERIC:
                            if (HSSFDateUtil.isCellDateFormatted(cell)) {
                                Date date = cell.getDateCellValue();
                                if (date != null) {
                                    value.append(dateFormat.format(date));
                                }
                            } else {
                                value.append(decimalFormat.format(cell.getNumericCellValue()));
                            }
                            break;
                        case FORMULA:
                            // 导入时如果为公式生成的数据则无值
                            if (cell.getStringCellValue().equals("")) {
                                value.append(cell.getNumericCellValue());
                            } else {
                                value.append(cell.getStringCellValue());
                            }
                            break;
                        case BLANK:
                            break;
                        case ERROR:
                            break;
                        case BOOLEAN:
                            value.append((cell.getBooleanCellValue() == true ? "Y" : "N"));
                            break;
                        default:
                            value.setLength(0);
                    }
                }
                String s = value.toString().trim();
                if (columnIndex == 0 && s.equals("")) {// 若第一列为空，则向后判断5列
                    // 若都为空则不导入该行数据
                    if (row.getCell(new Short("1").shortValue()) == null
                            && row.getCell(new Short("2").shortValue()) == null
                            && row.getCell(new Short("3").shortValue()) == null
                            && row.getCell(new Short("4").shortValue()) == null
                            && row.getCell(new Short("5").shortValue()) == null) {
                        break;
                    }
                }
                values[columnIndex] = value.toString();
                hasValue = true;

                // 到每行的第五列的时候开始判断前5列是否都为空,若都为空则不导入该行数据 (该方法需要与以上方法同时使用)
                if (columnIndex == 4) {
                    if ("".equals(values[0]) && "".equals(values[1]) && "".equals(values[2])
                            && "".equals(values[3]) && "".equals(values[4])) {
                        hasValue = false;
                        break;
                    }
                }
            }

            if (hasValue) {
                result.add(values);
            }
        }
        inputStream.close();
        String[][] returnArray = new String[result.size()][rowSize];
        for (int i = 0; i < returnArray.length; i++) {
            returnArray[i] = (String[]) result.get(i);
        }
        return returnArray;
    }


    /**
     * 解析一个Excel文件的某个特定的sheet sheet号码从1开始
     * @param file excel文件
     * @param ignoreRows 忽略的行数
     * @param index sheet的页码
     * @return
     * @throws FileNotFoundException
     * @throws IOException
     */
    public static String[][] getData(File file, int ignoreRows, int index) throws FileNotFoundException, IOException {
        ArrayList result = new ArrayList();
        int rowSize = 0;
        BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));

        return getData(in, ignoreRows, index);
    }

}
