package sinosoftgz.utils.encrypt;

import org.apache.commons.codec.binary.Base64;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import java.io.UnsupportedEncodingException;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * RSA工具类
 * @author Liang Wenxu
 * @since 2021-2-1
 */
public class RsaUtil {
    /**
     * 随机生成密钥对
     * @throws NoSuchAlgorithmException
     */
    public static StringKeyPair genKeyPair() throws NoSuchAlgorithmException {
        // KeyPairGenerator类用于生成公钥和私钥对，基于RSA算法生成对象
        KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance("RSA");
        // 初始化密钥对生成器，密钥大小为96-1024位
        keyPairGen.initialize(1024, new SecureRandom());
        // 生成一个密钥对，保存在keyPair中
        KeyPair keyPair = keyPairGen.generateKeyPair();
        RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();   // 得到私钥
        RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();  // 得到公钥
        // 得到公钥字符串
        String publicKeyString = new String(Base64.encodeBase64(publicKey.getEncoded()));
        // 得到私钥字符串
        String privateKeyString = new String(Base64.encodeBase64((privateKey.getEncoded())));
        return new StringKeyPair(publicKeyString, privateKeyString);
    }

    /**
     * 使用公钥加密
     * @param str 原文
     * @param publicKey 公钥
     * @return 返回加密后的密文（BASE64编码）
     * @throws NoSuchAlgorithmException 运行环境不支持RSA加密时抛出（检查JDK版本问题）
     * @throws InvalidKeyException 公钥错误时抛出
     * @throws NoSuchPaddingException 包模式不支持时抛出（检查JDK版本问题）
     * @throws UnsupportedEncodingException 字符编码不支持时抛出
     * @throws BadPaddingException 包模式不支持时抛出（检查JDK版本问题）
     * @throws IllegalBlockSizeException 通常在原文过大时抛出
     * @throws InvalidKeySpecException 公钥编码方式错误
     */
    public static String encrypt( String str, String publicKey ) throws NoSuchAlgorithmException, InvalidKeyException, NoSuchPaddingException, UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException, InvalidKeySpecException {
        //base64编码的公钥
        byte[] decoded = Base64.decodeBase64(publicKey);
        RSAPublicKey pubKey = (RSAPublicKey) KeyFactory.getInstance("RSA").generatePublic(new X509EncodedKeySpec(decoded));
        //RSA加密
        Cipher cipher = Cipher.getInstance("RSA");
        cipher.init(Cipher.ENCRYPT_MODE, pubKey);
        String outStr = Base64.encodeBase64String(cipher.doFinal(str.getBytes("UTF-8")));
        return outStr;
    }

    /**
     * 使用私钥解密
     * @param str 密位
     * @param privateKey 私钥
     * @return 返回明文
     * @throws UnsupportedEncodingException 字符编码不支持时抛出
     * @throws NoSuchAlgorithmException 运行环境不支持RSA加密时抛出（检查JDK版本问题）
     * @throws BadPaddingException 包模式不支持时抛出（检查JDK版本问题）
     * @throws NoSuchPaddingException 包模式不支持时抛出（检查JDK版本问题）
     * @throws IllegalBlockSizeException 通常在原文过大时抛出
     * @throws InvalidKeySpecException 公钥编码方式错误
     * @throws InvalidKeyException 公钥错误时抛出
     */
    public static String decrypt(String str, String privateKey) throws UnsupportedEncodingException, NoSuchAlgorithmException, BadPaddingException, IllegalBlockSizeException, InvalidKeySpecException, InvalidKeyException, NoSuchPaddingException {
        //64位解码加密后的字符串
        byte[] inputByte = Base64.decodeBase64(str.getBytes("UTF-8"));
        //base64编码的私钥
        byte[] decoded = Base64.decodeBase64(privateKey);
        RSAPrivateKey priKey = (RSAPrivateKey) KeyFactory.getInstance("RSA").generatePrivate(new PKCS8EncodedKeySpec(decoded));
        //RSA解密
        Cipher cipher = Cipher.getInstance("RSA");
        cipher.init(Cipher.DECRYPT_MODE, priKey);
        String outStr = new String(cipher.doFinal(inputByte));
        return outStr;
    }
}
