package sinosoftgz.utils.data;

import sinosoftgz.utils.lang.Lang;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 树形数据结构工具类
 * @author Liang Wenxu
 * @since 2018/12/20
 */
public class TreeUtils {
    /**
     * 将列表转换为树
     * @param list
     * @return
     */
    public static <T extends TreeNode> List<T> list2Tree(final List<T> list, Class<T> tClass) {
        List<T> tree = new LinkedList<>();
        List<T> visitedNodes = new LinkedList<>();
        // 通过广度遍历，按层级进行处理
        // 定义未遍历过的节点列表
        List<T> unvisitedNodes = list.stream().filter(i -> Lang.isEmpty(i.getPid())).collect(Collectors.toList());
        // lastParents暂存上一次遍历时的父级节点，以缩减回溯耗时
        Map<String, T> lastParents = new HashMap<>(list.size());
        while(!unvisitedNodes.isEmpty()) {
            for(T tn : unvisitedNodes) {
                if(tn.getChildren() == null) {
                    tn.setChildren(new ArrayList<>());
                }
                // 判断是否根节点
                if(Lang.isEmpty(tn.getPid()) || "0".equals(tn.getPid())) {
                    tree.add(tn);
                } else {
                    // 添加到父节点
                    T parent = lastParents.get(tn.getPid());
                    if(parent != null) {
                        parent.getChildren().add(tn);
                    } else {
                        tree.add(tn);
                    }

                }
                lastParents.put(tn.getId(), tn);
                visitedNodes.add(tn);
            }
            // 处理下一层
            List<String> pids = unvisitedNodes.stream().map(T::getId).collect(Collectors.toList());
            List<String> visitedIds = visitedNodes.stream().map(T::getId).collect(Collectors.toList());
            unvisitedNodes = list.stream()
                    .filter(n -> pids.contains(n.getPid()) && !visitedIds.contains(n.getId()))
                    .collect(Collectors.toList());
        }

        return tree;
    }


    /**
     * 将树转换为list（深度遍历）
     * @param id
     * @param tree
     * @return
     */
    private static List<AbstractTreeNode> tree2listDeepth(final String id, final List<AbstractTreeNode> tree) {
        List<AbstractTreeNode> visitedNodes = new LinkedList<>();
        List<AbstractTreeNode> unvisitedNodes = tree;

        while(!unvisitedNodes.isEmpty()) {
            AbstractTreeNode currNode = unvisitedNodes.remove(0);

            List<AbstractTreeNode> newNodes = currNode.getChildren()
                    .stream()
                    .filter(node -> !visitedNodes.contains(node))
                    .collect(Collectors.toList());
            unvisitedNodes.addAll(0, newNodes);
            visitedNodes.add(currNode);
        }

        return visitedNodes;
    }

    /**
     * 将树转换为list（广度遍历）
     * @param tree
     * @return
     */
    public static List<AbstractTreeNode> tree2listBreadth(final List<AbstractTreeNode> tree) {
        List<AbstractTreeNode> visitedNodes = new LinkedList<>();
        List<AbstractTreeNode> unvisitedNodes = new ArrayList<>();
        Collections.copy(tree, unvisitedNodes);

        while(!unvisitedNodes.isEmpty()) {
            List<AbstractTreeNode> newNodes = unvisitedNodes
                    .stream()
                    .map(i -> i.getChildren())
                    .flatMap(Collection::stream)
                    .filter(node -> !visitedNodes.contains(node))
                    .collect(Collectors.toList());

            visitedNodes.addAll(unvisitedNodes);
            unvisitedNodes = newNodes;
        }

        return visitedNodes;
    }
}
